function  [nodeBel, edgeBel, logZ] = UGM_Infer_Chain(nodePot, edgePot, edgeStruct, logspace)
% INPUT
% nodePot(node,class)
% edgePot(class,class,edge) where e is referenced by V,E (must be the same
% between feature engine and inference engine)
% logspace is 1 if potentials are in log-domain, 0 otherwise
%
% OUTPUT
% nodeBel(node,class) - marginal beliefs
% edgeBel(class,class,e) - pairwise beliefs
% logZ - negative of free energy
%
% NOTE: This code assumes that the edges are in order!
%		(use UGM_Infer_Tree if they are not)

if edgeStruct.useMex
    if logspace, error('UGM_Infer_Chain: Mex not yet suported in log-domain.'); end
    [nodeBel,edgeBel,logZ] = UGM_Infer_ChainC(nodePot,edgePot,edgeStruct.nStates,logspace);
else
    [nNodes,maxState] = size(nodePot);
    nEdges = size(edgePot,3);
    edgeEnds = edgeStruct.edgeEnds;
    nStates = edgeStruct.nStates;
    maximize = 0;
    
    % accumulation operator
    if logspace
      opFun = @plus;     
    else     
      opFun = @times;
    end

    % Forward Pass
    [alpha,kappa] = UGM_ChainFwd(nodePot,edgePot,nStates,maximize,logspace);
    
    % Backward Pass
    beta = zeros(nNodes,maxState);
    beta(nNodes,1:nStates(nNodes)) = 1;
    for n = nNodes-1:-1:1
        msgIn = opFun( nodePot(n+1,1:nStates(n+1)), beta(n+1,1:nStates(n+1)) );
        tmp = bsxfun(opFun, msgIn, edgePot(1:nStates(n),1:nStates(n+1),n));
        if logspace
          beta(n,1:nStates(n)) = logsumexp(tmp,2)';
          beta(n,1:nStates(n)) = beta(n,1:nStates(n)) - logsumexp(beta(n,1:nStates(n)));
%           logZ_beta = log(sum(beta(n,1:nStates(n))));
%           beta(n,1:nStates(n)) = log(beta(n,1:nStates(n))) - logZ_beta;
        else
          beta(n,1:nStates(n)) = sum(tmp,2)';
          beta(n,1:nStates(n)) = beta(n,1:nStates(n))/sum(beta(n,1:nStates(n)));
        end
    end

    % Compute Node Beliefs
    nodeBel = zeros(size(nodePot));
    for n = 1:nNodes
        tmp = opFun(alpha(n,1:nStates(n)), beta(n,1:nStates(n)));
        if logspace
          nodeBel(n,1:nStates(n)) = tmp - logsumexp(tmp);
        else
          nodeBel(n,1:nStates(n)) = tmp/sum(tmp);
        end
    end

    % Compute Edge Beliefs
    edgeBel = zeros(size(edgePot));
    for n = 1:nNodes-1
        backPreM = opFun(nodePot(n+1,1:nStates(n+1)),beta(n+1,1:nStates(n+1)));                
        pairBack = bsxfun(opFun, backPreM, edgePot(1:nStates(n), 1:nStates(n+1), n));
        tmp = bsxfun(opFun, alpha(n,1:nStates(n))', pairBack);
        if logspace
          edgeBel(:,:,n) = tmp - logsumexp(tmp(:));
        else
          edgeBel(:,:,n) = tmp./sum(tmp(:));
        end     
    end

    % Compute logZ
    if logspace
      logZ = sum(kappa);
    else
      logZ = sum(log(kappa));
    end
end