function [new_msg, i] = UGM_LoopyBP(nodePot,edgePot,edgeStruct,stepsize,maximize,logspace)
% MESSAGES = UGM_LOOPYBP(NODEPOT, EDGEPOT, EDGESTRUCT, STEPSIZE, MAXIMIZE, LOGSPACE)
%
% Performs Loopy BP on pairwise MRF using sequential message update
% schedule.  If MAXIMIZE=1 then Max-Product or Max-Sum variant is
% performed.  If LOGSPACE=1 then updates are performed in log-domain,
% NODEPOT and EDGEPOT should be log-potentials, and log-messages are
% returned.  Otherwise, if LOGSPACE=0 all computations are in probability
% domain.  STEPSIZE is a damping factor for message updates in (0,1).
%
% Initial Implementation: M. Schmidt
% 
% 2014: J. Pacheco
% Max-Sum updates.
%

[nNodes,maxState] = size(nodePot);
nEdges = size(edgePot,3);
edgeEnds = edgeStruct.edgeEnds;
V = edgeStruct.V;
E = edgeStruct.E;
nStates = double(edgeStruct.nStates);

% accumulation operators
if logspace
  opFun = @plus;     
  opFunInv = @minus;
  opNorm = @logsumexp;
else     
  opFun = @times;
  opFunInv = @rdivide;
  opNorm = @sum;
end

% Initialize
new_msg = zeros(maxState,nEdges*2);
for e = 1:nEdges
    n1 = edgeEnds(e,1);
    n2 = edgeEnds(e,2);
    if logspace
      new_msg(1:nStates(n2),e) = -log(nStates(n2)); % Message from n1 => n2
      new_msg(1:nStates(n1),e+nEdges) = -log(nStates(n1)); % Message from n2 => n1
    else
      new_msg(1:nStates(n2),e) = 1.0/double(nStates(n2)); % Message from n1 => n2
      new_msg(1:nStates(n1),e+nEdges) = 1/double(nStates(n1)); % Message from n2 => n1
    end
end
old_msg = new_msg;

% Update Messages
for i = 1:edgeStruct.maxIter
    if i==1, damp = 1.0; else damp = stepsize; end
    for n = 1:nNodes
        % Find Neighbors
        edges = UGM_getEdges(n,edgeStruct);

        % Send a message to each neighbor
        for e = edges
            n1 = edgeEnds(e,1);
            n2 = edgeEnds(e,2);

            % get edge potential
            if n == edgeEnds(e,2)
                pot_ij = edgePot(1:nStates(n1),1:nStates(n2),e);
            else
                pot_ij = edgePot(1:nStates(n1),1:nStates(n2),e)';
            end

            % Compute temp = product of all incoming msgs except j
            temp = nodePot(n,1:nStates(n))';
            for e2 = edges
                if e ~= e2
                    if n == edgeEnds(e2,2)
                        temp = opFun(temp, new_msg(1:nStates(n),e2));
                    else
                        temp = opFun(temp, new_msg(1:nStates(n),e2+nEdges));
                    end
                end
            end

            % Max-Product / Max-Sum
            if maximize
                if edgeStruct.useMex && (~logspace)
                  newm = max_mult(pot_ij,temp);
                elseif logspace
                  newm = max_sumM(pot_ij,temp);
                else
                  newm = max_multM(pot_ij,temp);
                end
            % Loopy BP log-domain
            elseif logspace
                temp = exp(temp - max(temp(:)));
                pot_ij = exp(pot_ij - max(pot_ij(:)));
                newm = log( pot_ij*temp );
            % Loopy BP probability domain
            else
                newm = pot_ij * temp;
            end
                                    
            % Normalize & Damp
            norm_const = opNorm(newm);
            newm = opFunInv(newm,norm_const);            
            if n == edgeEnds(e,2)
                new_msg(1:nStates(n1),e+nEdges) = ...
                  damp * newm + (1-damp) * old_msg(1:nStates(n1),e+nEdges);
            else
                new_msg(1:nStates(n2),e) = ...
                  damp * newm + (1-damp) * old_msg(1:nStates(n2),e);
            end            
        end
    end

    % Check convergence
    if logspace, 
      msgDiff = norm( exp(new_msg(:)) - exp(old_msg(:)) ); 
    else
      msgDiff = norm( new_msg(:) - old_msg(:) );
    end
    if msgDiff < edgeStruct.convTol, break; end

    % save old message
    old_msg = new_msg;
end

