function [nodeBel, x, stats] = DPMP_smooth(y, x0, N, model, funHandles, display_iter, maxIters)
% DPMP_smooth Approximate MAP inference for time series models with D-PMP.
%
%   Input:
%     y : The observed time series (dim_y x T)
%     x0 : The initial particle set (dim_x x N x T)
%     N : The total number of D-PMP particles allowed. For example, if you set N
%       to be 10, then each iteration D-PMP will propose 5 new particles to make
%       a total of 10 and then select 5 particles to retain.
%     model : The time series model. Should have the following fields:
%       - dim_x : dimension of the x's
%       - dim_y : dimension of the y's
%       - pi0_logpdf : Function handle to calculate `log p(x_0)`
%       - pi0_rand : Function handle to sample from `p(x_0)`
%       - trans_logpdf : Function handle to calculate `log p(x_t | x_{t-1}, t)`
%       - trans_rand : Function handle to sample from `p(x_t | x_{t-1}, t)`
%       - obs_logpdf : Function handle to calculate `log p(y_t | x_t, t)`
%       - obs_rand : Function handle to sample from `p(y_t | x_t, t)`
%     funHandles : A struct with function handles to be passed to DPMP_Infer.
%       Carries the callback and proposal functions, etc.
%     display_iter : DPMP_Infer will invoke the callback every `display_iter`
%       iterations.
%     maxIters : The maximum number of iterations to run D-PMP.
%
%   Returns
%     nodeBel : The resulting log pseudo max-marginals (T x N). In other words,
%       the "marginal belief" that each node is in each state.
%     x : The final particle set (dim_x x N/2 x T)
%     stats : A struct with various statistics about the D-PMP execution.

  assert(size(x0, 3) == size(y, 2));
  T = size(x0, 3);

  % D-PMP options
  verbose = true;
  method = 'dpmpmax';             % Values: lbp, trw, dpmpmax, dpmpsum, gpmp
  useMex = true;
  maxItersLBP = 5;                % LBP/TRW Iterations
  convTolBP = 1e-4;               % LBP/TRW Convergence tolerance
  stepsize = 1.0;                 % LBP/TRW damping
  Temp = 1;                       % D-PMP particle selection temperature
  msgPass = 'lbp';                % lbp or trw
  doTieRes = false;               % Resolve ties? (expensive)
  dpmpOpt = DPMP_setOptions(verbose, display_iter, method,  msgPass, N, ...
    maxIters, maxItersLBP, 1.0, Temp, stepsize, convTolBP, useMex, ...
    doTieRes);

  if isfield(funHandles, 'sampleProb')
    dpmpOpt.sampleProb = funHandles.sampleProb;
  end

  % Note: proposals and callback are to be handled by the caller
  funHandles.funEvalModel = @(x, eS) ssm_eval_model(x, y, model, eS);

  % init edge struct and LBP schedule
  adj = zeros(T, T);
  for i = 2:T
    adj(i - 1, i) = 1;
    adj(i, i - 1) = 1;
  end
  edgeStruct = DPMP_makeEdgeStructCont(adj, N, model.dim_x, useMex, ...
    maxItersLBP);
  sched = DPMP_makeFwdBwdSched(edgeStruct);

  [nodeBel, x, stats] = ...
    DPMP_Infer(x0, edgeStruct, funHandles, dpmpOpt, sched, []);
end
