close all;
clear variables;
rng(1);

nParticles = 10;                % Number of particles
maxIters = 50;                  % D-PMP Iterations

% Initialize the time series model. In this example, we use a highly
% non-linear model designed to illustrate the drawbacks of particle
% filtering.
model = ex_model(10, 1);
T = 50;

% Generate a sample from the model of length T.
[x_true, y] = ssm_sample_model(model, T);

% Calculate the MAP on a dense discretization with 100 points between -25
% and 25. Also, calculate the log probability and L1 error for the dense
% MAP estimate.
disp('... Calculating dense MAP');
[xDense, xDenseMAP, nodeBelDense] = ssm1d_dense_MAP(y, model, 100, -25, 25);
xDenseMAPlogprob = ssm_log_prob(xDenseMAP, y, model);
xDenseMAPerr = total(abs(x_true - xDenseMAP));

% Init the funHandles struct which will carry our displayFun callback
% function and proposal functions.
funHandles = struct();

% The ex_callback function plots the true state, dense max-marginals, D-PMP
% particles, and D-PMP MAP solution. The display_iter variable controls how
% often the display function is called. By setting it to 1, D-PMP will call
% displayFun after every iteration.
display_iter = 1;
funHandles.displayFun = @(x, xMAP, eS, stats, nB) ex_callback(x, x_true, ...
    xDense(:,:,1), xDenseMAP, nodeBelDense, xMAP, y, eS, stats, nB);

% In this example, we use only one proposal: a forward sampling proposal.
% To propose a new particle at x_t, it randomly select a particle at
% x_{t-1} and then samples from the transition dynamics: p(x_t | x_{t-1}).
funHandles.proposal{1} = @(x, nAdd, eS) ssm_fwd_proposal(x, nAdd, model, eS);
funHandles.propType{1} = 'Forward';

% Here we initialize the particle set that D-PMP starts out with. You can
% use any (dim_x x N x T) matrix you like here but intelligent
% initializations produce much better performance.
x0 = ssm_init_prior(nParticles, T, model);
% x0 = zeros(1, nParticles, T);

% Finally, we run D-PMP. In this example, we are using a helper function
% specifically designed for smoothing time series models. You can check out
% DPMP_smooth to see how DPMP_Infer is actually being called.
disp('... Running D-PMP');
[nodeBelDPMP, xDPMP, stats] = ...
  DPMP_smooth(y, x0, nParticles, model, funHandles, display_iter, maxIters);

%% Plot sample

fig = figure('Position', [0, 0, 840, 380]);

subplot(1, 2, 1);
nodeBelimg = -exp(bsxfun(@minus, nodeBelDense, max(nodeBelDense, [], 2)));
imagesc(1:T, xDense(:,:,1), nodeBelimg');
colormap('gray');
alpha(0.25);

% imagesc flips the y-axis for some reason.
set(gca, 'YDir', 'normal');

hold on;
plot(1:T, x_true);
legend('x_t');
hold off;

subplot(1, 2, 2);
plot(1:T, y, 'ro');
legend('y_t');

%% Plot log-probability per iteration

fig = figure();
plot(1:maxIters, stats.logP, 'r', 'LineWidth', 1.5);

hdenseMAP = refline(0, xDenseMAPlogprob);
set(hdenseMAP, 'LineStyle', '-.');

xlabel('Iteration');
ylabel('Log-probability');
title(sprintf('Nonlinear example model, log-probability vs iter. (%d particles)', nParticles));

%% Plot L1 error per iteration

fig = figure();
plot(1:maxIters, stats.L1err, 'r', 'LineWidth', 1.5);

hdenseMAP = refline(0, xDenseMAPerr);
set(hdenseMAP, 'LineStyle', '-.');

xlabel('Iteration');
ylabel('L1 error');
title(sprintf('Nonlinear example model, L1 error vs iter. (%d particles)', nParticles));
