function model = ex_model(var_u, var_v)
% ex_model Builds a time series model struct for the model as described in
%          Example 1 of http://perso.telecom-paristech.fr/~cappe/Publications/Self-archive/06particle-cmg.pdf
%
%   Input:
%     var_u : variance of u, the latent series noise
%     var_v : variance of v, the observation noise
%   Returns:
%     model : a struct with fields for defining the distributions including ...
%       - dim_x : dimension of the x's
%       - dim_y : dimension of the y's
%       - pi0_logpdf : Function handle to calculate `log p(x_0)`
%       - pi0_rand : Function handle to sample from `p(x_0)`
%       - trans_logpdf : Function handle to calculate `log p(x_t | x_{t-1}, t)`
%       - trans_rand : Function handle to sample from `p(x_t | x_{t-1}, t)`
%       - obs_logpdf : Function handle to calculate `log p(y_t | x_t, t)`
%       - obs_rand : Function handle to sample from `p(y_t | x_t, t)`

  model = struct();
  model.dim_x = 1;
  model.dim_y = 1;

  sig_u = sqrt(var_u);
  sig_v = sqrt(var_v);
  [model.sig_u, model.sig_v] = deal(sig_u, sig_v);

  trans_x = @(x_prev, t) x_prev / 2 + 25 * x_prev / (1 + x_prev ^ 2) + 8 * cos(1.2 * double(t));
  obs_y = @(x_t) (x_t ^ 2) / 20;

  model.pi0_logpdf = @(x_0) normal_logpdf(x_0, 0, sig_u);
  model.pi0_rand = @() sig_u * randn();

  model.trans_logpdf = @(x_t, x_prev, t) normal_logpdf(x_t, trans_x(x_prev, t), sig_u);
  model.trans_rand = @(x_prev, t) trans_x(x_prev, t) + sig_u * randn();

  model.obs_logpdf = @(y_t, x_t, t) normal_logpdf(y_t, obs_y(x_t), sig_v);
  model.obs_rand = @(x_t, t) obs_y(x_t) + sig_v * randn();
end
