function [nodePot, edgePot] = ssm_eval_model(x, y, model, edgeStruct)
% ssm_eval_model Evaluate all of the node and pairwise potentials for a SSM model
%
%   Input:
%     x : dim_x x N x T, particles
%     y : dim_y x T, observed y values
%     model : the SSM struct including fields for evaluating log potentials
%     edgeStruct
%   Returns:
%     nodePot : T x N, unary log potentials
%     edgePot : N x N x T, pairwise log potentials

  assert(size(x, 3) == size(y, 2));

  [T, nEdges, nStates] = deal(edgeStruct.nNodes, edgeStruct.nEdges, ...
      edgeStruct.nStates);

  % Eval node unary potentials
  nodePot = nan(T, max(nStates));

  % First node
  for p = 1:nStates(1)
    nodePot(1, p) = model.pi0_logpdf(x(:,p,1)) ...
                  + model.obs_logpdf(y(:,1), x(:,p,1), 1);
  end

  % Rest
  for t = 2:T
    for p = 1:nStates(t)
      nodePot(t, p) = model.obs_logpdf(y(:,t), x(:,p,t), t);
    end
  end

  % Pairwise potentials
  edgePot = nan(max(nStates), max(nStates), nEdges);
  for t = 2:T
    for p1 = 1:nStates(t - 1)
      for p2 = 1:nStates(t)
        x1 = x(:,p1,t - 1);
        x2 = x(:,p2,t);
        edgePot(p1, p2, t - 1) = model.trans_logpdf(x2, x1, t);
      end
    end
  end
end
