function [nodeBel, x, stats, edgeBel] = DPMP_Infer( ...
  x, ...            % Particles: NDIM x NPARTICLES x NNODES
  edgeStruct, ...   % MRF structure from DPMP_makeEdgeStructCont
  funHandles, ...   % Function callbacks for proposals / eval / display
  opt, ...          % PMP Options from DPMP_setOptions
  sched, ...        % Message passing schedule (e.g. from DPMP_makeFwdBwdSched)
  rho ...           % RMP edge appearance probs.  Set to [] if using LBP.
  )
% DPMP_INFER - Diverse Particle Max-Product inference.
%
% REFERENCES:
% J. Pacheco, S. Zuffi, M. Black, E. Sudderth, "Preserving modes and
%   messages via diverse particle selection.", ICML 2014
%
% J. Pacheco 2014
%

  % unpack useful stuff
  [ nParticles, nNodes, nEdges ] = deal( ...
    double(edgeStruct.nStates), edgeStruct.nNodes, edgeStruct.nEdges );
  [ Temp, maxIters, maxItersLBP, verbose, display_iter, sampleProb, ...
    stepsize, doTieRes, display_iter ] = deal( ...
    opt.Temp, opt.maxIters, opt.maxItersLBP, opt.verbose, opt.display_iter, ...
    opt.sampleProb, opt.stepsize, opt.doTieRes, opt.display_iter );   
      
  % Init stuff
  nParticlesSelect = ceil(nParticles/2);
  cdfSample = [0; cumsum(sampleProb(:))];      
  [nodePot, edgePot] = funHandles.funEvalModel( x, edgeStruct );
  if(display_iter)
    h_aug = figure('InvertHardcopy','off','Color',[1 1 1]);
    h_sub = figure('InvertHardcopy','off','Color',[1 1 1]);
  end
        
  % init stats
  stats.logPall = []; 
  stats.logP = [];
  if strcmpi(opt.msgPass,'trw'), stats.logPbound = [];  end
  stats.iters = [];  
  stats.unique = [];
  stats.tWall = [];
  
  % Main Loop 
  logPbest = -Inf;  xMAP = [];
  for dpmp_iters=1:maxIters
    if verbose, fprintf('Iter %d: \n', dpmp_iters); end
    t_start_pmp = tic();
    
    % sample new particles
    if dpmp_iters > 1
      nStatesAdd = nParticles - nParticlesAccept;
      [~, idxProp] = histc(rand(), cdfSample);
      x_new = funHandles.proposal{idxProp}(x, nStatesAdd, edgeStruct);
      if verbose, fprintf('\t%s Proposal\n', funHandles.propType{idxProp}); end

      % augment particles
      x_aug = zeros(size(x,1), max(nParticles), nNodes);
      x_aug(:, 1:max(nParticlesSelect), :) = x;
      for v=1:nNodes
        if nStatesAdd(v)==0, continue; end
        start_idx = nParticlesAccept(v)+1;
        end_idx = nParticles(v);
        x_aug(:,start_idx:end_idx,v) = x_new(:,1:nStatesAdd(v),v);
      end
    
      % update potentials : 
      % If funEvalNode and funEvalPair handles specified then avoids
      % recomputing potentials on existing particles.  
      if isfield(funHandles, 'funEvalNode') && ~isempty(funHandles.funEvalNode)
        
        % node potentials
        edgeStruct_new = edgeStruct;
        edgeStruct_new.nStates = nStatesAdd;     
        nodePot_new = funHandles.funEvalNode( x_new, edgeStruct_new );
        for v=1:nNodes          
          start_idx = nParticlesAccept(v)+1;
          end_idx = nParticles(v);          
          nodePot(v,start_idx:end_idx) = nodePot_new(v,1:nStatesAdd(v));
        end
                
        % edge potentials
        % TODO: Avoid recomputing pairwise terms for old particles
        edgeStruct.nStates = int32(nParticles);        
        edgePot = funHandles.funEvalPair( x_aug, edgeStruct );        
        
      else
        edgeStruct.nStates = int32(nParticles);
        [ nodePot, edgePot ] = funHandles.funEvalModel( x_aug, edgeStruct );
      end
    else
      x_aug = x;
    end        
    
       
    % run Max-Sum
    t_start = tic;
    switch lower(opt.msgPass)
      case 'lbp'
        if verbose, fprintf('\tRunning LBP...'); end
        if edgeStruct.useMex
          [msg, lbp_iters, logPall] = DPMP_MaxSumC(...
            nodePot,edgePot,int32(edgeStruct.edgeEnds),int32(edgeStruct.nStates),...
            int32(edgeStruct.V),int32(edgeStruct.E),int32(edgeStruct.maxIter),...
            edgeStruct.convTol,stepsize,int32(sched));    
          logPall = logPall(1:lbp_iters);
        else          
          [msg, lbp_iters, logPall] = ...
            DPMP_MaxSum(nodePot, edgePot, edgeStruct, stepsize, sched);  
        end
      case 'trw'        
        if verbose, fprintf('\tRunning TRW...'); end
        if edgeStruct.useMex
          [msg, lbp_iters, logPall, logPbound] = DPMP_TRW_C(...
            nodePot,edgePot,int32(edgeStruct.edgeEnds),int32(edgeStruct.nStates),...
            int32(edgeStruct.V),int32(edgeStruct.E),int32(edgeStruct.maxIter),...
            edgeStruct.convTol,stepsize,int32(sched),rho);    
          logPall = logPall(1:lbp_iters);
          logPbound = logPbound(1:lbp_iters);
        else
          [msg, lbp_iters, ~, logPall, logPbound] = ...
            DPMP_TRW(nodePot, edgePot, edgeStruct, stepsize, sched, rho);          
        end
        stats.logPbound = cat(1,stats.logPbound,logPbound);  
      otherwise
        error('Unrecognized message passing method %s.', opt.msgPass);
    end
    t_stop = toc( t_start );
    stats.tWall = cat(1,stats.tWall,t_stop);
    stats.iters = cat(1,stats.iters,lbp_iters);
    stats.logPall = cat(1,stats.logPall,logPall);
            
    % output stats
    if verbose
      if lbp_iters==maxItersLBP, fprintf('done.  Message passing did not converge!');
      else fprintf('done %d iterations.', lbp_iters);
      end
      fprintf(' (%0.1fs)\n', t_stop);
    end    
    
    % get BP labeling
    switch lower(opt.msgPass)
      case 'lbp'
        nodeBel_aug = DPMP_getLogBeliefs(msg, nodePot, edgePot, edgeStruct);      
      case 'trw'
        nodeBel_aug = DPMP_getTRWLogBeliefs(msg, nodePot, edgePot, edgeStruct, rho);      
    end
    [xBP, nTies] = DPMP_getMAPLabel(doTieRes, nodeBel_aug, nodePot, edgePot, edgeStruct);
    logP_BP = DPMP_getLabelProb(xBP, nodePot, edgePot, edgeStruct ); 
    stats.logPall(end) = logP_BP;
    
    % get current MAP
    if logP_BP > logPbest
      logPbest = logP_BP;
      xMAP = xBP;
    end    
    stats.logP = cat(1,stats.logP,logPbest);
    stats.unique = cat(1,stats.unique,numel(xMAP)-nTies);
    if verbose
      fprintf('\t%d ties of %d nodes.\n', nTies, numel(xMAP)); 
    end
    
    % display
    if (mod(dpmp_iters,display_iter) == 0) && isfield(funHandles,'displayFun');
      figure(h_aug);
      stats = funHandles.displayFun(x_aug,xMAP,edgeStruct,stats,nodeBel_aug);
      drawnow;
    end
        
    % particle selection    
    switch lower(opt.selectType)
      case 'dpmpmax'
        [x, acceptIdx, nParticlesAccept] = DPMP_selectDiverse(...
          x_aug, nParticlesSelect, msg, nodePot, edgePot, edgeStruct, Temp, ...
          opt.msgPass, rho, xMAP);
      case 'dpmpsum'
        [x, acceptIdx, nParticlesAccept] = DPMP_selectLazyGreedy(...
          x_aug, nParticlesSelect, msg, nodePot, edgePot, edgeStruct, Temp, ...
          opt.msgPass, rho, xMAP);
      case 'dpmpsumrmp'
        [x, acceptIdx, nParticlesAccept] = DPMP_selectLazyGreedyRMP(...
          x_aug, nParticlesSelect, msg, nodePot, edgePot, edgeStruct, Temp, ...
          opt.msgPass, rho, xMAP);
%       case 'dpmplp'
%         [x, acceptIdx, nParticlesAccept] = DPMP_selectLP(...
%           x_aug, nParticlesSelect, msg, nodePot, edgePot, edgeStruct, Temp, ...
%           opt.msgPass, rho, nodeBel_aug);
      case 'tpmp'        
        [x, acceptIdx] = DPMP_selectMBest( ...
          x_aug, nParticlesSelect, nodeBel_aug, edgeStruct, xMAP );
        nParticlesAccept = nParticlesSelect;
    end
    S = 1:max(nParticlesAccept);
    [nodePot(:,S), edgePot(S,S,:)] = DPMP_subsetPotentials( ...
      nParticlesAccept, acceptIdx, nodePot, edgePot, edgeStruct);
    edgeStruct.nStates = nParticlesAccept; 
    xMAP = ones( size( xMAP ) );
    
    % clean up unused portions of potentials
    Srest = (max(nParticlesAccept)+1):max(nParticles);
    nodePot(:,Srest) = 0;
    edgePot(:,Srest,:) = 0;
    edgePot(Srest,:,:) = 0;
    
    % end iteration
    t_stop_pmp = toc( t_start_pmp );
    if verbose 
      fprintf('\tdone (%0.3fs)\n', t_stop_pmp);
    end
    
    % display
    if (mod(dpmp_iters,display_iter) == 0) && isfield(funHandles,'displayFun');
      figure(h_sub);
      stats = funHandles.displayFun(x,xMAP,edgeStruct,stats,[]);
      drawnow;
    end    
  end
  
  % final message passing iteration
  t_start = tic;
  if verbose, fprintf('Finalizing:\n'); end
  switch lower(opt.msgPass)
    case 'lbp'
      if verbose, fprintf('\tRunning LBP...'); end
      if edgeStruct.useMex
        [msg_sub, lbp_iters] = DPMP_MaxSumC(...
            nodePot,edgePot,int32(edgeStruct.edgeEnds),int32(edgeStruct.nStates),...
            int32(edgeStruct.V),int32(edgeStruct.E),int32(edgeStruct.maxIter),...
            edgeStruct.convTol,stepsize,int32(sched));  
      else
        [msg_sub, lbp_iters] = DPMP_MaxSum(nodePot, edgePot, edgeStruct, stepsize, sched);  
      end
      if nargout>4
        [nodeBel, edgeBel] = DPMP_getLogBeliefs(msg_sub, nodePot, edgePot, edgeStruct);
      else
        nodeBel = DPMP_getLogBeliefs(msg_sub, nodePot, edgePot, edgeStruct);
      end  
    case 'trw'        
      if verbose, fprintf('\tRunning TRW...'); end
      if edgeStruct.useMex
        [msg_sub, lbp_iters] = DPMP_TRW_C(...
          nodePot,edgePot,int32(edgeStruct.edgeEnds),int32(edgeStruct.nStates),...
          int32(edgeStruct.V),int32(edgeStruct.E),int32(edgeStruct.maxIter),...
          edgeStruct.convTol,stepsize,int32(sched),rho);    
      else
        [msg_sub, lbp_iters] = DPMP_TRW(nodePot, edgePot, edgeStruct, stepsize, sched, rho);  
      end
      if nargout>4
        [nodeBel, edgeBel] = DPMP_getTRWLogBeliefs(msg_sub, nodePot, edgePot, edgeStruct, rho);
      else
        nodeBel = DPMP_getTRWLogBeliefs(msg_sub, nodePot, edgePot, edgeStruct, rho);
      end
  end
  t_stop = toc( t_start );
  
  % output stats
  if verbose
    if lbp_iters==maxItersLBP, fprintf('done.  Message passing did not converge!');
    else fprintf('done %d iterations.', lbp_iters);
    end
    fprintf(' (%0.1fs)\n', t_stop);
  end
  
  % save stats
%   stats.msg_aug = msg;
%   stats.msg = msg_sub;
  stats.nParticlesAccept = nParticlesAccept;
  stats.x_aug = x_aug;
  stats.edgeStruct = edgeStruct;
  
end

