function [nodeBel, iters, logP, logPbound, edgeBel] = DPMP_Infer_TRW(...
  nodePot, ...    % Node (log)-potential: NNODES x NSTATES
  edgePot, ...    % Edge (log)-potential: NSTATES x NSTATES x NEDGES
  edgeStruct, ... % MRF structure from UGM_makeEdgeStruct
  stepsize, ...   % RMP stepsize
  sched, ...      % Message update schedule (e.g. DPMP_makeFwdBwdSched)
  mu ...          % Edge appearance probabilities
  )
% DPMP_INFER_TRW - Run MAP inference with parallel TRW message passing.
%
% J. Pacheco 2014
%
% Based on code by: M. Schmidt
%

  if nargin < 5
    mu = 1;
  end

  [nNodes, maxStates] = size(nodePot);
  nEdges = size(edgePot,3);

  if isscalar(mu) % Weights not provided, construct them using one of the methods below
    % Compute Edge Appearance Probabilities
    if mu == 0
      mu = ones(nEdges,1); % Ordinary BP (not a valid distribution over trees, so not convex)
    elseif mu == 1
      % Generate Random Spanning Trees until all edges are covered
      [nNodes, maxStates] = size(nodePot);
      edgeEnds = edgeStruct.edgeEnds;

      i = 0;
      edgeAppears = zeros(nEdges,1);
      while 1
        i = i+1;
        edgeAppears = edgeAppears+minSpan(nNodes,[edgeEnds rand(nEdges,1)]);
        if all(edgeAppears > 0)
          break;
        end
      end    
      mu = edgeAppears/i;
    elseif mu == 2
      % Compute all spanning trees of the dense graph (not a valid distribution over trees for over graphs)
      mu = ((nNodes-1)/nEdges)*ones(nEdges,1);
    end
  end

  % Run TRW-T
  if edgeStruct.useMex
    [msg, iters, logP, logPbound] = DPMP_TRW_C(nodePot,edgePot,int32(edgeStruct.edgeEnds),...
      int32(edgeStruct.nStates),int32(edgeStruct.V),int32(edgeStruct.E),...
      int32(edgeStruct.maxIter),edgeStruct.convTol,stepsize,int32(sched),mu);    
    logP = logP(1:iters);
    logPbound = logPbound(1:iters);
  else    
    [msg, iters, ~, logP, logPbound] = DPMP_TRW(nodePot,edgePot,edgeStruct,stepsize,sched,mu);   
  end
  
  % compute beliefs
  if nargout>4
    [nodeBel, edgeBel] = DPMP_getTRWLogBeliefs(msg, nodePot, edgePot, edgeStruct, mu);    
  else
    nodeBel = DPMP_getTRWLogBeliefs(msg, nodePot, edgePot, edgeStruct, mu);    
  end
end

