double sumarray( double* arr, int N, int n, int m ) {
/*
 * Summation of an array representing an NxM matrix
 * of elements.  Array index arr[i + N*j] corresponds to
 * matrix element [i][j].  This sums the block-matrix with rows 1:n
 * and columns 1:m.  For Nx1 arrays set m=1;
 */
  double res = 0.0;
  int i, j;
  for (i=0; i<n; ++i ) {
    for (j=0; j<m; ++j )
      res += arr[i + N*j];
  }
  return res;
}

double maxarray( double* arr, int N, int n, int m ) {
/*
 * Maximum of an array representing an NxM matrix
 * of elements.  Array index arr[i + N*j] corresponds to
 * matrix element [i][j].  This maximizes the block-matrix with rows 1:n
 * and columns 1:m.  For Nx1 arrays set m=1;
 */
  double res = -INFINITY;
  int i, j;
  for (i=0; i<n; ++i ) {
    for (j=0; j<m; ++j )
      res = fmax(res, arr[i + N*j]);
  }
  return res;
}

double argmaxarray( double* arr, int N, int n, int m ) {
/*
 * Argmax of an array representing an NxM matrix
 * of elements.  Array index arr[i + N*j] corresponds to
 * matrix element [i][j].  This maximizes the block-matrix with rows 1:n
 * and columns 1:m.  For Nx1 arrays set m=1;
 */
  double res = -INFINITY;
  double maxval = -INFINITY;
  int i, j;
  for (i=0; i<n; ++i ) {
    for (j=0; j<m; ++j )
      if (arr[i + N*j]>maxval) {
	maxval = arr[i + N*j];
	res = i + N*j;
      }
  }
  return res;
}

double logsumexp(double* arr, int N, int n, int m) {
/*
 * Numerically stable summation of an array representing an NxM matrix
 * of elements in logspace.  Array index arr[i + N*j] corresponds to
 * matrix element [i][j].  This sums the block-matrix with rows 1:n
 * and columns 1:m.  For Nx1 arrays set m=1;
 */

  /* find maximum */
  double maxval = -INFINITY;
  int i, j;
  for (i=0; i < n; ++i ) {
    for (j=0; j < m; ++j ) {
      maxval = fmax(arr[i + N*j], maxval);
    }
  }

  /* logsumexp */
  double res = 0.0;
  for (i=0; i < n; ++i ) {
    for (j=0; j < m; ++j ) {
      res += exp(arr[i + N*j] - maxval);
    }
  }

  return log(res) + maxval;
}

void printarray(double* arr, int N) {
  int i;
  printf("\n[  ");
  for (i=0; i<N; ++i)
    printf("%f  ", arr[i]);
  printf("]\n");
}
 
 
/* getLabelProb : Compute probability of label. */
double
getLabelProb(
  const int * mapState,
  const double * nodePot,
  const double * edgePot,
  const int * edgeEnds,
  const int nNodes,
  const int maxState,
  const int nEdges
  )
{
  double logP = 0;
  int n, s, e, n1, n2, s1, s2;

  /* compute unary probabilities */
  for(n=0; n<nNodes; ++n) {
    logP += nodePot[n + mapState[n]*nNodes];
  }

  /* compute edge probabilities */
  for(e=0; e<nEdges; ++e) {
    n1 = edgeEnds[e] - 1;
    n2 = edgeEnds[e + nEdges] - 1;
    s1 = mapState[n1];
    s2 = mapState[n2];
    logP += edgePot[ s1 + maxState*s2 + maxState*maxState*e ];
  }
  return logP;
}


/* getMAPlabel : Compute MAP label as maximizers of node marginals. */
void
getMAPlabel(
  int *mapState,
  const double *nodeBel,
  const int nNodes,
  const int *nStates
  )
{
  int n,s;
  double maxval;
  
  for(n=0; n<nNodes; ++n) {
    maxval = -INFINITY;
    for(s=0; s<nStates[n]; ++s) {
      if(nodeBel[n + nNodes*s] > maxval) {
        maxval = nodeBel[n + nNodes*s];
        mapState[n] = s;
      }
    }
  }
}
