function [cliques, cliqueEdges, cliquePot, separators] = DPMP_makeJunctionTree( nodePot, edgePot, edgeStruct, ordering )
% DPMP_MAKEJUNCTIONTREE - Construct junction tree from graph.
%
% J. Pacheco, 204
% Based on code by M. Schmidt
%
  debug = 0;

  [nNodes,maxState] = size(nodePot);
  nEdges = size(edgePot,3);
  edgeEnds = edgeStruct.edgeEnds;
  V = edgeStruct.V;
  E = edgeStruct.E;
  nStates = edgeStruct.nStates;
  
  if nargin < 4
    ordering = 1:nNodes;
  end

  %% Triangulate graph and find cliques
  adj = UGM_VE2adj(V,E,edgeEnds);
  adj_new = adj;
  cliques = cell(0,1);
  for n = ordering(:)'

      % Add extra
      neighbors = find(adj(:,n));
      for n1 = neighbors(:)'
          for n2 = neighbors(:)'
              if n1 ~= n2
                  adj(n1,n2) = 1;
                  adj(n2,n1) = 1;
                  adj_new(n1,n2) = 1;
                  adj_new(n1,n2) = 1;
              end
          end
      end

      % Add to set of cliques
      clique(n,n) = 1;
      clique(n,neighbors) = 1;

      % Remove node
      adj(n,:) = 0;
      adj(:,n) = 0;

      % If not a subset of an existing clique, add to set of cliques
      cand = [n;neighbors];
      add = 1;
      for c = 1:length(cliques)
          if isempty(setdiff(cand,cliques{c}))
              add = 0;
              break;
          end
      end
      if add
          cliques{end+1,1} = cand;
      end
  end
  nCliques = length(cliques);

  if debug;
      fprintf('Cliques:\n')
      cliques{:}
  end
  
  %% Make initial potentials
  nodePotMissing = ones(nNodes,1);
  edgePotMissing = ones(nEdges,1);
  for c = 1:nCliques
      if debug
        fprintf('Clique %d: %s\n',c,sprintf(' %d',cliques{c}));
      end
      nodes = cliques{c};

      % Initialize clique potentials
      if prod(double(nStates(nodes))) > 5e6
        ME = MException('DPMP_makeJunctionTree:CliqueTooLarge', 'Clique size too large.');
        throw(ME);
      end
      cliquePot{c} = zeros([nStates(nodes)' 1]);

      ind = cell(length(nodes),1);
      for nodeInd = 1:length(nodes)
          ind{nodeInd} = 1:nStates(nodes(nodeInd));
      end

      for nodeInd = 1:length(nodes)
          n = nodes(nodeInd);
          if nodePotMissing(n)
              if debug
                fprintf('Including node potential for node %d in clique %d\n',n,c);
              end
              nodePotMissing(n) = 0;

              ind_sub = ind;
              for s = 1:nStates(n)
                  ind_sub{nodeInd} = s;
                  cliquePot{c}(ind_sub{:}) = cliquePot{c}(ind_sub{:}) + nodePot(n,s);
              end
          end

          edges = UGM_getEdges(n,edgeStruct);
          for e = edges
              n1 = edgeEnds(e,1);
              n2 = edgeEnds(e,2);
              if ismember(n1,nodes) && ismember(n2,nodes) && edgePotMissing(e)
                  if debug
                    fprintf('Including edge potential for edge %d-%d in clique %d\n',edgeEnds(e,1),edgeEnds(e,2),c);
                  end
                  edgePotMissing(e) = 0;

                  ind_sub = ind;
                  nodeInd1 = find(nodes==n1);
                  nodeInd2 = find(nodes==n2);
                  for s1 = 1:nStates(n1)
                      for s2 = 1:nStates(n2)
                          ind_sub{nodeInd1} = s1;
                          ind_sub{nodeInd2} = s2;
                          cliquePot{c}(ind_sub{:}) = cliquePot{c}(ind_sub{:}) + edgePot(s1,s2,e);
                      end
                  end
              end
          end
      end
      if debug
        fprintf('\n');
      end
  end
  if debug
      fprintf('Clique Potentials:\n');
      cliquePot{:}
  end
  
  %% Makes edges between cliques that satisfy RIP
  weights = zeros(0,3);
  for c1 = 1:nCliques
      for c2 = c1+1:nCliques
          sep = length(intersect(cliques{c1},cliques{c2}));
          if sep > 0
              weights(end+1,:) = [c1 c2 -sep];
          end
      end
  end
  edges = find(minSpan(nCliques,weights));
  cliqueEdges = weights(edges,1:2);
  nCliqueEdges = size(cliqueEdges,1);
  [V,E] = UGM_makeEdgeVE(cliqueEdges,nCliques);
  separators = {};
  for e = 1:nCliqueEdges
      separators{e,1} = intersect(cliques{cliqueEdges(e,1)},cliques{cliqueEdges(e,2)});
  end
  if debug
      cliqueEdges
      fprintf('Separators:\n');
      separators{:}
  end

end

