#ifndef _DB_H_
#define _DB_H_

extern "C" {
#include <ociapr.h>
}
#include <stdio.h>

#include "core.h"
#include "mod_tim.h"
#include "mod_rectangle.h"


// some SQL and OCI function codes
#define FT_INSERT                3
#define FT_SELECT                4
#define FT_UPDATE                5
#define FT_DELETE                9
#define FC_OOPEN                14
#define PARSE_NO_DEFER           0
#define PARSE_DEFER              1
#define PARSE_V7_LNG             2
#define NULL_VALUE_RET          -1
#define SELECT_COL_END        1007
#define NO_DATA_FOUND         1403

///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////
// DBMS CLASSES
//

//////////////////////////////
// Some descriptive structures

#define MAX_COLNAME_SIZE 40
#define MAX_COLS         256
#define VTSNAME         "VT$$S"
#define VTENAME         "VT$$E"
#define TTSNAME         "TT$$S"
#define TTENAME         "TT$$E"


// CColDesc and CTableDesc described the tables
struct CColDesc {
  sb2   m_sType;
  long  m_lSize;
  char  m_sName[MAX_COLNAME_SIZE+1];
};
struct CTableDesc {
  long m_lTotalColCount;
  long m_lVtSIndex;
  long m_lVtEIndex;
  long m_lTtSIndex;
  long m_lTtEIndex;
  CColDesc m_aCols[MAX_COLS];
};

//////////////////////////////////////////////////////////////////////
// CSession - the connection to oracle

class CValue {
  friend bool inline operator == ( const CValue & v1, const CValue & v2);

  // allocation values
  char * m_pData;
  long   m_iSize;
  bool   m_bIsNull;
public:
  // data values
  short  m_sFlags;
  short  m_rCode;
  unsigned short  m_irSize;

  CValue(int aSize);
  virtual ~CValue();
  CValue & operator = (const CValue & aValue);
  char * GetData() { return m_pData; }
  void SetData(const char * pData);
  long GetSize() { return m_iSize; }
  bool IsNull() { return m_bIsNull; }
  void SetNull(bool n) { m_bIsNull=n; m_sFlags=NULL_VALUE_RET; }
};

// a few forwarded declarations... 
class CSession;
class CCursor;
class CRow;

//////////////////////////////////////////////////////////////////////
// CSession - the connection to oracle

class CSession {
private:
  // private attributes
  long      m_lCursorCount;
  Lda_Def * m_pLda;
  bool      m_bConnected;
  bool      m_bOwner;

protected:
  // intern methods
  void Init();
public:
  // public methods
  CSession();
  virtual ~CSession();
  void Open(const char *);
  void Open(Lda_Def * pDef);

  void PrintError(int err);
private:
  // Counting the number of cursors
  friend class CCursor;
  void CursorCreated() { m_lCursorCount++; }
  void CursorDeleted() { m_lCursorCount--; }
};



//////////////////////////////////////////////////////////////////////
// CCursor - the manager of queries
class CCursor {
private:
  // private attributes
  CSession * m_pSession;
  CRow     * m_pInsertionRow;
  long       m_lRowCount;
  CTableDesc m_tdDescription;
  Cda_Def    m_cda;
public:
  //public static atributes
  static int verbose; 
 
public:
  // public methods
  /* fetching */ 
  CCursor(CSession * pSession, const CString & aQuery);
  /* insertion */
  CCursor(CSession * pSession, const char * tableName, CRow * pRow);  
  virtual ~CCursor();
  
  CSession *   GetSession()     const { return m_pSession; }
  const CTableDesc * GetDescription() const { return & m_tdDescription; }
  CTableDesc *       GetDescription() { return & m_tdDescription; }
  bool         DoesInsertion()  const { return m_pInsertionRow!=NULL; }
  bool         IsSelect()       const { return m_cda.ft==FT_SELECT; }

  bool         HasVt() const { return (m_tdDescription.m_lVtSIndex>=0) && 
			        (     m_tdDescription.m_lVtEIndex>=0); }
  bool         HasTt() const { return (m_tdDescription.m_lTtSIndex>=0) && 
			        (     m_tdDescription.m_lTtEIndex>=0); }
  void Execute();  
  CString GetOrderedQuery(CString tableName);
  bool IsTime(int i) const { 
    const CTableDesc * pDesc = GetDescription();    
    return ( (i==pDesc->m_lVtSIndex) ||
	     (i==pDesc->m_lVtEIndex) || 
	     (i==pDesc->m_lTtSIndex) ||
	     (i==pDesc->m_lTtEIndex));    
  }
protected:
  // intern methods
  void Parse(const char * aQuery);
  void Describe();
  CString CreateInsertionQuery(const char * tName);
  int Verbose(void);

private:
  // counting rows
  friend class CRow;
  void RowCreated() { m_lRowCount++; }
  void RowDeleted() { m_lRowCount--; }
};

//////////////////////////////////////////////////////////////////////
// CRow - representative of a row
class CRow {
private:
  // attributes
  CCursor * m_pCursor;
  long      m_lInsertionCursorCount;

  CTime     m_tVS, m_tVE;
  CTime     m_tTS, m_tTE;

  CValue  * m_pValue[MAX_COLS];  
  void operator = (const CRow & ) {}
public:
  // public methods
  CRow(CCursor * pCursor);
  void CopyAllDataFrom(const CRow & aRow);

  virtual ~CRow();

  CCursor * GetCursor() const { return m_pCursor; }
  
  bool Fetch();

  bool IsTime(int i) const { 
    const CTableDesc * pDesc = GetCursor()->GetDescription();    
    return ( (i==pDesc->m_lVtSIndex) ||
	     (i==pDesc->m_lVtEIndex) || 
	     (i==pDesc->m_lTtSIndex) ||
	     (i==pDesc->m_lTtEIndex));    
  }
  
  const int    GetSize() const { return m_pCursor->m_tdDescription.m_lTotalColCount; }
  const char * GetValue(int i) const
  { 
    if(m_pValue[i]==NULL)
      return NULL;
    else
      return m_pValue[i]->GetData(); 
  }
  CValue & GetValueObject(int i) const {
    return *m_pValue[i];
  }

  const CTime & VtStart() const { return m_tVS; }
  const CTime & VtEnd()   const { return m_tVE; }
  const CTime & TtStart() const { return m_tTS; }
  const CTime & TtEnd()   const { return m_tTE; }
  void SetVtStart(CTime t);
  void SetVtEnd  (CTime t);
  void SetTtStart(CTime t);
  void SetTtEnd  (CTime t);

  CRectangle * GetBitemp(){
    CRectangle * pRect=new CRectangle;
    pRect->x1=VtStart();
    pRect->x2=VtEnd();
    pRect->y1=TtStart();
    pRect->y2=TtEnd();
    return pRect;
  }
  void SetBitemp(const CRectangle * pRect){
    SetVtStart(pRect->x1);
    SetVtEnd(pRect->x2);
    SetTtStart(pRect->y1);
    SetTtEnd(pRect->y2);
  }
  CRectangle * GetTempVt(){
    CRectangle * pRect=new CRectangle;
    pRect->x1=VtStart();
    pRect->x2=VtEnd();
    pRect->y1=BEGINNING_OF_TIME;
    pRect->y2=END_OF_TIME;
    return pRect;
  }
  void SetTempVt(const CRectangle * pRect){
    SetVtStart(pRect->x1);
    SetVtEnd(pRect->x2);
  }
  CRectangle * GetTempTt(){
    CRectangle * pRect=new CRectangle;
    pRect->x1=BEGINNING_OF_TIME;
    pRect->x2=END_OF_TIME;
    pRect->y1=TtStart();
    pRect->y2=TtEnd();
    return pRect;
  }
  void SetTempTt(const CRectangle * pRect){
    SetTtStart(pRect->y1);
    SetTtEnd(pRect->y2);
  }

protected:
  // intern functions
  void Allocate();

private:
  // counting insertion cursors
  friend class CCursor;
  void CursorCreated() { m_lInsertionCursorCount++; }
  void CursorDeleted() { m_lInsertionCursorCount--; }
};

bool inline operator == ( const CValue & v1, const CValue & v2){
  if (v1.m_bIsNull || v2.m_bIsNull) return v1.m_bIsNull==v2.m_bIsNull;
  int l1=strlen(v1.m_pData);
  int l2=strlen(v2.m_pData);
  if (l1!=l2) return false;
  return 0==memcmp(v1.m_pData, v2.m_pData, l1);  
}

bool inline IdenticalValues(const CRow &r1, const CRow &r2) {
  int s1=r1.GetSize();
  int s2=r2.GetSize();
  if(s1!=s2) return false;
  
  for(int i=0;i<s1;i++){
    if( (!r1.IsTime(i)) && (!r2.IsTime(i)) ){
      if( !(r1.GetValueObject(i)==r2.GetValueObject(i)) ) return false;
    }
  }
  return true;
}
#endif
