#ifndef _MATRIX_H_
#define _MATRIX_H_

#include "core.h"

#include "mod_rectangle.h"

/*
 * The CMatrixRect class is the component from which we 
 * will describe the matrix. CMatrixRect is the basic 
 * element that will be arranged in row/column style.
 * CMatrixRect is a CRectangle but has also pointers
 *   m_pNextX and m_pNextY
 * that points to the next matrix elements.
 *   More over it has some labels that are used by the
 * algorithm. All functionality is put into another class,
 * the CMatrix, that controls the CMatrixRects.
 */

class CMatrixRect : public CRectangle {
  DECLARE_DYNAMIC(CMatrixRect)
public:
  // To make a CMatrixRect act like a CRectangle
  CMatrixRect() {}
  CMatrixRect(const CRectangle & rect) : CRectangle(rect) {}
  CMatrixRect(const CTime & t1, const CTime & t2, 
	      const CTime & t3, const CTime & t4) : CRectangle(t1,t2,t3,t4) {}
  // To make a CMatrixRect be an element of a matrix
  Ref<CMatrixRect> m_pNextX, m_pNextY;
  
  // Labels used by the algorithm
  CString m_sColor;
  bool IsColored() { return m_sColor.GetLength()!=0; }
  bool m_bKeepX, m_bKeepY;
  bool m_bIncluded;
};

// Some inline functions to ease the notion of color

inline bool DifferentColor(CMatrixRect *a, CMatrixRect * b){
  return !(a->m_sColor == b->m_sColor);
}

inline bool IsBlankBorder(CMatrixRect *a, CMatrixRect * b){
  return !(a->IsColored() && b->IsColored());
}

inline bool IsColorBorder(CMatrixRect *a, CMatrixRect * b){
  return a->IsColored() && b->IsColored();
}

inline bool IsBlankStarter(CMatrixRect *a, CMatrixRect * b){
  return (a->IsColored() && !b->IsColored()) ||
    (!a->IsColored() && b->IsColored());
}

inline bool IsColorStarter(CMatrixRect *a, CMatrixRect * b){
  return DifferentColor(a,b);
}


typedef Ref<CMatrixRect> CMatRectRef;

/*
 * CMatrix is created from an initial CRectangle.
 * This creates a matrix containing a single 
 * CMatricRect object. 
 */

class CMatrix : public CAssociate {
  DECLARE_DYNAMIC(CMatrix)
private:
  CMatRectRef  m_rMat;
  CRectRef m_rRect;
  
  void SplitX(const CTime & x);
  void SplitY(const CTime & y);
  void Colorize();
public:
  void Create(CRectangle * );
  void AddRects(CRectSet * );
  void AssociateBlank(CAssociate *);
  void AssociateColor(CAssociate *);
};
typedef Ref<CMatrix> CMatRef;
#endif
