/*!****************************************************************************/
/*! File:      constraint.pl                                                  */
/*! Project:   Tiger                                                          */
/*! Software:  SWI Prolog 2.7.16, Oracle 8.0.4                                */
/*! Export:                                                                   */
/*!            c_add_constraints/1 (interpret.pl)                             */
/*!            c_uc2tuc/1 (interpret.pl)                                      */
/*!            c_check_ics/0 (interpret.pl)                                   */
/*! Import:                                                                   */
/*!            set_of/3 (tiger.pl)                                            */
/*!            s_scan/2 (scanner.pl)                                          */
/*!            p_parse/2 (parser.pl)                                          */
/*!            u_stat/3 (unparser.pl)                                         */
/*!            r_norm_query_expr/3 (rewrite.pl)                               */
/*!            m_sql_exec/5 (meta.pl)                                         */
/*!            t_trans_query/3 (trans.pl)                                     */
/*!            f_raise_err/2 (fejl.pl)                                        */
/*!****************************************************************************/

c_add_constraints([]).
c_add_constraints([IC|ICs]) :- c_add_ic(IC), c_add_constraints(ICs).

c_add_ic(ic(ICN-ICN1,Flag,notnull,TabN,ColN)) :-
  u_stat(query(Flag,sel(all,[cst(1,n(1))],[tab_ref(TabN,nil,nil)],
            isnull(col_ref(nil,ColN,_)),[],true),nil,[]), TDB_IC, []),
  m_sql_exec([],dml(nil,insert("TDB$ICS",nil,
           values([ICN,cst(TDB_IC,s(l))]))),
           nil, _, _),
  m_sql_exec([],dml(nil,insert("TDB$DEPS",nil,
           values([ICN1,cst("NN",s(32)),cst(TabN,s(32)),
                   cst("N",s(1)),cst("N",s(1))]))),
           nil, _, _).
c_add_ic(ic(ICN-ICN1,Flag,primary_key,TabN,ColN)) :- 
  u_stat(query(Flag,sel(all,[cst(1,n(i))],
      [tab_ref(TabN,"T1",nil),tab_ref(TabN,"T2",nil)],
      and(rel(nq,rowid("T1"),rowid("T2")),
          rel(eq,col_ref("T1",ColN,_),col_ref("T2",ColN,_))),
                       [],true),
               nil,[]),
         TDB_IC,[]),
  m_sql_exec([],dml(nil,insert("TDB$ICS",nil,
           values([ICN,cst(TDB_IC,s(l))]))),
           nil, _, _),
  m_sql_exec([],dml(nil,insert("TDB$DEPS",nil,
           values([ICN1,cst("PK",s(32)),cst(TabN,s(32)),
                   cst("N",s(1)),cst("N",s(1))]))),
           nil, _, _).
c_add_ic(ic(ICN-ICN1,Flag,references(TabN1,ColN1),TabN,ColN)) :- 
  u_stat(query(Flag,sel(all,[cst(1,n(i))],
                       [tab_ref(TabN,"T1",nil)],
                       not(in([col_ref("T1",ColN,_)],
                              sel(all,[col_ref("T2",ColN1,_)],
                                  [tab_ref(TabN1,"T2",nil)],true,[],true))),
                       [],true),
               nil,[]),
         TDB_IC,[]),
  m_sql_exec([],dml(nil,insert("TDB$ICS",nil,
           values([ICN,cst(TDB_IC,s(l))]))),
           nil, _, _),
  m_sql_exec([],dml(nil,insert("TDB$DEPS",nil,
           values([ICN1,cst("RI",s(32)),cst(TabN,s(32)),
                   cst("N",s(1)),cst("N",s(1))]))),
           nil, _, _),
  m_sql_exec([],dml(nil,insert("TDB$DEPS",nil,
           values([ICN1,cst("RI",s(32)),cst(TabN1,s(32)),
                   cst("N",s(1)),cst("Y",s(1))]))),
           nil, _, _).

/******************************************************************************/
/* Mode:	c_uc2tuc(+TableName)                                          */
/* Purpose:	migrates integrity constraints involving 'TableName'          */
/* Example:                                                                   */
/* Sideeffects:	DB is modified (TDB$ICS, Oracle constraints)                  */
/* Call:	exits always						      */
/* Redo:	fails always						      */
/******************************************************************************/

%% c_migrate_uc_pk(+TableName)
%% migrates an upward compatible primary key
c_migrate_uc_pk(TabName) :-
  m_sql_exec([var(TabName,s(_))],
    lit("SELECT UC.CONSTRAINT_NAME, UCC.COLUMN_NAME \
         FROM USER_CONSTRAINTS UC, USER_CONS_COLUMNS UCC \
         WHERE UC.CONSTRAINT_TYPE='P' \
         AND UC.STATUS='ENABLED' \
         AND UC.TABLE_NAME=:1 \
         AND UCC.CONSTRAINT_NAME=UC.CONSTRAINT_NAME"),
    nil, query,
    [ConsName,AttrName]),
  !,
  m_sql_exec([], alter(TabName,disable(ConsName)), nil, _, _),
  u_stat(query(nil,sel(all,[cst(1,n(i))],
      [tab_ref(TabName,"T1",nil),tab_ref(TabName,"T2",nil)],
      and(rel(nq,rowid("T1"),rowid("T2")),
          rel(eq,col_ref("T1",AttrName,_),col_ref("T2",AttrName,_))),
                       [],true),
               nil,[]),
         Query,[]),
  m_sql_exec([],dml(nil,insert("TDB$ICS",nil,
           values([cst(ConsName,s(_)),cst(Query,s(l))]))),
           nil, _, _),
  m_sql_exec([],dml(nil,insert("TDB$DEPS",nil,
           values([cst(ConsName,s(32)),cst("PK",s(32)),cst(TabName,s(32)),
                   cst("N",s(1)),cst("N",s(1))]))),
           nil, _, _).
c_migrate_uc_pk(_).

%% c_migrate_uc_ri(+TableName)
%% migrates an upward compatible referential integrity
c_migrate_uc_ri(TabName) :-
  m_sql_exec([var(TabName,s(_))],
    lit("SELECT UC.CONSTRAINT_NAME, UCC.COLUMN_NAME, R_UCC.TABLE_NAME, R_UCC.COLUMN_NAME
         FROM USER_CONSTRAINTS UC, USER_CONS_COLUMNS UCC, USER_CONS_COLUMNS R_UCC
         WHERE UC.CONSTRAINT_TYPE='R'
         AND UC.STATUS='ENABLED'
         AND UCC.TABLE_NAME=:1
         AND UCC.CONSTRAINT_NAME=UC.CONSTRAINT_NAME
         AND R_UCC.CONSTRAINT_NAME=UC.R_CONSTRAINT_NAME"),
    nil, query,
    [ConsName,AttrName,RTabName,RAttrName]),
  m_sql_exec([], alter(TabName,disable(ConsName)), nil, _, _),
  u_stat(query(nil,sel(all,[cst(1,n(i))],
                       [tab_ref(TabName,"T1",nil)],
                       not(in([col_ref("T1",AttrName,_)],
                              sel(all,[col_ref("T2",RAttrName,_)],
                                  [tab_ref(RTabName,"T2",nil)],true,[],true))),
                       [],true),
               nil,[]),
         Query,[]),
  m_sql_exec([],dml(nil,insert("TDB$ICS",nil,
           values([cst(ConsName,s(_)),cst(Query,s(l))]))),
           nil, _, _),
  m_sql_exec([],dml(nil,insert("TDB$DEPS",nil,
           values([cst(ConsName,s(32)),cst("RI",s(32)),cst(TabName,s(32)),
                   cst("N",s(1)),cst("N",s(1))]))),
           nil, _, _),
  m_sql_exec([],dml(nil,insert("TDB$DEPS",nil,
           values([cst(ConsName,s(32)),cst("RI",s(32)),cst(RTabName,s(32)),
                   cst("N",s(1)),cst("Y",s(1))]))),
           nil, _, _),
  fail.
c_migrate_uc_ri(_).

%% c_migrate_uc_indirect_ri(+TableName)
%% migrates an indirect upward compatible referential integrity
c_migrate_uc_indirect_ri(RTabName) :-
  m_sql_exec([var(RTabName,s(_))],
    lit("SELECT UC.CONSTRAINT_NAME, UC.TABLE_NAME, R_UCC.COLUMN_NAME \
         FROM USER_CONSTRAINTS UC, USER_CONS_COLUMNS R_UCC \
         WHERE UC.CONSTRAINT_TYPE='R' \
         AND UC.STATUS='ENABLED' \
         AND R_UCC.TABLE_NAME=:1 \
         AND R_UCC.CONSTRAINT_NAME=UC.R_CONSTRAINT_NAME"),
    nil, query,
    [ConsName,TabName,RAttrName]),
  m_sql_exec([var(ConsName,s(_))],
    lit("SELECT T.COLUMN_NAME \
         FROM USER_CONS_COLUMNS T \
         WHERE T.CONSTRAINT_NAME=:1"),
    nil, query,
    [AttrName]),
  m_sql_exec([], alter(TabName,disable(ConsName)), nil, _, _),
  u_stat(query(nil,sel(all,[cst(1,n(i))],
                       [tab_ref(TabName,"T1",nil)],
                       not(in([col_ref("T1",AttrName,_)],
                              sel(all,[col_ref("T2",RAttrName,_)],
                                  [tab_ref(RTabName,"T2",nil)],true,[],true))),
                       [],true),
               nil,[]),
         Query,[]),
  m_sql_exec([],dml(nil,insert("TDB$ICS",nil,
           values([cst(ConsName,s(_)),cst(Query,s(l))]))),
           nil, _, _),
  m_sql_exec([],dml(nil,insert("TDB$DEPS",nil,
           values([cst(ConsName,s(32)),cst("RI",s(32)),cst(TabName,s(32)),
                   cst("N",s(1)),cst("N",s(1))]))),
           nil, _, _),
  m_sql_exec([],dml(nil,insert("TDB$DEPS",nil,
           values([cst(ConsName,s(32)),cst("RI",s(32)),cst(RTabName,s(32)),
                   cst("N",s(1)),cst("Y",s(1))]))),
           nil, _, _),
  fail.
c_migrate_uc_indirect_ri(_).

%% c_uc2tuc(+TableName)
%% turns uc constraints involving table 'TabName' into tuc constraints
c_uc2tuc(TabName) :-
  c_migrate_uc_indirect_ri(TabName),
  c_migrate_uc_ri(TabName),
  c_migrate_uc_pk(TabName).

/******************************************************************************/
/* Mode:	c_check_ics                                                   */
/* Purpose:	checks all relevant integrity constraints                     */
/* Example:                                                                   */
/* Sideeffects:	error flag is set if DB is inconsistent                       */
/* Call:	exits if DB is consistent                                     */
/* Redo:	fails always						      */
/******************************************************************************/

%% c_check_ics
%% checks whether the DB is still consistent
c_check_ics :-
  clause(transaction_status(XRMS,_,ICs), true),
  findall(cst(R,s(_)), member(R,XRMS), Values),
  (  Values==[]
  -> ICs1=ICs
  ;  set_of(ICName,
            m_sql_exec([],
              query(nil,sel(all,
                 [col_ref("T","OBJECT",s(l))],
                 [tab_ref("TDB$DEPS","T",nil)],
                 and(rel(nq,col_ref("T","TYPE",s(_)),cst("VIEW",s(_))),
                     in([col_ref("T","BASE_OBJECT",s(l))],Values)),[],true),
                    nil,[]),
              nil,query,[ICName]),
            ICNames),
     union(ICNames, ICs, ICs1)
  ),
  c_check_ics2(ICs1).

c_check_ics2([]).
c_check_ics2([IC|ICs]) :-
  m_sql_exec([],
           query(nil,sel(all,
                        [col_ref("T","QUERY",s(l))],
                        [tab_ref("TDB$ICS","T",nil)],
                        rel(eq,col_ref("T","NAME",s(_)),cst(IC,s(_))),[],true),
                 nil,[]),
           nil,query,[Query]),
  s_scan(Query, Tokens),
  p_parse(Tokens, query(Flag,QExpr,_,_)),
  r_norm_query_expr(QExpr, Flag, QExprN),
  t_trans_query(query(Flag,QExprN,nil+nil,[]), _, QueryStmt),
  (  m_sql_exec([], QueryStmt, nil, query, [_])
  -> f_raise_err(inconsistent,IC)
  ;  true
  ),
  c_check_ics2(ICs).

/******************************************************************************/
/* Mode:	c_drop_ics(+TableName)                                        */
/* Purpose:	drops all integrity constraints that involve 'TableName'      */
/* Example:                                                                   */
/* Sideeffects:	DB is modified (TDB$ICS)                                      */
/* Call:	exits always						      */
/* Redo:	fails always						      */
/******************************************************************************/

%% c_drop_ics(+TableName)
%% drops entries in TDB$ICS related to table 'TabName'
c_drop_ics(TabName) :-
  m_sql_exec([],
    dml(nil,delete("TDB$ICS",nil,
        or(rel(eq,col_ref("TDB$ICS","TAB_NAME",s(_)),cst(TabName,s(_))),
           rel(eq,col_ref("TDB$ICS","R_TAB_NAME",s(_)),cst(TabName,s(_)))))),
           nil,dml,_).

