/*!****************************************************************************/
/*! File:      deps.pl                                                        */
/*! Project:   Tiger                                                          */
/*! Software:  SWI Prolog 2.7.16, Oracle 8.0.4                                */
/*! Export:                                                                   */
/*!            d_get_deps/2 (interpret.pl)                                    */
/*!            d_check_deps/0 (interpret.pl)                                  */
/*!            d_check_meta_oper/0 (interpret.pl)                             */
/*!            d_process_deps/0 (interpret.pl)                                */
/*! Import:                                                                   */
/*!            set_of/3 (tiger.pl)                                            */
/*!            m_sql_exec/5 (meta.pl)                                         */
/*!            f_raise_err/2 (fejl.pl)                                        */
/*!****************************************************************************/

%% d_get_deps(+(Query)Expr, -(RelName-IsView))
%% enumerates relation/names that occur in (query)expressions
d_get_deps(sel(_,_,F,W,_,_),R)    :- !, d_get_deps(F,R); d_get_deps(W,R).
d_get_deps(except(A,B),R)         :- !, d_get_deps(A,R); d_get_deps(B,R).
d_get_deps(union(A,B),R)          :- !, d_get_deps(A,R); d_get_deps(B,R).
d_get_deps(isct(A,B),R)           :- !, d_get_deps(A,R); d_get_deps(B,R).
d_get_deps(exists(A),R)           :- !, d_get_deps(A,R).
d_get_deps(in(_,A),R)             :- !, d_get_deps(A,R).
d_get_deps(and(A,B),R)            :- !, d_get_deps(A,R), d_get_deps(B,R).
d_get_deps(or(A,B),R)             :- !, d_get_deps(A,R), d_get_deps(B,R).
d_get_deps(not(A),R)              :- !, d_get_deps(A,R).
d_get_deps(rel(_,_,A),R)          :- !, d_get_deps(A,R).
d_get_deps([tab_ref(derived_tab(_,A),_,_)|B],R) :- !,
  d_get_deps(A,R); d_get_deps(B,R).
d_get_deps([tab_ref(T,_,_)|B],R)  :-
  (  m_sql_exec([var(T,s(_))],
              lit("SELECT 1 FROM USER_VIEWS WHERE VIEW_NAME=:1 \
                   UNION \
                   SELECT 1 FROM TDB$VIEWS WHERE NAME=:1"),
               nil, query, [_])
  -> R=T-"Y"
  ;  R=T-"N"
  )
  ; d_get_deps(B,R).

/******************************************************************************/
/* Mode:	d_check_deps                                                  */
/* Purpose:	checks whether a table can be dropped                         */
/* Example:                                                                   */
/* Sideeffects:	error flag is set if table cannot be dropped                  */
/* Call:	exits if dependencies allow to process stmt                   */
/* Redo:	fails always						      */
/******************************************************************************/

%% checks whether a table dropping violates dependencies
d_check_deps :-
  (  clause(candidate_action(drop,TabN), true)
  -> (  m_sql_exec([var(TabN,s(_))],
                 lit("SELECT 1 FROM TDB$DEPS T WHERE T.BASE_OBJECT=:1 \
                      AND T.IS_PROHIBITIVE='Y'"), nil, query, [1])
     -> f_raise_err(deps_violation,_)
     ;  true
     )
  ;
  true
  ).

/******************************************************************************/
/* Mode:        d_check_meta_oper                                             */
/* Purpose:     to check (and prevent) any illegal attempt at                 */
/*              modifying/dropping a metatable                                */
/* Example:                                                                   */
/* Sideeffects:	error flag is set if attempted action concerns a              */
/*              metatable                                                     */
/* Call:        exits if no metatable is affected by candidate action         */
/* Redo:                                                                      */
/******************************************************************************/

d_check_meta_oper :- 
  ((clause(candidate_action(Action,TabN), true),
    member(Action, [drop,delete,insert,update]), 
    append([0'T,0'D,0'B,0'$|_], _, TabN))
  -> f_raise_err(illegal_operation_on_metatable,_)
  ; true
  ). 

/******************************************************************************/
/* Mode:	d_process_deps                                                */
/* Purpose:	updates dependencies, i.e., removes dangling ics and updates  */
/*              the XMRS                                                      */
/* Example:                                                                   */
/* Sideeffects:	transaction_status updated                                    */
/* Call:	exits always                                                  */
/* Redo:	fails always						      */
/******************************************************************************/

%% update_XMRS(+TableName)
update_XMRS(TabN) :-
  comp_deps_down([TabN], [], XMRS),
  check4views(XMRS),
  comp_deps_up(XMRS, [], XMRS1),
  retract(transaction_status(XMRS2,LCT,ICs)),
  union(XMRS1, XMRS2, XMRS3),
  assert(transaction_status(XMRS3,LCT,ICs)).

%% comp_deps_down(+RelationSet, +IntermediateClosure, -FinalClosure)
comp_deps_down([], XMRS, XMRS).
comp_deps_down([T|R], XMRS, XMRS1) :-
  set_of(TabN,
         m_sql_exec([var(T,s(_))],
                  lit("SELECT REFERENCED_NAME FROM USER_DEPENDENCIES \
                       WHERE NAME=:1"), nil, query, [TabN]),
         TabNs),
  subtract(TabNs, XMRS, TabNs1),
  union(R, TabNs1, R1),
  comp_deps_down(R1, [T|XMRS], XMRS1).

%% check4views(+TableSet)
%% checks for a temporal view update
check4views(XMRS) :-
  findall(cst(R,s(_)), member(R,XMRS), Values),
  (  m_sql_exec([],
        query(nil,sel(all,[col_ref("T","OBJECT",s(l))],
                      [tab_ref("TDB$DEPS","T",nil)],
                      and(rel(eq,col_ref("T","TYPE",s(32)),cst("VIEW",s(_))),
                          in([col_ref("T","OBJECT",s(l))],Values)),[],true),
                      nil,[]),
              nil,query,[ViewN])
  -> f_raise_err(view_update,ViewN)
  ;  true
  ).

%% comp_deps_up(+RelationSet, +IntermediateClosure, -FinalClosure)
comp_deps_up([T|R], XMRS, XMRS1) :-
  set_of(TabN,
         m_sql_exec([var(T,s(_))],
                  lit("SELECT OBJECT FROM TDB$DEPS \
                       WHERE BASE_OBJECT=:1 AND TYPE='VIEW'\
                       UNION \
                       SELECT NAME FROM USER_DEPENDENCIES \
                       WHERE REFERENCED_NAME=:1"), nil, query, [TabN]),
         TabNs),
  subtract(TabNs, XMRS, TabNs1),
  union(R, TabNs1, R1),
  comp_deps_down(R1, [T|XMRS], XMRS1).


%% deletes dependencies after a drop operation
d_process_deps :-
  (  clause(candidate_action(Action,TabN), true)
  -> (  Action == drop
     -> m_sql_exec([var(TabN,s(_))],
		 lit("DELETE FROM TDB$ICS WHERE NAME IN (SELECT OBJECT \
			FROM TDB$DEPS WHERE BASE_OBJECT=:1)"), _, dml, _),
	m_sql_exec([var(TabN,s(_))],
		 lit("DELETE FROM TDB$VIEWS WHERE NAME = :1"), _, dml, _),
        m_sql_exec([var(TabN,s(_))],
                 lit("DELETE FROM TDB$DEPS WHERE OBJECT=:1"), nil, dml, _),
	m_sql_exec([var(TabN,s(_))],
		 lit("DELETE FROM TDB$DEPS WHERE OBJECT IN (SELECT OBJECT \
			FROM TDB$DEPS WHERE BASE_OBJECT=:1)"), nil, dml, _)
     ;  memberchk(Action,[insert,delete,update])
     -> update_XMRS(TabN)
     ;  /* else */
	format("~2ninternal error: action '~w' in process_deps~2n", [Action]),
	fail
     )
  ;  true
  ).

