/*!****************************************************************************/
/*! File:      fejl.pl                                                        */
/*! Project:   Tiger                                                          */
/*! Software:  SWI Prolog 2.7.16, Oracle 8.0.4                                */
/*! Export:                                                                   */
/*!            f_raise_err/2 (interpret.pl,scanner.pl,parser.pl,rewrite.pl,   */
/*!                                deps.pl,eval.pl,trans.pl,constraint.pl)    */
/*!                                                                           */
/*!            f_print_err/0 (interpret.pl)                                   */
/*! Import:                                                                   */
/*!            u_fac/4 (unparser.pl)                                          */
/*!****************************************************************************/

/******************************************************************************/
/* Mode:	f_raise_err(+ErrorNumber, +ErrorPointer)	              */
/* Purpose:	Sets an error flag containing error# and error-pointer        */
/* Example:		       				                      */
/* Sideeffects:	Error flag is asserted.		 			      */
/* Call:        Fails always						      */
/* Redo:        Fails always						      */
/******************************************************************************/

f_raise_err(ErrorNumber, ErrorPointer) :-
  \+ error(_, _),
  assertz(error(ErrorNumber, ErrorPointer)),
  fail.

/******************************************************************************/
/* Mode:	f_print_err					              */
/* Purpose:	Writes an error message to the screen                         */
/* Example:								      */
/* Sideeffects:	Error message is written to screen			      */
/* Call:        Fails always						      */
/* Redo:        Fails always						      */
/******************************************************************************/

f_print_err :-
  nl,
  (  clause(error(Id,Args), true)
  -> error_msg(Id, Args),
     retractall(error(_,_))
  ;  format("Syntax error; please report to tigeradm@cs.auc.dk", [])
  ),
  nl, nl.

error_msg(file_err, F) :- !,
  format("File error: '~w' does not exist or wong permissions", [F]).

error_msg(missing_eos, _) :- !,
  format("String not properly terminated: missing '", []).

error_msg(orderby_expected, id(S)) :- !,
  format("ORDER BY clause or coalescing expected instead of '~s'", [S]).
error_msg(orderby_expected, X) :- !,
  format("ORDER BY clause or coalescing expected instead of '~w'", [X]).

error_msg(db_already_open, _) :- !,
  format("A database is open; close it before opening a new one", []).

error_msg(time_fmt, X) :- !,
  u_fac(X,ins,Str,[]),
  format("Interval format error: '~s' not recognized", [Str]),
  nls_period_sep(_,Sep),
  format("~n  An interval must be of the form 'TS1 ~w TS2'", [Sep]),
  format("~n  A timestamp must be of the form YYYY/MM/DD~~HH24:Mi:SS", []),
  format("~n  A timestamp can be truncated, i.e., specifying YYYY is OK", []).

error_msg(nls_date_format, _) :- !,
  format("You may not change NLS_DATE_FORMAT (implementation restriction)", []),
  format("~n  All timestamps must be of the form YYYY/MM/DD~~HH24:Mi:SS", []),
  format("~n  Timestamps can be truncated, i.e., specifying YYYY is OK", []).

error_msg(nls_period_sep, _) :- !,
  format("Illegal NLS_PERIOD_SEP specification", []),
  format("~n  Example: ALTER SESSION SET NLS_PERIOD_SEP = '-'", []),
  format("~n  The separator must be a single char.", []),
  format("~n  It may not be a letter, number, /, ~~, or :", []).

error_msg(no_db_open, _) :- !,
  format("No database is open", []).

error_msg(unknown_type, Expr) :- !,
  format("Can't determine the type of '~w'", [Expr]).

error_msg(unknown_column, Expr) :- !,
  u_fac(Expr,ins,Str,[]),
  format("Column '~s' is not visible", [Str]).

error_msg(ts_in_tuc, D) :- !,
  format("You may not use ~s in a TUC statement", [D]).

error_msg(expected_instead, T1-T2) :- !,
  format("'~w' expected instead of '~w'", [T1,T2]).

error_msg(deps_violation, _) :- !,
  format("operation not allowed because of a dependency violation", []).

error_msg(not_implemented, M) :- !,
  format("Sorry, '~s' has not been implemented yet", [M]).

error_msg(inconsistent, ICName) :- !,
  format("Database is inconsistent; constraint '~s' is violated!", [ICName]).

error_msg(view_update, ViewName) :- !,
  format("View updates (of '~s') are not supported.", [ViewName]).

error_msg(bi_tab_expected, TabN) :- !,
  format("With this statement '~s' must be bitemporal.", [TabN]).

error_msg(vt_tab_expected, TabN) :- !,
  format("With this statement '~s' must be a valid time table.", [TabN]).

error_msg(tt_tab_expected, TabN) :- !,
  format("With this statement '~s' must be a transaction time table.", 
         [TabN]).

error_msg(altering_vt_table, _) :- !,
  format("Table already possesses implicit valid time attribute", []).

error_msg(altering_tt_table, _) :- !,
  format("Table already possesses implicit transaction time attribute", []).

error_msg(vt_overspecified, _) :- !, 
  format("Only one valid time flag is allowed", []).

error_msg(tt_overspecified, _) :- !, 
  format("Only one transaction time flag is allowed", []).

error_msg(on_off_expected, Y) :- !, 
  format("ON or OFF expected instead of ~s", [Y]).

error_msg(select_item_expected, X) :- !, 
  format("~s is not a legal expression in a SELECT clause", [X]).

error_msg(keyword_table_view_expected, T) :- !, 
  format("...", [T]).

error_msg(datatype_expected, T) :- !, 
  format("~s is not a valid datatype", [T]).

error_msg(unknown_expr, E) :- !, 
  format("Expression '~w' is unknown", [E]).

error_msg(ambiguous_column, col_ref(nil,C,_)) :- !, 
  format("'~s' is an ambiguous column name", [C]).

error_msg(vtr_in_delete, _) :- !, 
  format("Valid time range cannot be specified in a DELETE statement", []). 

error_msg(vt_range_notallowed, _) :- !, 
  format("Valid time range cannot be specified in this context", []).

error_msg(unknown_ic, IC) :- !, 
  format("~s is an unknown integrity constraint", [IC]).

error_msg(unknown_from_item, H-Flag) :- !, 
  format("Unexpected construct in FROM clause", [H,Flag]).

error_msg(duration_not_supported, _) :- !, 
  format("Sorry, DURATION is still not supported", []). 

error_msg(illegal_operation_on_metatable, _) :- !, 
  format("Illegal attempt to modify/drop a Tiger metatable", []). 

error_msg(Id, Args) :- !,
  format("unrecognized error: id='~w', args='~w'", [Id,Args]).

