/*!****************************************************************************/
/*! File:      meta.pl                                                        */
/*! Project:   Tiger                                                          */
/*! Software:  SWI Prolog 2.7.16, Oracle 8.0.4                                */
/*! Export:                                                                   */
/*!            m_get_col/4 (rewrite.pl)                                       */
/*!            m_update_cols/2 (interpret.pl)                                 */
/*!            m_table_type/3 (interpret.pl,parser.pl,rewrite.pl,eval.pl,     */
/*!                                trans.pl)                                  */
/*!            m_sql_exec/5 (interpret.pl,views.pl,deps.pl,eval.pl,           */
/*!                                constraint.pl)                             */
/*! Import:                                                                   */
/*!            u_stat/3 (unparser.pl)                                         */
/*!            r_type/2 (rewrite.pl)                                          */
/*!            r_leftmost_select/2 (rewrite.pl)                               */
/*!            z_reduce_start/2 (zeit.pl)                                     */
/*!            z_reduce_end/2 (zeit.pl)                                       */
/*!****************************************************************************/

%% map_ora_type(+OracleType, -ColumnType)
map_ora_type("NUMBER", n(_)).
map_ora_type("FLOAT", n(f)).
map_ora_type("LONG", s(l)).
map_ora_type("VARCHAR2", s(_)).
map_ora_type("CHAR", s(_)).
map_ora_type("DATE", ts).

%% handle_periods(+ColumnName, +ColumnType, -ColumnName, -ColumnType)
handle_periods(ColName, ts, Dim, per) :- append(Dim, "$S", ColName), !.
handle_periods(ColName, ts, _, _)      :- append(_, "$E", ColName), !, fail.
handle_periods(ColName, ColType, ColName, ColType).

%% m_cache_cols(TabN)
m_cache_cols(TabN) :-
  findall(ColN1-ColT1,
          ( m_sql_exec([var(TabN,s(_))],
	             lit("SELECT COLUMN_NAME, DATA_TYPE FROM COLS \
	                  WHERE TABLE_NAME = :1 ORDER BY COLUMN_ID"),
	             nil, query,
	             [ColN,OraType]),
	    map_ora_type(OraType, ColT),
            handle_periods(ColN, ColT, ColN1, ColT1)),
          ColInfo),
  assert(cols(TabN,ColInfo)).

%% m_get_col(+TabDef, +ExcludedColumnNames, -ColumnName, -ColumnType)
m_get_col(derived_tab(flag(_,tt(seq,_),_),_), NoNames, "TT$", per) :-
  \+ memberchk("TT$", NoNames).
m_get_col(derived_tab(flag(vt(VTM,_),_,VTR),_), NoNames, "VT$", per) :-
  \+ (VTM \== seq, VTR == nil,  memberchk("VT$", NoNames)).
m_get_col(derived_tab(_,QExpr), NoNames, ColName, ColType) :- !,
  r_leftmost_select(QExpr, SelList),
  member(sel_col(Expr,ColName), SelList),
    \+ memberchk(ColName, NoNames),
    r_type(Expr, ColType).
m_get_col(TabN, NoNames, ColN, ColT) :-
  (  clause(cols(TabN,ColInfo), true)
  -> true
  ;  m_cache_cols(TabN),
     clause(cols(TabN,ColInfo), true)
  ),
  member(ColN-ColT, ColInfo),
  \+ memberchk(ColN, NoNames).

%% m_update_cols(+TableName, +ColumnName)
%% ColumnName must either be "TT$" or "VT$"
m_update_cols(TabN, ColN) :-
  retract(cols(TabN,ColInfo)),
  assert(cols(TabN,[ColN-per|ColInfo])),
  !.

/******************************************************************************/

%% m_table_type(+TableDefinition, +ExcludedTypes, -TableType)
m_table_type(derived_tab(flag(vt(VTM,_),tt(TTM,_),VTR),_), ExclTypes, TabType) :-
  (  VTM == tuc, TTM == tuc, VTR == nil
  -> TabType = sn
  ;  TTM == tuc
  -> TabType = vt
  ;  VTM == tuc, VTR = nil
  -> TabType = tt
  ;  TabType = bi
  ),
  \+ memberchk(TabType, ExclTypes).
m_table_type(TabN, ExclTypes, TabT) :-
  (  clause(cols(TabN,ColInfo), true)
  -> true
  ;  m_cache_cols(TabN),
     clause(cols(TabN,ColInfo), true)
  ),
  (  memberchk("VT$"-_,ColInfo), memberchk("TT$"-_,ColInfo)
  -> TabT = bi
  ;  memberchk("VT$"-_,ColInfo)
  -> TabT = vt
  ;  memberchk("TT$"-_,ColInfo)
  -> TabT = tt
  ;  TabT = sn
  ),
  \+ memberchk(TabT, ExclTypes).

/******************************************************************************/
/* Mode:              get_result(+StmtType, +Cursor, +ColTypes, -NumTuples,   */
/*                               -Result                                      */
/* Purpose:           retrieves the result of a stmt; called from m_sql_exec  */
/* Example:                                                                   */
/* Sideeffects:       moves DB cursor                                         */
/* Call:              exits always                                            */
/* Redo:              succeeds if there are more result tuples (queries only  */
/******************************************************************************/

%% sql_describe_query(+ColumnTypes,+Cursor,+ArgCount,-HeadColumn)
%% ColumnTypes is a list of types (e.g., [per,n(i),s(32)]) or nil
sql_describe_query([per|T],C,AC,[V|Vs]) :-
  ora_describe_column(C,AC,X), !,
  (append(V,"$$S",X) -> true; V="PERIOD"),
  AC2 is AC+2,
  sql_describe_query(T,C,AC2,Vs).
sql_describe_query([_|T],C,AC,[V|Vs]) :-
  ora_describe_column(C,AC,V), !,
  AC1 is AC+1,
  sql_describe_query(T,C,AC1,Vs).
sql_describe_query([],_,_,[]).
sql_describe_query(nil,C,AC,[V|Vs]) :-
  ora_describe_column(C,AC,V), !,
  AC1 is AC+1,
  sql_describe_query(nil,C,AC1,Vs).
sql_describe_query(nil,_,_,[]).

%% make_period(+StartPoint, +EndPoint, +PeriodSep, -Period)
make_period(null, null, _, null) :- !.
make_period(S, null, Sep, P)     :- !,
  z_reduce_start(S,S1),
  append(S1, [0' , Sep, 0' , 0'N, 0'O, 0'W], P).
make_period(null, E, Sep, P)     :- !,
  z_reduce_end(E,E1),
  append("NOW", [0' , Sep | E1], P).
make_period(S, E, Sep, P)        :- !,
  z_reduce_start(S,S1),
  z_reduce_end(E,E1),
  append(S1, [0' , Sep | E1], P).

%% sql_get_all_columns(+Types,+Cursor,+ArgCount,-Tuple)
sql_get_all_columns([per|T],C,AC,[P|Vs]) :- !,
  ora_get_column(C,AC,S),
  AC1 is AC+1,
  ora_get_column(C,AC1,E),
  clause(nls_period_sep(Sep,_),true),
  make_period(S, E, Sep, P),
  AC2 is AC1+1,
  sql_get_all_columns(T,C,AC2,Vs).
sql_get_all_columns([_|T],C,AC,[V|Vs]) :- !,
  ora_get_column(C,AC,V), !,
  AC1 is AC+1,
  sql_get_all_columns(T,C,AC1,Vs).
sql_get_all_columns([],_,_,[]).
sql_get_all_columns(nil,C,AC,[V|Vs]) :-
  ora_get_column(C,AC,V), !,
  AC1 is AC+1,
  sql_get_all_columns(nil,C,AC1,Vs).
sql_get_all_columns(nil,_,_,[]).

%% sql_redo_fetch_row(+Cursor)
sql_redo_fetch_row(C) :- 
  ora_fetch_row(C), 
  (true; sql_redo_fetch_row(C)).

%% get_result(+StmtType, +Cursor, +ColumnTypes, -NumTuples, -Result)
get_result(query,C,Types,_,header-Header) :-
  sql_describe_query(Types,C,0,Header).
get_result(query,C,Types,_,Tpl) :-
  sql_redo_fetch_row(C),
  sql_get_all_columns(Types,C,0,Tpl).
get_result(dml,_,_,NumTpls,NumTpls).
get_result(other,_,_,_,_).

/******************************************************************************/
/* Mode:	m_sql_exec(+Input,+Stat,-StmtType,-Result)                    */
/* Purpose:	execute the sql statement 'Stat'		              */
/* Example:                                                                   */
/* Sideeffects:	DB is modified, cursor/1 is added                             */
/* Call:	exits always						      */
/* Redo:	fails always						      */
/******************************************************************************/

%% sql_put_all_variables(+Input,+Cursor,+VarNumber)
sql_put_all_variables([],_,_).
sql_put_all_variables([var(V,T)|Input],C,IC) :-
  (  clause(verbose,true)
  -> (  T=s(_)
     -> format(" >:~d = '~s'~n",[IC,V])
     ;  format(" >:~d = ~w~n",[IC,V])
     )
  ;  true
  ),
  ora_put_var(C,IC,V),
  IC1 is IC+1,
  sql_put_all_variables(Input,C,IC1).

%% m_sql_exec(+Input, +SQLStmt, +ColumnTypes, -StmtType, -Result)
m_sql_exec(I, SQLStmt, Types, StmtType, Res) :-
  u_stat(SQLStmt, SQLStr, []),
  (clause(verbose,true) -> format("~s;~n",[SQLStr]); write('.'), ttyflush),
  ora_open_cursor(C),
  assert(cursor(C)),
  ora_compile_statement(C, SQLStr, StmtType),
  sql_put_all_variables(I, C, 1),
  ora_exec_cursor(C, NumTpls),
  !,
  get_result(StmtType, C, Types, NumTpls, Res).

