/*!****************************************************************************/
/*! File:      scanner.pl                                                     */
/*! Project:   Tiger                                                          */
/*! Software:  SWI Prolog 2.7.16, Oracle 8.0.4                                */
/*! Export:                                                                   */
/*!            s_readstring/1 (interpret.pl)                                  */
/*!            s_scan/2 (interpret.pl,views.pl,constraint.pl,zeit.pl)         */
/*!                                                                           */
/*! Import:                                                                   */
/*!            check_option/1 (tiger.pl)                                      */
/*!            f_raise_err/2 (fejl.pl)                                        */
/*!****************************************************************************/

/******************************************************************************/
/* Mode:           s_readstring(-String)                                      */
/* Purpose:        Reads a command from standard IO                           */
/* Example:                                                                   */
/* Sideeffects:    None                                                       */
/* Call:           Exists always                                              */
/* Redo:           Fails always                                               */
/******************************************************************************/

%% s_readstring(-String)
%% Reads and returns a semicolon terminated command from standard IO
%% A trailing double semicolon ;; can be used to bypass Tiger
%% Rrailing semicolons are not part of the returned string
s_readstring(Str) :-
  prompt(_, 'ATSQL> '),
  ( check_option('-p') -> Pr='' ; Pr='       ' ),
  s_readlines(S, Pr, Flag),
  (  Flag==backend
  -> Str=backend(S)
  ;  Str=S
  ).

%% s_readlines(-String, +Prompt, -BackendFlag)
s_readlines(L,P,F) :-
  get0(X),
  (  X==10 /* start over on carriage return */
  -> ( check_option('-p') -> prompt(_,P) ; true ),
     s_readlines(L,P,F)
  ;  X==(-1) /* stop reading on end of file */
  -> L="eof", F=timedb
  ;  /* else (the "normal" case) */
     get0(Y),
     (  X==0';, Y==10
     -> s_readlines(L,P,F)
     ;  prompt(_,P),
        s_readln(X,Y,L,F)
     )
  ).

%% s_readln(+FirstCharRead, +SecondCharRead, -String, -BackendFlag)
s_readln(X, Y, L, F) :- 
  (Y==0'; -> get_non_blank(Z); get0(Z)),
  (  (Z==(-1); Y==0';, Z==10)
  -> (X==0'; -> F=backend, L=[]; F=timedb, L=[X])
  ;  L=[X|L1], s_readln(Y,Z,L1,F)
  ). 

get_non_blank(X) :-
  get0(Y),
  (Y==0'  -> get_non_blank(X); X = Y).

/******************************************************************************/
/* Mode:           keyword(?Word)                                             */
/* Purpose:                                                                   */
/* Example:                                                                   */
/* Sideeffects:    None                                                       */
/* Call:           Exists always                                              */
/* Redo:           Fails always                                               */
/******************************************************************************/

keyword("ABS",abs).		        keyword("INTO",into).                
keyword("ADD",add).                     keyword("LIKE",like).         
keyword("ALL",all).		        keyword("MIN",min).                  
keyword("ALTER",alter).                 keyword("NOT",not).                  
keyword("AND",and).                     keyword("OR",or).                      
keyword("AS",as).		        keyword("PRIMARY",primary).          
keyword("ASC",asc).                     keyword("ROLLBACK",rollback).         
keyword("ASSERTION",assertion).	        keyword("SET",set).                  
keyword("AVG",avg).		        keyword("TABLE",table).              
keyword("BETWEEN",between).             keyword("UPDATE",update).             
keyword("BY",by).		        keyword("VARCHAR",varchar).          
keyword("CHAR",char).                   keyword("WHERE",where).               
keyword("CHECK",check).		        keyword("HAVING",having).            
keyword("COMMIT",commit).               keyword("INSERT",insert).              
keyword("COUNT",count).                 keyword("INTERSECT",isct).           
keyword("CREATE",create).	        keyword("KEY",key).                  
keyword("DELETE",delete).	        keyword("MAX",max).                  
keyword("DESC",desc).                   keyword("MODIFY",modify).              
keyword("DISTINCT",distinct).	        keyword("NULL",null).                
keyword("DROP",drop).                   keyword("ORDER",order).               
keyword("EXCEPT",except).	        keyword("REFERENCES",references).    
keyword("EXISTS",exists).               keyword("SELECT",select).             
keyword("FLOAT",float).                 keyword("SUM",sum).                   
keyword("FROM",from).		        keyword("UNION",union).              
keyword("GROUP",group).		        keyword("VALUES",values).            
keyword("IN",in).	                keyword("VIEW",view).                
keyword("INTEGER",integer).	        keyword("DEFAULT",default).          
keyword("IS",is).                       keyword("CONSTRAINT",constraint).
keyword("NVL",nvl).                     keyword("SEQUENCE",sequence).
keyword("SESSION",session).

keyword("BEGIN",begin).                 keyword("NSEQ",nonsequenced).          
keyword("BEGINNING",beginning). 	keyword("OVERLAPS",ove).             
keyword("CONTAINS",con).        	keyword("PERIOD",period).            
keyword("CURRENT",current).		keyword("PRECEDES",pre).              
keyword("DATE",date).           	keyword("SECOND",second).             
keyword("DAY",day).			keyword("SEQ",sequenced).            
keyword("DURATION",duration).		keyword("SEQUENCED",sequenced).       
keyword("END",end).			keyword("SYSDATE",sysdate).           
keyword("FIRST",first).         	keyword("TIMESTAMP",timestamp).      
keyword("FOREVER",forever).		keyword("TO",to).                     
keyword("HOUR",hour).           	keyword("TRANSACTION",transaction).   
keyword("INTERVAL",interval).   	keyword("TT",transaction).            
keyword("LAST",last).           	keyword("TTIME",ttime).              
keyword("MEETS",mee).			keyword("VALID",valid).             
keyword("MINUTE",minute).		keyword("VT",valid).                 
keyword("MONTH",month).         	keyword("VTIME",vtime).               
keyword("NONSEQUENCED",nonsequenced).	keyword("YEAR",year).                
keyword("NOW",now).             

keyword("VERBOSE",verbose).	        keyword("QUIT",quit).
keyword("BATCH",batch).                 keyword("OPEN",open).
keyword("CLOSE",close).                 keyword("ON",on).
keyword("OFF",off).                     keyword("CHANGE",change).

/******************************************************************************/

s_char(C1)     --> [C], {C >= 0'a, C =< 0'z -> C1 is C - 0'a + 0'A}.  
s_char(C)      --> [C], {C >= 0'A, C =< 0'Z}. 

s_digit(D)     --> [D], {D >= 0'0, D =< 0'9}.

id_char(C)   --> s_char(C), !.
id_char(D)   --> s_digit(D), !.
id_char(C)   --> [C], {memberchk(C,[0'#,0'$,0'_])}, !.

s_id([C|Cs]) --> id_char(C), s_id(Cs), !.
s_id([])     --> [].

s_int([D|Ds])  --> s_digit(D), !, s_int(Ds).
s_int([])      --> [].

s_exp2([0'+|Ds]) --> "+", !, s_int(Ds).
s_exp2([0'-|Ds]) --> "-", !, s_int(Ds).
s_exp2(Ds)       --> s_int(Ds).

s_exp([0'e|Sign]) --> s_char(0'E), !, s_exp2(Sign).
s_exp([])         --> [].

s_num([D|Ds])  --> s_digit(D), !, s_num(Ds).
s_num([0'.|N]) --> ".", s_int(M), s_exp(E), {append(M, E, N)}.
s_num([])      --> [].

s_str([0''|R]) --> "'", "'", !, s_str(R).
s_str([])	  --> "'", !.
s_str([C|Cs])  --> [C], !, s_str(Cs).
s_str([])      --> {f_raise_err(missing_eos,[])}.

/******************************************************************************/
/* Mode:           token(-Token, +String, -RestString)                        */
/* Purpose:        Consumes next token from string.                           */
/* Example:                                                                   */
/* Sideeffects:    None                                                       */
/* Call:           Exits always                                               */
/* Redo:           Fails always                                               */
/******************************************************************************/

skip_comment	--> "*/", !.
skip_comment	--> [_], skip_comment.

token(T)	--> " ", !, token(T).
token(Token)	--> "/*", skip_comment, token(Token). 
token(gq)	--> ">=", !.
token(lq)	--> "<=", !.
token(nq)	--> "<>", !.
token(nq)       --> "!=", !.
token(gr)	--> ">", !.
token(ls)	--> "<", !.
token(eq)	--> "=", !.
token('+')	--> "+", !.
token('*')	--> "*", !.
token('-')	--> "-", !.
token('/')	--> "/", !.
token('(')	--> "(", !.
token(')')	--> ")", !.
token('.')	--> ".", !.
token(',')	--> ",", !.
token(':')	--> ":", !.
token('%')	--> "%", !.
token('|')	--> "|", !.
token('[')	--> "[", !.
token('~')	--> "~", !.
token('@')	--> "@", !.
token(';')	--> ";", !.
token(str(Str))	--> "'", !, s_str(Str).
token(T)        --> s_char(C), !, s_id(Cs),
                    {keyword([C|Cs],KW) -> T=KW; T=id([C|Cs])}.
token(num(Num))	--> s_digit(N), !, s_num(Rest), {number_chars(Num, [N|Rest])}.
token(T)	--> [_], token(T).
token([])	--> [].

/******************************************************************************/
/* Mode:           s_scan(+String, -TokenList)                                */
/* Purpose:        Tokenizes a string                                         */
/* Sideeffects:    None                                                       */
/* Call:           Exits always                                               */
/* Redo:           Fails always                                               */
/******************************************************************************/

%% s_scan(+String, -TokenList)
s_scan(backend(S), backend(S)) :- !.
s_scan([], []) :- !.
s_scan(String, [Token|TokenList]) :-
  token(Token, String, RestString), !, 
  s_scan(RestString, TokenList).
s_scan(_, nil).

