/*!****************************************************************************/
/*! File:      unparser.pl                                                    */
/*! Project:   Tiger                                                          */
/*! Software:  SWI Prolog 2.7.16, Oracle 8.0.4                                */
/*! Export:                                                                   */
/*!            u_stat/3 (interpret.pl,meta.pl,constraint.pl)                  */
/*!            u_query_expr/3 (trans.pl)                                      */
/*!            u_fac/4 (fejl.pl)                                              */
/*! Import:                                                                   */
/*!            bf/3 (tiger.pl)                                                */
/*!            nls_period_seperator/2 (tiger.pl)                              */
/*!            r_type/2 (rewrite.pl)                                          */
/*!            z_get_current/1 (zeit.pl)                                      */
/*!            z_u_per/8 (zeit.pl)                                            */
/*!            z_u_ts/6 (zeit.pl)                                             */
/*!            z_unfold_per/3 (zeit.pl)                                       */
/*!****************************************************************************/

u_tab_type(sn) --> !, "".
u_tab_type(vt) --> !, " AS VALID".
u_tab_type(tt) --> !, " AS TRANSACTION".
u_tab_type(bi) --> !, " AS VALID AND TRANSACTION".

u_flag(nil) --> !, "".
u_flag(flag(vt(VTM,VTD),tt(TTM,TTD),VTR)) -->
  u_vt_mode(VTM),
  u_domain(VTD),
  u_tt_mode(TTM),
  u_domain(TTD),
  u_range(VTR).

u_vt_mode(seq) --> !, "SEQUENCED VALID ".
u_vt_mode(ns)  --> !, "NONSEQUENCED VALID ".
u_vt_mode(tuc) --> !, "".

u_tt_mode(seq) --> !, "AND SEQUENCED TRANSACTION ".
u_tt_mode(ns)  --> !, "AND NONSEQUENCED TRANSACTION ".
u_tt_mode(tuc) --> !, "".

u_domain(nil)   --> !, "".
u_domain(Per)   --> !,
  {nls_period_sep(Sep,_)},
  z_u_per(Per, "PERIOD '", [Sep], "' ", no_auto_gran, atsql).

u_range(nil)   --> !, "".
u_range(E)     --> !, "SET VALID ", u_fac(E,ins), " ".

u_coal(nil)    --> !, "".
u_coal(F+S)    --> !, u_dim(F), u_dim(S).

u_dim(vt)      --> !, "(VALID)".
u_dim(tt)      --> !, "(TRANSACTION)".
u_dim(nil)     --> !, "".

/******************************************************************************/
/* Mode:            u_stat(+SQL_stat,-Chars,+Chars)                           */
/* Purpose:         generates Oracle-SQL statement                            */
/* Sideeffects:     none                                                      */
/* Call:            exits if stmt is a legal Oracle stmt                      */
/* Redo:            fails always                                              */
/******************************************************************************/

u_stat(query(F,QE,C,O)) --> u_flag(F), u_query_expr(QE), u_coal(C), u_orderby(O).

u_stat(create_table(R,colDefs(ColNames,QueryExpr))) --> !,
  "CREATE TABLE",
  u_id(R, " "),
  u_col_names(ColNames),
  " AS ",
  !,
  u_query_expr(QueryExpr).

u_stat(create_table(R,ColDefs,TabType)) -->
  "CREATE TABLE ",
  u_id(R, ""),
  "(", u_col_defs(ColDefs, ""), !, ")",
  u_tab_type(TabType).

u_stat(create_view(ViewName,ColNames,Flag,QueryExpr,Coal)) --> !,
  "CREATE OR REPLACE VIEW ",
  u_id(ViewName, ""),
  u_col_names(ColNames),
  " AS ",
  u_flag(Flag),
  u_query_expr(QueryExpr),
  u_coal(Coal).

u_stat(drop_table(R))   --> !, "DROP TABLE ", u_id(R, "").
u_stat(drop_view(V))    --> !, "DROP VIEW ", u_id(V, "").

u_stat(dml(Flag,DMLStmt)) --> u_flag(Flag), u_dml_stmt(DMLStmt).

u_stat(savepoint(X)) --> !, "SAVEPOINT ", u_id(X, "").
u_stat(rollback(X))  --> !, "ROLLBACK TO ", u_id(X, "").
u_stat(iterate(U))   --> !, "ITERATE ", u_stat(U).
u_stat(alter(T,A))   --> "ALTER TABLE", u_id(T, " "), u_alter_cmd(A).
u_stat(rename(A,B))  --> "RENAME", u_id(A, " "), u_id(B, " TO ").
u_stat(lit(S))       --> !, S.

u_col_names2([])    --> !, "".
u_col_names2([N|R]) --> 
  (  {last(0'$,N)}
  -> u_id(N,","), u_id(N,"$S,"), "$E"
  ;  u_id(N,",")
  ),
  u_col_names2(R).

u_col_names(nil)      --> !, "".
u_col_names([CN|CNs]) --> 
  " (", 
  (  {last(0'$,CN)}
  -> u_id(CN,""), u_id(CN,"$S,"), "$E"
  ;  u_id(CN,"")
  ),
  u_col_names2(CNs),
  ") ".

u_alter_cmd(add(ColDefs))    --> " ADD (", !, u_col_defs(ColDefs, ""), ")".
u_alter_cmd(enable(CN))      --> " ENABLE CONSTRAINT ", u_id(CN, "").
u_alter_cmd(disable(CN))     --> " DISABLE CONSTRAINT ", u_id(CN, "").

u_query_expr(lit(S))      --> !, S.
u_query_expr(values(S))   --> " VALUES (", u_exprlist(S,ins), ")".
u_query_expr(union(A,B))  --> !, u_query_expr(A), " UNION ", u_query_expr(B). 
u_query_expr(except(A,B)) --> !, u_query_expr(A), " MINUS ", u_query_expr(B).
u_query_expr(isct(A,B))   --> !, u_query_expr(A), " INTERSECT ", u_query_expr(B).
u_query_expr(sel(M,S,F,W,G,H)) -->
  "SELECT ", u_mode(M), u_exprlist(S,ins),
  " FROM ", u_from(F),
  u_where(W),
  u_groupby(G),
  u_having(H).

u_dml_stmt(insert(TabName,ColNames,QueryExpr)) -->
  "INSERT INTO ",
  u_id(TabName, ""),
  u_col_names(ColNames),
  u_stat(query(nil,QueryExpr,nil,[])).

u_dml_stmt(delete(P,A,W)) --> !,
  "DELETE FROM",
  u_id(P, " "),
  u_id(A, " "),
  u_where(W).

u_dml_stmt(update(P,A,U,W)) --> !,
  "UPDATE ",
  u_id(P, ""),
  u_id(A, " "),
  " SET ",
  u_set(U),
  u_where(W).

u_set(set([E1],[E2])) --> !, u_fac(E1,ins), "=", u_fac(E2,ins), !.
u_set(set(E1,[H|T])) --> !,
  "(",
  u_exprlist(E1,ins),
  ")=(",
  u_query_expr(sel(all,[H|T],[],true,[],true)),
  ")".
u_set(set(E,SQ))  --> "(", u_exprlist(E,ins), ")=(", u_query_expr(SQ), ")", !.

u_col_defs([D|R], Sep) --> Sep, u_col_def(D), u_col_defs(R, ", ").
u_col_defs([], _)      --> "".

u_col_def(colDef(C,per,period(S,E),_)) --> !,
  u_id(C, ""),   "$$S DATE DEFAULT ", u_fac(S,ins),
  u_id(C, ", "), "$$E DATE DEFAULT ", u_fac(E,ins).
u_col_def(colDef(C,per,nil,_)) --> !,
  u_id(C, ""),   "$$S DATE",
  u_id(C, ", "), "$$E DATE".
u_col_def(colDef(C,T,D,Props)) -->
  u_id(C, ""),
  " ",
  u_type(T),
  u_default(D),
  u_props(Props).

u_props([]) --> !, "".
u_props([ic(Name,Flag,Prop)|Props]) -->
  u_constraint_name(Name),
  u_flag(Flag),
  u_prop(Prop," "),
  u_props(Props).

u_default(nil) --> !, "".
u_default(D)   --> " DEFAULT", u_fac(D,ins).

u_constraint_name(seq(S,O))  --> !, u_fac(seq(S,O),_).
u_constraint_name(Name)      --> " CONSTRAINT", Name.

u_prop(nil, _)           --> "".
u_prop(notnull, Sep)     --> Sep, "NOT NULL".
u_prop(primary_key, Sep) --> Sep, "PRIMARY KEY".

u_type(s(X))            --> {number_chars(X,Str)}, "CHAR(", Str, ")".
u_type(n(T))            --> {var(T)}, !, "NUMBER".
u_type(n(f))            --> "FLOAT".
u_type(n(i))            --> "INTEGER".
u_type(ts)              --> "DATE". /* SQL-92: TIMESTAMP */

u_from([])              --> "DUAL".
u_from([X])             --> !, u_tab_ref(X).
u_from([H|T])           --> u_tab_ref(H), ", ", u_from(T).

u_where(C)              --> " WHERE ", u_cond(C), !.
u_where(_)              --> "".

u_groupby(G)            --> " GROUP BY ", u_exprlist(G,ins), !.
u_groupby(_)            --> "".

u_having(H)             --> " HAVING ", u_cond(H), !.
u_having(_)             --> "".

u_orderby(O)            --> " ORDER BY ", u_exprlist(O,ins), !.
u_orderby(_)            --> "".

u_condlist([X],_)	--> u_cond(X), !.
u_condlist([H|T],Op)	--> u_cond(H), Op, u_condlist(T,Op), !.
u_condlist([H|_],_)     --> u_cond(H), !.
u_condlist([_|T],Op)    --> u_condlist(T,Op).

u_cond(not(like(X,Y)))	       --> !, u_fac(X,evl), " NOT LIKE ", u_fac(Y,evl).
u_cond(not(between(X,Y,Z)))    --> !, u_fac(X,evl), " NOT BETWEEN ", 
                                   u_fac(Y,evl), " AND ", u_fac(Z,evl).
u_cond(not(isnull(E)))         --> !, u_fac(E,ins), " IS NOT NULL".
u_cond(not(ex(S)))	       --> !, "NOT EXISTS (", u_query_expr(S), ")".
u_cond(not(in(X,Y)))           --> !, "(", u_exprlist(X,evl), ") NOT IN (", 
                                   u_exprlist(Y,evl), ")".
u_cond(rel(mee,A,B))           --> !, u_cond(rel(ls,end(A),begin(B))).
u_cond(rel(pre,A,B))           --> !, u_cond(rel(ls,end(A),begin(B))).
u_cond(rel(ove,A,B))           --> !, u_cond(and(rel(gr,end(A),begin(B)),
                                                 rel(gr,end(B),begin(A)))).
u_cond(rel(con,ttime(T),now))  --> !, T, ".TT$$E IS NULL".
u_cond(rel(con,A,B))           --> !, u_cond(and(rel(lq,begin(A),begin(B)),
                                                 rel(gq,end(A),end(B)))).
u_cond(rel(eq,col_ref(nil,C,_),now)) --> !, C, " IS NULL".
u_cond(rel(R,X,Y))             --> !, u_fac(X,evl), u_rel(R), u_fac(Y,evl).
u_cond(ove(P))                 --> !, {z_unfold_per(isct(P),SList,EList)},
                                   u_fac(SList,evl), "<", u_fac(EList,evl).
u_cond(and(A,B))               --> !, u_condlist([A,B], " AND ").
u_cond(or(A,B))                --> !, "(", u_condlist([A,B], " OR "), ")".
u_cond(ex(S))   	       --> !, "EXISTS (", u_query_expr(S), ")".
u_cond(like(X,Y))	       --> !, u_fac(X,evl), " LIKE ", u_fac(Y,evl).
u_cond(between(X,Y,Z))	       --> !, u_fac(X,evl), " BETWEEN ", 
                                   u_fac(Y,evl), " AND ", u_fac(Z,evl).
u_cond(not(CL))		       --> !, "NOT ", u_condlist(CL," OR ").
u_cond(in(X,Y))	               --> !, "(", u_exprlist(X,evl), ") IN (",
                                   u_exprlist(Y,evl), ")".
u_cond(in(X,Y))	               --> !, "(", u_fac(X,evl), ") IN (",
                                   u_exprlist(Y,evl), ")".
u_cond(isnull(E))	       --> !, u_fac(E,evl), " IS NULL".

u_rel(eq)		--> "=".
u_rel(nq)		--> "<>".
u_rel(ls)		--> "<".
u_rel(lq)		--> "<=".
u_rel(gr)		--> ">".
u_rel(gq)		--> ">=".

u_tab_def(derived_tab(Flag,QE)) --> !, "(", u_stat(query(Flag,QE,nil,[])), ")".
u_tab_def(T)                    --> !, u_id(T, "").

u_tab_ref(tab_ref(TabDef,A,C))  --> u_tab_def(TabDef), u_id(A, " "), u_coal(C).

u_exprlist([E],M)         --> !, u_fac(E,M).
u_exprlist([E|L],M)       --> !, u_fac(E,M), ",", u_exprlist(L,M).
u_exprlist(E,_)           --> u_query_expr(E).

u_op(add) --> !, "+".
u_op(sub) --> !, "-".
u_op(mul) --> !, "*".
u_op(div) --> !, "/".
u_op(cat) --> !, "||".

%% u_per_alias(+ColumnAlias,+Extension)
u_per_alias(nil,_) --> !.
u_per_alias(C,E)   --> !, u_id(C," "), E.

%% u_fac(+Factor, +Mode)
%% Mode is either 'ins' or 'evl'.  If it is 'evl' then NOW is instantiated 
%% to the current time.

u_fac(bin(Op,X,Y),M)     --> !, u_fac(X,M), u_op(Op), u_fac(Y,M).

u_fac(col_ref(nil,"VT$$E",_),evl) --> !,
  "NVL(VT$$E,", {z_get_current(Ts)}, u_fac(Ts,evl), ")".
u_fac(col_ref(nil,"TT$$E",_),evl) --> !,
  "NVL(TT$$E,", {z_get_current(Ts)}, u_fac(Ts,evl), ")".
u_fac(col_ref(nil,C,_),_)    --> !, u_id(C,"").

u_fac(col_ref(A,"VT$$E",_),evl) --> !,
  "NVL(", u_id(A,""), ".VT$$E,", {z_get_current(Ts)}, u_fac(Ts,evl), ")".
u_fac(col_ref(A,"TT$$E",_),evl) --> !,
  "NVL(", u_id(A,""), ".TT$$E,", {z_get_current(Ts)}, u_fac(Ts,evl), ")".
u_fac(col_ref(A,C,_),_)    --> !, u_id(A,""), u_id(C,".").

u_fac(sel_col(all,_),_)    --> !, "*".
u_fac(sel_col(all(A),_),_) --> !, u_id(A,""), ".*".
u_fac(sel_col(cst(S-E,per),C),ins) --> !,
  z_u_ts(S,auto_gran,start,sql-ins), u_per_alias(C,"$S"), ",",
  z_u_ts(E,auto_gran,end,sql-ins), u_per_alias(C,"$E").
u_fac(sel_col(P,C),ins)    -->
  {r_type(P,per)}, !,
  {z_unfold_per(P,S,E)}, 
  u_fac(S,ins), u_per_alias(C,"$S"), ",",
  u_fac(E,ins), u_per_alias(C,"$E").
u_fac(sel_col(E,C),_)      --> !, u_fac(E,_), u_id(C," ").

u_fac(add_grl(TS,sec,N),M) --> !,
  "TO_DATE(", u_fac(TS,M), ")+1/86400*(", u_fac(cst(N,n(i)),M), ")". 
u_fac(add_grl(TS,min,N),_) --> !,
  "TO_DATE(", u_fac(TS,M), ")+1/1440*(", u_fac(cst(N,n(i)),M), ")". 
u_fac(add_grl(TS,hou,N),_) --> !,
  "TO_DATE(", u_fac(TS,M), ")+1/24*(", u_fac(cst(N,n(i)),M), ")". 
u_fac(add_grl(TS,day,N),_) --> !,
  "TO_DATE(", u_fac(TS,M), ")+ ", u_fac(cst(N,n(i)),M). 
u_fac(add_grl(TS,mon,N),_) --> !,
  "ADD_MONTHS(", u_fac(TS,M), ",",  u_fac(cst(N,n(i)),M), ")". 
u_fac(add_grl(TS,yea,N),_) --> !,
  "ADD_MONTHS(", u_fac(TS,M), ",12*(",  u_fac(cst(N,n(i)),M), "))". 

u_fac(cst(C,n(_)),_)       --> !, {number_chars(C,Str)}, Str.
u_fac(cst(S-E,per),M)      --> !,
  z_u_per(cst(S-E,per), "", ",", "", auto_gran, sql-M).
u_fac(cst(P,ts),M)         --> !, z_u_ts(cst(P,ts), auto_gran, start, sql-M).
u_fac(cst(C,_),_)          --> !, "'", C, "'".

u_fac(period(T1,T2),M)    --> !, u_fac(T1,M), ",", u_fac(T2,M). 
u_fac(vtime(R),ins)       --> !, R, ".VT$$S,", R, ".VT$$E".
u_fac(vtime(R),ins)       --> !, R, ".VT$$S,", R, ".VT$$E".
u_fac(ttime(R),_)         --> !, R, ".TT$$S,", R, ".TT$$E".
u_fac(null,_)             --> !, " NULL ".
u_fac(nvl(E1,E2),M)       --> !, "NVL(", u_fac(E1,M), ",", u_fac(E2,M), ")".
u_fac(now,ins)            --> !, "NULL".
u_fac(now,evl)            --> !, {z_get_current(Ts)}, u_fac(Ts,ins).
u_fac(sysdate,_)          --> !, "SYSDATE".
u_fac(begin(P),M)         --> !, {z_unfold_per(P,S,_)}, u_fac(S,M).
u_fac(end(P),M)           --> !, {z_unfold_per(P,_,E)}, u_fac(E,M).
		       
u_fac(seq(S,curr),_)      --> !, u_id(S,""), ".CURRVAL".
u_fac(seq(S,next),_)      --> !, u_id(S,""), ".NEXTVAL".
u_fac(rowid(T),_)         --> !, u_id(T,""), ".ROWID".
u_fac(dist(X),_)          --> !, "DISTINCT", u_fac(X,_).
u_fac(agg(count,M,all),_) --> !, "COUNT(", u_mode(M), "*)".
u_fac(agg(A,M,C),_)       --> !, u_agg(A), "(", u_mode(M), u_fac(C,evl), ")".
u_fac(abs(E),M)           --> !, "ABS(", u_fac(E,M), ")".
u_fac(mod(X,Y),M)         --> !, "MOD(", u_fac(X,M), ",", u_fac(Y,M), ")".
u_fac(trunc(X),M)         --> !, "TRUNC(", u_fac(X,M), ")".
u_fac(asc(X),M)           --> !, u_fac(X,M).
u_fac(desc(X),M)          --> !, u_fac(X,M), " DESC ".
u_fac(input(N),_)         --> !, ":", {number_chars(N,Str)}, Str.
u_fac(gst([X]),M)         --> !, u_fac(X,M).
u_fac(gst([H|T]),_)       --> !, "GREATEST(", u_exprlist([H|T],evl), ")".
u_fac(lst([X]),M)         --> !, u_fac(X,M).
u_fac(lst([H|T]),_)       --> !, "LEAST(", u_exprlist([H|T],evl), ")".
u_fac(un_op(Op,F),M)      --> !, u_op(Op), u_fac(F,M).
u_fac(E,M)                -->
  {functor(E,X,_), memberchk(X, [add,sub,mul,quo])}, "(", u_fac(E,M), ")".
u_fac(Id,_)               --> u_id(Id, "").

u_id(nil, _)          --> !, "".
u_id(X+Y, Sep)        --> !, Sep, u_id(X, ""), u_id(Y, "").
u_id(id(X),_)         --> !, X.
u_id([H|T], Sep)      --> !, Sep, bf([H|T]).
u_id(C, _)            --> {atom(C), !, atom_chars(C,L)}, L.
u_id(N, _)            --> {number(N), number_chars(N,Str)}, !, Str.

u_mode(distinct)      --> !, " DISTINCT ".
u_mode(_)             --> "".

u_agg(max)   --> "MAX".
u_agg(min)   --> "MIN".
u_agg(avg)   --> "AVG".
u_agg(sum)   --> "SUM".
u_agg(count) --> "COUNT".

