/******************************************************************************/
/* Module:      coalescing.pl                                                 */
/* Project:     TimeDB 1.04                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SICStus Prolog                                                */
/* Machine:     SPARC/Solaris, PowerMac/MacOS                                 */
/* Export:                                                                    */
/*              get_conditions/5 (evaluate.pl)                                */
/*              add_alias/3 (evaluate.pl)                                     */
/*              do_temporal_reduction/4 (evaluate.pl)                         */
/* Import:                                                                    */
/*              drop_aux_tables/1 (basics.pl)                                 */
/*              generate_table_name/2 (basics.pl)                             */
/*              sql_exec/4 (interface.pl)                                     */
/******************************************************************************/

%% create_eq_list(+Ref1, +Ref2, +ColumnList, -Conditions)
create_eq_list(_, _, [], []) :- !.
create_eq_list(A1, A2, [Attr|Attr_List], 
               [rel(eq, A1-Attr, A2-Attr)|Eq_Attr_List]) :-
  create_eq_list(A1, A2, Attr_List, Eq_Attr_List).


get_conditions1([], []) :- !.
get_conditions1([Attr|Columns], [rel(eq, 'a0'-Attr, 'a1'-Attr)|Conds]) :-
	get_conditions1(Columns, Conds).

%% get_conditions(+Time, +Columns, -Conditions, -DirStart, -DirEnd)
get_conditions(valid, ['vts_#$','vte_#$'|Cols], Conds, ['vts_#$', 'vte_#$'], Cols) :-
	!,
	get_conditions1(Cols, Conds).
get_conditions(system, ['sts_#$','ste_#$'|Cols], Conds, ['sts_#$', 'ste_#$'], Cols) :-
	!,
	get_conditions1(Cols, Conds).
get_conditions(validsystem, ['vts_#$','vte_#$','sts_#$','ste_#$'|Cols], 
               Conds, ['vts_#$', 'vte_#$', 'sts_#$', 'ste_#$'], Cols) :-
	!,
	get_conditions1(Cols, Conds).
get_conditions(systemvalid, ['sts_#$','ste_#$', 'vts_#$','vte_#$'|Cols], 
               Conds, ['sts_#$', 'ste_#$', 'vts_#$', 'vte_#$'], Cols) :-
	!,
	get_conditions1(Cols, Conds).


%% add_alias(+Alias, +Attr_List, -Alias_Attr_List)
add_alias(_, [], []) :- !.
add_alias(Alias, [Attr|Attr_List], [Alias-Attr|Alias_Attr_List]) :-
  add_alias(Alias, Attr_List, Alias_Attr_List).


/********************************************************************************/
/* Mode:          unitemporal_reduction(+Table-ResultType, +TempType)           */
/* Purpose:       Temporal reduction in either valid or system time direction.  */
/* Example:                                                                     */
/* Sideeffects:   None                                                          */
/* Call:          Exits always                                                  */
/* Redo:          Fails always                                                  */
/********************************************************************************/

%% temp_reduction_loop(+TimeStart, +TimeEnd, +Table, +Conditions)
temp_reduction_loop(TS, TE, Table, Conditions) :-
	!,
	sql_exec([], update(Table-a0,
			    set([a0-TE],
				[sel([max(abs(a1-TE))]),
				 from([Table-a1]),
				 where([and([rel(gq, abs(a0-TE), abs(a1-TS)),
					     rel(ls, abs(a0-TE), abs(a1-TE))|
					     Conditions])])]),
			    where([and([ex(
				  [sel([all]),
				   from([Table-a1]),
				   where([and([rel(gq, abs(a0-TE), abs(a1-TS)),
					       rel(ls, abs(a0-TE), abs(a1-TE))|
					       Conditions])])])])])),
		 NumRowsProc, _),
	(NumRowsProc == 0, !; 
         temp_reduction_loop(TS, TE, Table, Conditions)).


unitemporal_reduction(Table-Cols, TempType) :-
	get_conditions(TempType, Cols, Conditions, [TS, TE], _),
	temp_reduction_loop(TS, TE, Table, Conditions),
	!,
	sql_exec([], delete(Table-a0, where([and([ex(
			    [sel([a1-all]),
			     from([Table-a1]),
			     where([and([rel(ls, abs(a1-TS), abs(a0-TS)),
					 rel(eq, abs(a1-TE), abs(a0-TE))|
					 Conditions])])])])])), _, _).


/********************************************************************************/
/* Mode:           expand_rel(+Table, +Cols, +Time,                             */
/*                            -Result, -ResCols, -TempAttrs)                    */
/* Purpose:        Creates SQL-relation out of relation Table expanded by       */
/*                 attribute SST                                                */
/* Example:                                                                     */
/* Sideeffects:    None                                                         */
/* Call:           Exists always                                                */
/* Redo:           Fails always                                                 */
/********************************************************************************/

expand_rel(Table, Cols, Time, Result, Cols1, [MDS, MDE, SDS, SDE]) :-
	get_conditions(Time, Cols, Conditions, [MDS, MDE, SDS, SDE], Cols1),
	generate_table_name("exp", Result),	
	add_alias("a1", [MDS, MDE, SDS, SDE|Cols1], Alias_Cols),
	sql_exec([],
	  create(Result,
	         ["sst", MDS, MDE, SDS, SDE|Cols1],
	         [union([
                   [sel([abs(a0-MDS)|Alias_Cols]),
                    from([Table-a0, Table-a1]),
                    where([and([rel(lq, abs(a1-MDS), abs(a0-MDS)), 
                                rel(gr, abs(a1-MDE), abs(a0-MDS))|
                                Conditions])])],
                   [sel([abs(a0-MDE)|Alias_Cols]),
                    from([Table-a0, Table-a1]),
                    where([and([rel(gr, abs(a1-MDE), abs(a0-MDE)), 
	                        rel(ls, abs(a1-MDS), abs(a0-MDE))|
                                Conditions])])]])]
           ), _, _).

/********************************************************************************/
/* Mode:           reduce_sub_dir(+Table, +Columns, +[MDS, MDE, SDS, SDE])      */
/* Purpose:        Temporal reduction in minor time direction.                  */
/* Example:                                                                     */
/* Sideeffects:    None                                                         */
/* Call:           Exists always                                                */
/* Redo:           Fails always                                                 */
/********************************************************************************/

%% update_sub_dir(+Table, +TimeAttrs, -Conditions)
update_sub_dir(Table, [MDS, MDE, SDS, SDE], Conditions) :-
	sql_exec([],  
           update(Table-a0,
                  set([a0-SDE],
                      [sel([max(abs(a1-SDE))]),
                       from([Table-a1]),
                       where([and([rel(gq, abs(a0-SDE), abs(a1-SDS)),
                        	   rel(ls, abs(a0-SDE), abs(a1-SDE)),
                                   rel(eq, abs(a0-sst), abs(a1-sst))|
                                   Conditions])])]),
                  where([and([ex([sel([all]),
                                  from([Table-a1]),
                                  where([and([rel(gq, abs(a0-SDE), abs(a1-SDS)),
                                              rel(ls, abs(a0-SDE), abs(a1-SDE)),
                                              rel(eq, abs(a0-sst), abs(a1-sst))|
                                              Conditions])])])])])),
           N, _),
	N \== 0,
	!,
	update_sub_dir(Table, [MDS, MDE, SDS, SDE], Conditions).
update_sub_dir(_, _, _).

reduce_sub_dir(Table, Columns, [MDS, MDE, SDS, SDE]) :-
	create_eq_list('a0', 'a1', Columns, Conditions),
	update_sub_dir(Table, [MDS, MDE, SDS, SDE], Conditions),
	sql_exec([],  
           delete(Table-a0,
                  where([and([ex([sel([all]),
                       from([Table-a1]),
                       where([and([rel(eq, abs(a0-SDE), abs(a1-SDE)),
                                   rel(ls, abs(a1-SDS), abs(a0-SDS)),
                                   rel(eq, abs(a0-sst), abs(a1-sst))|
                                   Conditions])])])])])),
            _, _). 

/********************************************************************************/
/* Mode:           insert_endtime_stamps(+Table, +Columns, +TimeAttrs)          */
/* Purpose:        Adds end marks on major time direction for coalescing in     */
/*                 major time direction.                                        */
/* Example:                                                                     */
/* Sideeffects:    None                                                         */
/* Call:           Exists always                                                */
/* Redo:           Fails always                                                 */
/********************************************************************************/

insert_endtime_stamps(Table, Cols, [MDS, MDE, SDS, SDE]) :-
	create_eq_list('a0', 'a1', Cols, Conditions),
	add_alias("a0", [MDS, MDE, SDS, SDE|Cols], Alias_Attrs),
	sql_exec([],
	   insert(Table,
		  [sel([abs(a0-MDE)|Alias_Attrs]),
                   from([Table-a0]),           
                   where([and([notex(
                                [sel([all]),
                                 from([Table-a1]),
                                 where([and([rel(gq, abs(a0-MDE), abs(a1-MDS)), 
                                             rel(ls, abs(a0-MDE), abs(a1-MDE)),
                                             or([and([rel(lq, abs(a0-SDE), abs(a1-SDE)), 
                                                      rel(gr, abs(a0-SDE), abs(a1-SDS))
                                                      ]), 
                                                 and([rel(lq, abs(a1-SDE), abs(a0-SDE)),
                                                      rel(gr, abs(a1-SDE), abs(a0-SDS))
                                                     ])
                                                ])
                                             |Conditions])])
                                 ])])])
                   ]
   ), _, _).


/********************************************************************************/
/* Mode:           coalesce_main_dir(+Table, +Cols, +TimeAttrs, -Result, -Cols) */
/* Purpose:        Coalesces rectangles in major time direction.                */
/* Example:                                                                     */
/* Sideeffects:    None                                                         */
/* Call:           Exists always                                                */
/* Redo:           Fails always                                                 */
/********************************************************************************/
  
coalesce_main_dir(Table, Cols, [MDS, MDE, SDS, SDE], Result, Cols) :-
	generate_table_name("exp", Result),
	create_eq_list('a0', 'a1', Cols, Conditions12),
	create_eq_list("a0", "a3", Cols, Conditions13),
	add_alias("a0", Cols, Alias_Attrs),
	sql_exec([],
	create(Result,
	       [MDS, MDE, SDS, SDE|Cols],
	       [seld([abs(a0-sst), abs(a1-sst), abs(a0-SDS), abs(a0-SDE)|Alias_Attrs]),
	        from([Table-a0, Table-a1]),    
	        where([and([rel(ls, abs(a0-sst), abs(a1-sst)),
	                  or([and([rel(lq, abs(a0-MDS), abs(a1-MDS)),
	                           rel(gq, abs(a0-MDE), abs(a1-MDS))]),
	                      and([rel(gq, abs(a0-MDS), abs(a1-MDS)),
	                           rel(ls, abs(a0-MDE), abs(a1-MDE))]),
	                      and([rel(gr, abs(a0-MDS), abs(a1-MDS)),
	                           rel(lq, abs(a0-MDE), abs(a1-MDE))])]), 
	                  or([and([rel(lq, abs(a1-SDS), abs(a0-SDE)),
	                           rel(gq, abs(a1-SDE), abs(a0-SDE))
	                         ]), 
	                      and([rel(lq, abs(a0-SDS), abs(a1-SDE)),
	                           rel(gq, abs(a0-SDE), abs(a1-SDE))
	                          ]), 
	                      and([rel(gq, abs(a0-SDS), abs(a1-SDS)), 
	                           rel(lq, abs(a0-SDE), abs(a1-SDE))
	                         ]), 
	                      and([rel(lq, abs(a0-SDS), abs(a1-SDS)),
	                           rel(gq, abs(a0-SDE), abs(a1-SDE))
	                         ])
	                    ]),           
	                  notex([sel([all]),
	                         from([Table-a3]),
	                         where([and([rel(gr, abs(a3-sst), abs(a0-sst)),
	                                  rel(ls, abs(a3-sst), abs(a1-sst)),
	                                  or([and([rel(lq, abs(a0-MDS), abs(a3-MDS)),
	                                           rel(gq, abs(a0-MDE), abs(a3-MDS))]),
	                                      and([rel(gq, abs(a0-MDS), abs(a3-MDS)),
	                                           rel(ls, abs(a0-MDE), abs(a3-MDE))]),
	                                      and([rel(gr, abs(a0-MDS), abs(a3-MDS)),
	                                           rel(lq, abs(a0-MDE), abs(a3-MDE))])]),
	                                  or([and([rel(lq, abs(a3-SDS), abs(a0-SDE)), 
	                                           rel(gq, abs(a3-SDE), abs(a0-SDE))]),
	                                      and([rel(lq, abs(a0-SDS), abs(a3-SDE)), 
	                                           rel(gq, abs(a0-SDE), abs(a3-SDE))]),
	                                      and([rel(gq, abs(a0-SDS), abs(a3-SDS)), 
	                                           rel(lq, abs(a0-SDE), abs(a3-SDE))]),
	                                      and([rel(lq, abs(a0-SDS), abs(a3-SDS)), 
	                                           rel(gq, abs(a0-SDE), abs(a3-SDE))])
	                                    ])
	                                  |Conditions13
	                                ])
	                            ])
	                       ])
	                  |Conditions12])])]), 
	      _, _),
	drop_aux_tables([Table]).

/********************************************************************************/
/* Mode:            reduce_system_time(+Table, +Cols, +TimeAttrs)               */
/* Purpose:        Coalesces rectangles in major time direction.                */
/* Example:                                                                     */
/* Sideeffects:    None                                                         */
/* Call:           Exists always                                                */
/* Redo:           Fails always                                                 */
/********************************************************************************/
 
%% update_main_dir(+Table, +TimeAttrs, +Conditions)
update_main_dir(Table, [MDS, MDE, SDS, SDE], Conditions) :-
	sql_exec([],  
           update(Table-a0,
                  set([a0-MDE],
                      [sel([max(abs(a1-MDE))]),
                       from([Table-a1]),
                       where([and([rel(gq, abs(a0-MDE), abs(a1-MDS)),
                                   rel(ls, abs(a0-MDE), abs(a1-MDE)),
                                   rel(eq, abs(a0-SDS), abs(a1-SDS)),
                                   rel(eq, abs(a0-SDE), abs(a1-SDE))|
                                   Conditions])])]),
                  where([and([ex([sel([all]),
                                  from([Table-a1]),
                                  where([and([rel(gq, abs(a0-MDE), abs(a1-MDS)),
                                              rel(ls, abs(a0-MDE), abs(a1-MDE)),
                                              rel(eq, abs(a0-SDS), abs(a1-SDS)),
                                              rel(eq, abs(a0-SDE), abs(a1-SDE))|
                                              Conditions])])])])])),
           N, _),
	N \== 0,
	!,
	update_main_dir(Table, [MDS, MDE, SDS, SDE], Conditions).
update_main_dir(_, _, _).

reduce_system_time(Table, Cols, [MDS, MDE, SDS, SDE]) :-
	create_eq_list('a0', 'a1', Cols, Conditions),
	update_main_dir(Table, [MDS, MDE, SDS, SDE], Conditions),
	sql_exec([],  
           delete(Table-a0,
                  where([and([ex([sel([all]),
                       from([Table-"a1"]),
                       where([and([rel(eq, abs(a0-SDS), abs(a1-SDS)),
                                   rel(eq, abs(a0-SDE), abs(a1-SDE)),
                                   rel(eq, abs(a0-MDE), abs(a1-MDE)),
                                   rel(ls, abs(a1-MDS), abs(a0-MDS))|
                                   Conditions])])])])])),
            _, _). 


/********************************************************************************/
/* Mode:          bitemporal_reduction(+Table-Cols, +TempType, -Result-ResCols) */
/* Purpose:       Bitemporal reduction valid-system or system-valid.            */
/* Example:                                                                     */
/* Sideeffects:   None                                                          */
/* Call:          Exits always                                                  */
/* Redo:          Fails always                                                  */
/********************************************************************************/

bitemporal_reduction(Table-Cols, validsystem, Result-ResCols) :-
	expand_rel(Table, Cols, validsystem, Res1, Cols1, TimeAttrs),
	reduce_sub_dir(Res1, Cols1, TimeAttrs),
	insert_endtime_stamps(Res1, Cols1, TimeAttrs),
	coalesce_main_dir(Res1, Cols1, TimeAttrs, Result, ResCols),
	reduce_system_time(Result, ResCols, TimeAttrs),
	drop_aux_tables([Table]).

bitemporal_reduction(Table-Cols, systemvalid, Result-ResCols) :-
	expand_rel(Table, Cols, systemvalid, Res1, Cols1, TimeAttrs),
	reduce_sub_dir(Res1, Cols1, TimeAttrs),
	insert_endtime_stamps(Res1, Cols1, TimeAttrs),
	coalesce_main_dir(Res1, Cols1, TimeAttrs, Result, ResCols),
	reduce_system_time(Result, ResCols, TimeAttrs),
	drop_aux_tables([Table]).


/********************************************************************************/
/* Mode:          do_temporal_reduction(+Coalescing, +Table-Columns, +TempType, */
/*                                      -Result-ResultColumns)                  */
/* Purpose:       Temporal reduction.                                           */
/* Example:                                                                     */
/* Sideeffects:   None                                                          */
/* Call:          Exits always                                                  */
/* Redo:          Fails always                                                  */
/********************************************************************************/

do_temporal_reduction(nored, Table, _, Table) :- !.
do_temporal_reduction(red, Table, snapshot, Table) :- !.
do_temporal_reduction(red, Table-Columns, valid, Table-Columns) :- 
	!, unitemporal_reduction(Table-Columns, valid).
do_temporal_reduction(red, Table-Columns, system, Table-Columns) :- 
	!, unitemporal_reduction(Table-Columns, system).
do_temporal_reduction(red, Table-Columns, validsystem, Result-ResultCols) :- 
	!, bitemporal_reduction(Table-Columns, validsystem, Result-ResultCols).
do_temporal_reduction(red, Table-Columns, systemvalid, Result-ResultCols) :-
	!, bitemporal_reduction(Table-Columns, systemvalid, Result-ResultCols).
