/******************************************************************************/
/* Module:      errors.pl                                                     */
/* Project:     TimeDB 1.04                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SICStus Prolog                                                */
/* Machine:     SPARC/Solaris, PowerMac/MacOS                                 */
/* Export:                                                                    */
/*              generate_error_flag/2 (parser.pl,check.pl,translate.pl,time.pl,evaluate.pl,meta_data.pl,fiscal_calendar.pl,sql3_calendar.pl,calendar.pl)*/
/*              print_error/0 (timeDB.pl)                                     */
/* Import:                                                                    */
/*              get_op/3 (basics.pl)                                          */
/*              get_expression/3 (basics.pl)                                  */
/******************************************************************************/

/****************************************************************************/
/* Mode:	generate_error_flag(+ErrorNumber, +ErrorPointer)	    */
/* Purpose:	Generates an error flag containing error# and error-pointer */
/* Example:		       				                    */
/* Sideeffects:	Error flag is asserted.		 			    */
/* Call:        Fails always						    */
/* Redo:        Fails always						    */
/****************************************************************************/

generate_error_flag(ErrorNumber, ErrorPointer) :-
	\+ error(_, _), assertz(error(ErrorNumber, ErrorPointer)), fail.


/****************************************************************************/
/* Mode:	remove_error_flag				            */
/* Purpose:	Removes an error flag				            */
/* Example:							            */
/* Sideeffects:	Error flag is retracted.		 		    */
/* Call:        Succeeds always						    */
/* Redo:        Fails always						    */
/****************************************************************************/

remove_error_flag :- retract(error(_, _)), !, remove_error_flag.
remove_error_flag.


/****************************************************************************/
/* Mode:	print_error					            */
/* Purpose:	Writes an error message to the screen                       */
/* Example:								    */
/* Sideeffects:	Error message is written to screen			    */
/* Call:        Fails always						    */
/* Redo:        Fails always						    */
/****************************************************************************/

print_error :-
	error(db_already_open, _), !, 
	nl, write('A database is open, close it before opening a new one'), nl, nl,
	remove_error_flag.

print_error :-
	error(no_db_open, _), !, 
	nl, write('No database is open'), nl, nl,
	remove_error_flag.

print_error :-
	error(syntax_error, _), !, 
	nl, write('Syntax Error'), nl, nl,
	remove_error_flag.

print_error :-
	error(union_compatibility, Op), !, 
	nl, write('Error: set operation * '), write(Op), 
	write(' * not union-compatible'), 
	nl, nl,
	remove_error_flag.

print_error :-
	error(column_expected, Token), !, 
	nl, write('Error: column reference expected instead of * '),
	write(Token), write(' *'), nl, nl,
	remove_error_flag.

print_error :-
	error(single_column, _), !, 
	nl, write('Error: single column reference in IN-subquery expected'),
	nl, nl,
	remove_error_flag.
	
print_error :-
	error(invalid_column, []-Attr), !, 
	nl, write('Error: invalid or ambiguous column name * '),  write(Attr), 
        write(' *'), nl, nl,
	remove_error_flag.

print_error :-
	error(column_not_exists, C), !, 
	nl, write('Error: column * '),  write(C),  write(' * does not exist'), nl, nl,
	remove_error_flag.

print_error :-
	error(invalid_column, Table-Attr), !, 
	nl, write('Error: invalid or ambiguous column name * '), 
	write(Table), write('.'), write(Attr), write(' *'), nl, nl,
	remove_error_flag.

print_error :-
	error(table_not_exist, Table), !, 
	nl, 
	write('Error: (derived) table or view * '), 
        write(Table), write(' * does not exist or'), nl,
	write('       does not contain the referenced attributes'),
	nl, nl,
	remove_error_flag.

print_error :-
	error(table_already_exists, Table), !, 
	nl, write('Error: table or view  * '), write(Table), 
	write(' * already exists'), 
	nl, nl,
	remove_error_flag.

print_error :-
	error(table_not_exist1, Table), !, 
	nl, write('Error: table or view  * '), write(Table), 
	write(' * does not exist'), 
	nl, nl,
	remove_error_flag.

print_error :-
	error(incompatible_types, [Op, Type1, Type2]), !,
	get_op(Op, OpStr, []),
	append(Type1, OpStr, Str1),
	append(Str1, Type2, Str2),	
	atom_chars(Exp, Str2), 
	nl, write('Error: operation of form * '), 
	write(Exp), write(' * is not defined'), 
	nl, nl,
	remove_error_flag.

print_error :-
	error(incompatible_types_in_cond, [Op, Type1, Type2]), !, 
	get_op(Op, OpStr, []),
	append(Type1, OpStr, Str1),
	append(Str1, Type2, Str2),	
	atom_chars(Exp, Str2), 
	nl, write('Error: condition of form * '), 
	write(Exp), write(' * is not defined'), 
	nl, nl,
	remove_error_flag.

print_error :-
	error(subquery_incompatible, TempType), !, 
	nl, write('Error: '),
	write(TempType), write(' subquery incompatible with main query'), nl, nl,
	remove_error_flag.

print_error :-
	error(no_calendar, _), !,
	nl, write('Error: no calendar available'), nl, nl,
	remove_error_flag.

print_error :- 
	error(name_already_used, Name), !,
	nl, write('Error: name * '), write(Name),
	write(' * is already used by an existing object'), nl, nl,
	remove_error_flag.

print_error :- 
	error(error_msg, Msg), !,
	nl, write(Msg), nl, nl,
	remove_error_flag.

print_error :- 
	error(select_item, Token), !,
	nl,
	write('Error: select expression expected instead of * '), write(Token), 
        write(' *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(table_expected, Token), !,
	nl,
	write('Error: table, view or derived table expected '), nl,
	write('       instead of * '), write(Token), write(' *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(table_name_expected, Token), !,
	nl,
	write('Error: Identifier expected instead of * '), write(Token), 
	write(' *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(expected_instead, Token1-Token2), !,
	nl,
	write('Error: * '), write(Token1), write(' * expected instead of * '), 
	write(Token2), write(' *'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(no_expression_allowed, Exp), !,
	nl,
	write('Error: You must name the expression * '), write(Exp), 
        write(' * with a column alias'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(type_expected, Token), !,
	nl,
	write('Error: Data type float, integer, period, timestamp, interval or char '), 
	nl,
        write('       expected instead of '), write(Token),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(user_def_time, Time), !,
	nl,
	write('Error: Not allowed to redefine * '), write(Time), 
        write(' * time'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(table_set_expected, Token), !,
	nl,
	write('Error: SELECT or ( expected instead of * '), write(Token), write(' *'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(coalescing_in_snapshot_query, _), !,
	nl,
	write('Error: coalescing not possible in snapshot query'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(temp_incompatible, _), !,
	nl,
	write('Error: query temporally incompatible with table/view'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(columns_incompatible, _), !,
	nl,
	write('Error: query incompatible with columns in table/view'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(number_of_columns, _), !,
	nl,
	write('Error: invalid number of column names specified in derived table/view'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(interval_type_expected, Exp), !,
	get_expression(Exp, Exp1, []),
	atom_chars(Exp2, Exp1),
	nl,
	write('Error: expression of type period expected in * '),
	write(Exp2), write(' *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(event_type_expected, Exp), !,
	get_expression(Exp, Exp1, []),
	atom_chars(Exp2, Exp1),
	nl,
	write('Error: expression of type timestamp expected in * '),
	write(Exp2), write(' *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(span_type_expected, Exp), !,
	get_expression(Exp, Exp1, []),
	atom_chars(Exp2, Exp1),
	nl,
	write('Error: expression of type interval expected in * '),
	write(Exp2), write(' *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(create_table, Token), !,
	nl,
	write('Error: * as * or column definition expected instead of * '),
	write(Token), write(' *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(create_view, Token), !,
	nl,
	write('Error: * as * or * ( * expected instead of * '),
	write(Token), write(' *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(interval_as_valid, _), !,
	nl,
	write('Error: only period expressions can be renamed as valid'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(interval_event_tables, _), !,
	nl,
	write('Error: only period or timestamp expressions allowed in valid clause'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(ident_expected, T), !,
	nl,
	write('Error: identifier expected instead of * '), write(T), write(' *'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(derived_table_alias, T), !,
	nl,
	write('Error: alias expected for derived table instead of * '), 
	write(T), write(' *'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(alter_column_exists, _), !,
	nl,
	write('Error: column being added already exists in table'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(alter_column_not_exists, _), !,
	nl,
	write('Error: column being removed does not exist in table'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(alter_impossible, _), !,
	nl,
	write('Error: alter table leads to table with no columns'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(not_implemented_yet, T), !,
	nl,
	write('Error: * '), write(T), write(' * not implemented'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(invalid_valid_exp, _), !,
	nl,
	write('Error: invalid valid-expression in time-flag'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(identifier_expected, T), !,
	nl,
	write('Error: identifier expected instead of * '), write(T), write(' *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(datatype_expected, T), !,
	nl,
	write('Error: data type expected instead of * '), write(T), write(' *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(keyword_table_view_expected, T), !,
	nl,
	write('Error: one of the keywords TABLE, VIEW or ASSERTION expected '),
        write('instead of * '), write(T), write(' *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(invalid_granularity, T), !,
	nl,
	write('Error: Granularity * '), 
	write(T), write(' * not allowed as increment'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(invalid_update_column, T), !,
	nl,
	write('Error: invalid column reference * '),
	write(T),
	write(' * in left hand side of set-clause'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(single_value_expected, _), !,
	nl,
	write('Error: subquery in set-clause must return exactely one tupel'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(aggregates_niy, Agg), !,
	nl,
	write('Error: aggregate function * '), write(Agg), 
	write(' * not implemented yet'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(illegal_reference, T), !,
	nl,
	write('Error: illegal table reference * '), write(T), write(' *'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(wrong_view_time, _), !,
	nl,
	write('Error: only snapshot and valid time views allowed'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(invalid_update_columns, _), !,
	nl,
	write('Error: use * Col = Constant * for constant values or '), nl,
	write('           * ( Col1, Col2, ... ) = <Query> *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(reference_to_interval, Col), !,
	nl,
	write('Error: reference to column * '),
	write(Col),
	write(' * of type period not implemented'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(still_referenced_columns, Table), !,
	nl,
	write('Error: table * '),
	write(Table),
	write(' * contains referenced columns'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(assertion_not_exists, Name), !,
	nl,
	write('Error: assertion * '), write(Name), write(' * does not exist'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(referenced_col_incompatible, Type), !,
	nl,
	write('Error: referenced columm of incompatible type * '),
	write(Type),
	write(' *'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(tables_incompatible, T1-_), !,
	nl,
	write('Error: table of type * '),
	write(T1), write(' * in FROM clause temporally incompatible'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(no_vt_table, Table-system), !,
	nl,
	write('Error: transaction table '), write(Table),
	write(' contains no valid time'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(no_vt_table, Table-Type), !,
	nl,
	write('Error: '), write(Type), write(' table/view '), write(Table), 
	write(' contains no valid time'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(no_tt_table, Table-Type), !,
	nl,
	write('Error: '), write(Type), write(' table/view '), write(Table),
	write(' contains no transaction time'), 
	nl, nl,
	remove_error_flag.

print_error :- 
	error(calendar_expected, T), !,
	nl,
	write('Error: calendar name expected instead of * '), write(T), write(' *'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(calendar_expected, T), !,
	nl,
	write('Error: calendar name expected instead of * '), write(T), write(' *'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(view_update, _), !,
	nl,
	write('Error: view updates are not supported'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(duplicate_col_names, Col), !,
	nl,
	write('Error: column name * '), write(Col), write(' * not unique'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(column_name_ambiguous, _-Col), !,
	nl,
	write('Error: column name * '), write(Col), 
	write(' * in select-clause of derived table not unique'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(demo_clock, _), !,
	nl,
	write('Error: no system time available !'), nl, 
	write('       Use command << SET CLOCK TO TIMESTAMP ... >>'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(invalid_year, Year), !,
	nl,
	write('Error: Calendar starts with year '), write(Year),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(too_many_update_columns, Col), !,
	nl,
	write('Error: Too many values for update: '), write(Col),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(ri_incompatible, Table), !,
	nl,
	write('Error: Referential integrity constraint to table * '), 
	write(Table),
	write(' * temporally incompatible'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(invalid_table_granularity, Gran), !,
	nl,
	write('Error: currently unsupported granularity * '), write(Gran), write(' *'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(too_many_constraints, Column),
	nl,
	write('Error: too many constraints for column * '), write(Column), write(' *'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(invalid_set_values, _),
	nl,
	write('Column being updated does not exist or has wrong type'),
	nl, nl,
	remove_error_flag.

print_error :- 
	error(db_not_exists, DBS),
	atom_chars(DB, DBS),
	nl,
	write('Database <'), write(DB),
	write('> does not exist'),
	nl, nl,
	remove_error_flag.

print_error :- nl.

