/******************************************************************************/
/* Module:      evaluate.pl                                                   */
/* Project:     TimeDB 1.05                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SICStus Prolog                                                */
/* Machine:     SPARC/Solaris, PowerMac/MacOS                                 */
/* Export:                                                                    */
/*              create_derived_tables/2 (constraints.pl,timeDB.pl)            */
/*              evaluate/4 (constraints.pl,timeDB.pl)                         */
/* Import:                                                                    */
/*              drop_aux_tables/1 (basics.pl)                                 */
/*              generate_table_name/2 (basics.pl)                             */
/*              write_status/0 (basics.pl)                                    */
/*              generate_error_flag/2 (errors.pl)                             */
/*              check_temp_compatibility/2 (check.pl)                         */
/*              check_granularity/2 (check.pl)                                */
/*              check_update_value/4 (check.pl)                               */
/*              get_conditions/5 (coalescing.pl)                              */
/*              add_alias/3 (coalescing.pl)                                   */
/*              do_temporal_reduction/4 (coalescing.pl)                       */
/*              set_demo_tt/2 (time.pl)                                       */
/*              retract_demo_tt/0 (time.pl)                                   */
/*              get_tt_date/2 (time.pl)                                       */
/*              check_primary_key/1 (constraints.pl)                          */
/*              check_referential_integrity/0 (constraints.pl)                */
/*              check_assertions/0 (constraints.pl)                           */
/*              check_table_column_constraints/0 (constraints.pl)             */
/*              db_open/1 (interface.pl)                                      */
/*              db_close/0 (interface.pl)                                     */
/*              sql_exec/4 (interface.pl)                                     */
/*              add_meta_data/3 (meta_data.pl)                                */
/*              delete_meta_data/1 (meta_data.pl)                             */
/*              add_view_meta_data/5 (meta_data.pl)                           */
/*              delete_view_meta_data/1 (meta_data.pl)                        */
/*              insert_ref_int/5 (meta_data.pl)                               */
/*              delete_ref_int/1 (meta_data.pl)                               */
/*              add_assertion/3 (meta_data.pl)                                */
/*              delete_assertion/1 (meta_data.pl)                             */
/*              add_key/2 (meta_data.pl)                                      */
/*              delete_key/1 (meta_data.pl)                                   */
/*              delete_key_attributes/2 (meta_data.pl)                        */
/*              add_table_column_constraints/2 (meta_data.pl)                 */
/*              delete_table_column_constraints/1 (meta_data.pl)              */
/*              change_calendar/1 (calendar.pl)                               */
/*              granularity/1 (calendar.pl)                                   */
/*              get_inc/2 (calendar.pl)                                       */
/******************************************************************************/

:- ensure_loaded([coalescing]).
:- ensure_loaded([time]).
:- ensure_loaded([constraints]).


%% create_column_name(+N, -Column, -N1)
create_column_name(N, Column, N1) :-
        name(N, NStr),
        append("c#$_", NStr, ColStr),
        name(Column, ColStr),
        N1 is N + 1.


retract_update_table :-
        retract(update_table(_)),
        !,
        retract_update_table.
retract_update_table :-
        retract(check_table(_)),
        !,
        retract_update_table.
retract_update_table.


/****************************************************************************/
/* Mode:           create_derived_tables(+DerivedTables, -TableList)        */
/* Purpose:        Creates all derived table of a query.                    */
/* Example:                                                                 */
/* Sideeffects:    Tables are created                                       */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

create_derived_tables([], []) :- !.
create_derived_tables([Tab-_-_|Derived], Rest) :-
        member(Tab-_-_, Derived),
        !,
        create_derived_tables(Derived, Rest).
create_derived_tables([Tab-SetOpStr-Cols|Derived], [Tab|Rest]) :-
        create_derived_tables(Derived, Rest),
        evaluate(SetOpStr, _, Query, DropTables),
        Query \== [],
        !,
        sql_exec([], create(Tab, Cols, Query), _, _), 
        drop_aux_tables(DropTables).


/****************************************************************************/
/* Mode:           add_rollback_data(+TableName, +TimeType)                 */
/* Purpose:        Keeps track of tables whose system time needs to be      */
/*                 updated after commit.                                    */
/* Example:                                                                 */
/* Sideeffects:    Fact is asserted.                                        */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

assert_data(T) :- update_table(T), !.	    % tt needs to be updated after commit
assert_data(T) :- assert(update_table(T)).  % tt needs to be updated after commit

check_data(T) :- check_table(T), !.	    % this table has changed, but does not	
check_data(T) :- assert(check_table(T)).    % contain tt (use to check constraints)

add_rollback_data(T, system)     :- atom(T), assert_data(T).
add_rollback_data(T, bitemporal) :- atom(T), assert_data(T).
add_rollback_data(T, _)          :- atom(T), check_data(T).


/****************************************************************************/
/* Mode:           update_rollback_data(+SystemTimePoint)                   */
/* Purpose:        Updates system time (after commit) of changed tables.    */
/* Example:                                                                 */
/* Sideeffects:    Facts are retracted.                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

update_rollback_data(ST) :-
        retract(update_table(T)),
        !,
        sql_exec([], 
                 update(T, set([T-'sts_#$'], [sel([cst(ST, number)]), from([dual])]), 
                           where([rel(gr, cst(0, number), T-'sts_#$')])), 
                 _, _),
        sql_exec([], 
                 update(T, set([T-'ste_#$'], [sel([cst(ST, number)]), from([dual])]), 
                           where([rel(gr, cst(0, number), T-'ste_#$')])), 
                 _, _),
        update_rollback_data(ST).
update_rollback_data(_).


/****************************************************************************/
/* Mode:           get_equal_conditions(+Alias1, +Alias2, +Cols1, +Time1,   */
/*                                      +Cols2, +Time2, -Conditions)        */
/* Purpose:        Composes equal conditions out of column-lists.           */
/* Example:                                                                 */
/* Sideeffects:    Table is created                                         */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

get_equal_conditions(_, _, [], _, _, _, []) :- !.
get_equal_conditions(A1, A2, [Col1|Rest1], snapshot, [Col2|Rest2], snapshot,
                     [rel(eq, A1-Col1, A2-Col2)|Conds]) :-
        !, get_equal_conditions(A1, A2, Rest1, snapshot, Rest2, snapshot, Conds).
get_equal_conditions(A1, A2, ['vts_#$', 'vte_#$'|Rest1], Time,
                             ['vts_#$', 'vte_#$'|Rest2], Time, Conds) :-
        !, get_equal_conditions(A1, A2, Rest1, Time, Rest2, Time, Conds).
get_equal_conditions(A1, A2, ['sts_#$', 'ste_#$'|Rest1], Time,
                             ['sts_#$', 'ste_#$'|Rest2], Time, Conds) :-
        !, get_equal_conditions(A1, A2, Rest1, Time, Rest2, Time, Conds).
get_equal_conditions(A1, A2, [Col1|Rest1], Time, [Col2|Rest2], Time,
                     [rel(eq, A1-Col1, A2-Col2)|Conds]) :-
        !, get_equal_conditions(A1, A2, Rest1, Time, Rest2, Time, Conds).

/****************************************************************************/
/* Mode:           get_select(+Table, +TempType, +ColumnList,               */
/*                            -SelectClause, -Start, -End)                  */
/* Purpose:        Composes select clause out of column-list.               */
/* Example:                                                                 */
/* Sideeffects:    Table is created                                         */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

get_select1(_, [], []) :- !.
get_select1(Table, [Ref|Columns], [Table-Ref|SelectClause]) :-
        !,
        get_select1(Table, Columns, SelectClause).
        
get_select(Table, valid, ['vts_#$','vte_#$'|Cols], Select, 'vts_#$', 'vte_#$') :- 
        !,
        get_select1(Table, Cols, Select).
get_select(Table, system, ['sts_#$','ste_#$'|Cols], Select, 'sts_#$', 'ste_#$') :-
        !,
        get_select1(Table, Cols, Select).


/****************************************************************************/
/* Mode:           get_intersect_select(+Table1, +Temp1, +Table2, +Temp2,   */
/*                                      +Cols, +Conds, -Select, -Conditions)*/
/* Purpose:        Composes intersect of time attributes for select clause. */
/* Example:                                                                 */
/* Sideeffects:    Table is created                                         */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

get_intersect_select(Table1, snapshot, _, snapshot, Cols, Cond, Select, Cond) :-
        !,
        get_select1(Table1, Cols, Select).
get_intersect_select(Table1, Temp, Table2, Temp, Cols, Conds, 
        [expr(gst([Table1-Start, Table2-Start]), Start),
         expr(lst([Table1-End, Table2-End]), End)|Select],
        [and([rel(ls, gst([Table1-Start, Table2-Start]), 
                      lst([Table1-End, Table2-End]))|Conds])]) :-
        !,
        get_select(Table1, Temp, Cols, Select, Start, End).


/****************************************************************************/
/* Mode:           get_column_names(+N, +ResType, -ColumnNames, -ProjAttrs) */
/* Purpose:        Gets all column-names out of result-type.                */
/* Example:                                                                 */
/* Sideeffects:    Table is created                                         */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

get_column_names(_, [], [], []) :- !.
get_column_names(N, [valid-interval-_|ResType], 
           ['vts_#$','vte_#$'|Columns], ['vts_#$'-noref,'vte_#$'-noref|Attrs]) :-
        !,
        get_column_names(N, ResType, Columns, Attrs).
get_column_names(N, [system-interval-_|ResType], 
           ['sts_#$','ste_#$'|Columns], ['sts_#$'-noref,'ste_#$'-noref|Attrs]) :-
        !,
        get_column_names(N, ResType, Columns, Attrs).
get_column_names(N, [_-(C1, C2)-interval-_|ResType], 
                 [C1, C2|Columns], [C1-noref, C2-noref|Attrs]) :-
        !,
        create_column_name(N, C1, N1),
        create_column_name(N1, C2, N2),
        get_column_names(N2, ResType, Columns, Attrs).
get_column_names(N, [_-C-_-_|ResType], [C|Columns], [C-noref|Attrs]) :-
        !,
        create_column_name(N, C, N1),
        get_column_names(N1, ResType, Columns, Attrs).
        

/****************************************************************************/
/* Mode:           create_aux_table(+SelectExp, -ResultTable, -TempType)    */
/* Purpose:        Creates a table ResultTable using SelectExp.             */
/* Example:                                                                 */
/* Sideeffects:    Table is created                                         */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

get_table_time_type(ResType, valid) :- member(valid-interval-_, ResType), !.
get_table_time_type(_, snapshot).


%% create_table(+Result, +Cols, +Select, +From, +Where, +TempType)
create_table(Result, Cols, distinct(Select), From, Where, _) :-
        !,
        sql_exec([], create(Result, Cols, [seld(Select), From, Where]), _, _).
create_table(Result, Cols, Select, From, Where, _) :-
        sql_exec([], create(Result, Cols, [sel(Select), From, Where]), _, _),
        !.

create_aux_table([distinct(snapshot(S))-ResType, F, W], Res-Cols, TT) :-
        !,
        get_table_time_type(ResType,  TT),
        generate_table_name("aux", Res),
        get_column_names(0, ResType, Cols, _),
        create_table(Res, Cols, distinct(S), F, W, snapshot).
create_aux_table([snapshot(Select)-ResType, F, W], Res-Cols, TT) :-
        !,
        get_table_time_type(ResType,  TT),
        generate_table_name("aux", Res),
        get_column_names(0, ResType, Cols, _),
        create_table(Res, Cols, Select, F, W, snapshot).
create_aux_table([valid(Select)-ResType, F, W], Res-Cols, valid) :-
        !,
        generate_table_name("aux", Res),
        get_column_names(0, ResType, Cols, _),
        create_table(Res, Cols, Select, F, W, valid).
create_aux_table([distinct(valid(Select))-ResType, F, W], Res-Cols, valid) :-
        !,
        generate_table_name("aux", Res),
        get_column_names(0, ResType, Cols, _),
        create_table(Res, Cols, distinct(Select), F, W, valid).

create_aux_table([system(Select)-ResType, F, W], Res-Cols, system) :-
        !,
        generate_table_name("aux", Res),
        get_column_names(0, ResType, Cols, _),
        create_table(Res, Cols, Select, F, W, system).
create_aux_table([distinct(system(Select))-ResType, F, W], Res-Cols, system) :-
        !,
        generate_table_name("aux", Res),
        get_column_names(0, ResType, Cols, _),
        create_table(Res, Cols, distinct(Select), F, W, system).

create_aux_table([validsystem(Sel)-ResType, F, W], Res-Cols, validsystem) :-
        !,
        generate_table_name("aux", Res),
        get_column_names(0, ResType, Cols, _),
        create_table(Res, Cols, Sel, F, W, validsystem).
create_aux_table([distinct(validsystem(Sel))-ResType, F, W], Res-Cols, validsystem) :-
        !,
        generate_table_name("aux", Res),
        get_column_names(0, ResType, Cols, _),
        create_table(Res, Cols, distinct(Sel), F, W, validsystem).

create_aux_table([systemvalid(Sel)-ResType, F, W], Res-Cols, systemvalid) :-
        !,
        generate_table_name("aux", Res),
        get_column_names(0, ResType, Cols, _),
        create_table(Res, Cols, Sel, F, W, systemvalid).
create_aux_table([distinct(systemvalid(Sel))-ResType, F, W], Res-Cols, systemvalid) :-
        !,
        generate_table_name("aux", Res),
        get_column_names(0, ResType, Cols, _),
        create_table(Res, Cols, distinct(Sel), F, W, systemvalid).

        
/****************************************************************************/
/* Mode:           calculate_union(+ResTable, +OrTable)                     */
/* Purpose:        Calculates the union of the ResultTable and OrTable      */
/*                 and returns the result in ResultTable.                   */
/* Example:                                                                 */
/* Sideeffects:    Or-Table is dropped.                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

get_proj(snapshot(S), S) :- !.
get_proj(valid(S), S) :- !.
get_proj(system(S), S) :- !.
get_proj(systemvalid(S), S) :- !.
get_proj(validsystem(S), S) :- !.


calculate_union(Result-_, OrTable-_) :-
        !,
        sql_exec([], insert(Result, [sel([a0-all]), from([OrTable-a0])]), _, _),
        sql_exec([], tdrop(OrTable), _, _).

calculate_union(Result-_, [sel(S), From, Where]) :-
        !,      
        sql_exec([], insert(Result, [sel(S), From, Where]),  _, _).
calculate_union(Result-_, [seld(S), From, Where]) :-
        !,      
        sql_exec([], insert(Result, [seld(S), From, Where]),  _, _).

calculate_union(Result-_, [distinct(S1)-_, From, Where]) :-
        !,
        get_proj(S1, S2),
        sql_exec([], insert(Result, [seld(S2), From, Where]),  _, _).
calculate_union(Result-_, [S1-_, From, Where]) :-
        !,
        get_proj(S1, S),
        sql_exec([], insert(Result, [sel(S), From, Where]),  _, _).



/****************************************************************************/
/* Mode:           calculate_minus(+Table, +TableTemp, +SubTab, +SubTemp,   */
/*                                 -Result)                                 */
/* Purpose:        Calculates Table sub SubTable.                           */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

%% snapshot_minus(+Table, +MinusTable, +Conditions)
snapshot_minus(Table, MinusTable, Conditions) :-
        !,
        sql_exec([], 
                 delete(Table-a0,
                            where([and([ex([sel([all]),
                                            from([MinusTable-a1]),
                                            where([and(Conditions)])])])])), 
                  _, _).

%% temp_minus(+Start, +End, +Result, +Minus, +SelectClause, +Conditions)
temp_minus(TS, TE, ResultTable, MinusTable, Select, Conditions) :-
        !,
        sql_exec([], insert(ResultTable,
                     [sel([a0-TS, a1-TS|Select]),
                      from([ResultTable-a0, MinusTable-a1]),
                      where([and([rel(gr, abs(a1-TS), abs(a0-TS)),
                                  rel(ls, abs(a1-TS), abs(a0-TE))|
                                  Conditions])])]), _, _),
        sql_exec([], insert(ResultTable,
                     [sel([a1-TE, a0-TE|Select]),
                      from([ResultTable-a0, MinusTable-a1]),
                      where([and([rel(gr, abs(a1-TE), abs(a0-TS)),
                                  rel(ls, abs(a1-TE), abs(a0-TE))|
                                  Conditions])])]), _, _),
        sql_exec([], delete(ResultTable-a0, where([and([ex(
                            [sel([a1-all]),
                             from([MinusTable-a1]),
                             where([and(
                                    [or([and([rel(gq, abs(a0-TS), abs(a1-TS)),
                                              rel(ls, abs(a0-TS), abs(a1-TE))]),
                                         and([rel(gq, abs(a1-TS), abs(a0-TS)),
                                              rel(ls, abs(a1-TS), abs(a0-TE))])])|
                                    Conditions])])])])])), _, _).

%% bitemp_minus(+Time, +TimeAttrs, +Table-Cols, +Minus-_, -Result)
bitemp_minus(Time, [MDS, MDE, SDS, SDE], Table-Cols, Minus-_, 
             Result-[MDS, MDE, SDS, SDE|Cols1]) :-
        get_conditions(Time, Cols, Conditions, [MDS, MDE, SDS, SDE], Cols1),
        add_alias('a0', Cols1, Columns),
        generate_table_name("aux", Result),
        sql_exec([],
          create(Result,
                 [MDS, MDE, SDS, SDE|Cols1],
                 [sel([all]),
                  from([Table-"a0"]),           
                  where([and([notex([sel([all]),
                                     from([Minus-"a1"]),
                                     where([and(Conditions)])])])])]),
           _, _),
        sql_exec([],
          insert(Result,
             [sel(["a0"-MDS, "a1"-MDS, "a0"-SDS, "a0"-SDE|Columns]),
              from([Table-"a0", Minus-"a1"]),
              where([and([rel(ls, gst(["a0"-MDS, "a1"-MDS]), 
                                  lst(["a0"-MDE, "a1"-MDE])),
                          rel(ls, gst(["a0"-SDS, "a1"-SDS]), 
                                  lst(["a0"-SDE, "a1"-SDE])),
                          rel(ls, "a0"-MDS, "a1"-MDS)|Conditions])])]),
            _, _), 
        sql_exec([],
          insert(Result,
             [sel(["a1"-MDE, "a0"-MDE, "a0"-SDS, "a0"-SDE|Columns]),
              from([Table-"a0", Minus-"a1"]),
              where([and([rel(ls, gst(["a0"-MDS, "a1"-MDS]), 
                                  lst(["a0"-MDE, "a1"-MDE])),
                          rel(ls, gst(["a0"-SDS, "a1"-SDS]), 
                                  lst(["a0"-SDE, "a1"-SDE])),
                          rel(gr, "a0"-MDE, "a1"-MDE)|Conditions])])]),
          _, _), 
        sql_exec([],
          insert(Result,
             [sel([gst(["a0"-MDS, "a1"-MDS]), lst(["a0"-MDE, "a1"-MDE]),
                   "a0"-SDS, "a1"-SDS|Columns]),
              from([Table-"a0", Minus-"a1"]),
              where([and([rel(ls, gst(["a0"-MDS, "a1"-MDS]), 
                                  lst(["a0"-MDE, "a1"-MDE])),
                          rel(ls, gst(["a0"-SDS, "a1"-SDS]), 
                                  lst(["a0"-SDE, "a1"-SDE])),
                          rel(ls, "a0"-SDS, "a1"-SDS)|Conditions])])]),
          _, _), 
        sql_exec([],
          insert(Result,
             [sel([gst(["a0"-MDS, "a1"-MDS]), lst(["a0"-MDE, "a1"-MDE]),
                        "a1"-SDE, "a0"-SDE|Columns]),
              from([Table-"a0", Minus-"a1"]),
              where([and([rel(ls, gst(["a0"-MDS, "a1"-MDS]), 
                                  lst(["a0"-MDE, "a1"-MDE])),
                          rel(ls, gst(["a0"-SDS, "a1"-SDS]), 
                                  lst(["a0"-SDE, "a1"-SDE])),
                          rel(gr, "a0"-SDE, "a1"-SDE)|Conditions])])]),
          _, _),
        sql_exec([],
          delete(Result-"a0",
             where([and([ex([sel([all]),
                             from([Minus-"a1"]),
                             where([and([rel(ls, gst(["a0"-MDS, "a1"-MDS]), 
                                                 lst(["a0"-MDE, "a1"-MDE])),
                                         rel(ls, gst(["a0"-SDS, "a1"-SDS]), 
                                                 lst(["a0"-SDE, "a1"-SDE]))|
                                         Conditions])])])])])),
          _, _).


calculate_minus(Table1-Cols1, snapshot, Table2-Cols2, snapshot, Table1-Cols1) :-
        check_temp_compatibility(snapshot, snapshot), !,
        get_equal_conditions('a0', 'a1', Cols1, snapshot, Cols2, snapshot, Conds),      
        snapshot_minus(Table1, Table2, Conds),
        drop_aux_tables([Table2]).
calculate_minus(Table1-Cols1, validsystem, Table2-Cols2, Temp2, Result-ResCols) :-
        check_temp_compatibility(validsystem, Temp2),
        bitemp_minus(validsystem, ['vts_#$','vte_#$','sts_#$','ste_#$'], 
                     Table1-Cols1, Table2-Cols2, Result-ResCols),
        drop_aux_tables([Table1, Table2]).
calculate_minus(Table1-Cols1, systemvalid, Table2-Cols2, Temp2, Result-ResCols) :-
        check_temp_compatibility(systemvalid, Temp2),
        bitemp_minus(systemvalid, ['sts_#$','ste_#$','vts_#$','vte_#$'], 
                     Table1-Cols1, Table2-Cols2, Result-ResCols),
        drop_aux_tables([Table1, Table2]).
calculate_minus(Table1-Cols1, Temp1, Table2-Cols2, Temp2, Table1-Cols1) :-
        check_temp_compatibility(Temp1, Temp2),
        get_equal_conditions('a0', 'a1', Cols1, Temp1, Cols2, Temp2, Conditions),
        get_select('a0', Temp1, Cols1, Select, Start, End),
        temp_minus(Start, End, Table1, Table2, Select, Conditions),
        drop_aux_tables([Table2]).
calculate_minus(T1-_, _, T2-_, _, _) :- drop_aux_tables([T1, T2]), fail. % Error


/****************************************************************************/
/* Mode:           calculate_intersect(+Tab1, +Temp1, +Tab2, +Temp2,        */
/*                                     -Result, -TempRes)                   */
/* Purpose:        Calculates Table intersect IntersectTab.                 */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

calculate_intersect(Tab1-Cols1, T1, Tab2-Cols2, T2, Result-Cols1, T1) :- 
        check_temp_compatibility(T1, T2),
        !,
        get_equal_conditions('a0', 'a1', Cols1, T1, Cols2, T2, Cond1),
        get_intersect_select('a0', T1, 'a1', T2, Cols1, Cond1, Select, Cond2),
        generate_table_name("aux", Result),
        sql_exec([], 
                 create(Result, 
                        Cols1, 
                        [seld(Select), 
                         from([Tab1-'a0', Tab2-'a1']), 
                         where([and(Cond2)])]), 
                  _, _),
        sql_exec([], tdrop(Tab1), _, _),
        sql_exec([], tdrop(Tab2), _, _).
calculate_intersect(T1-_, _, T2-_, _, _, _) :- drop_aux_tables([T1, T2]), fail. 

/****************************************************************************/
/* Mode:           evaluate_and(+Table, +Minus, +Intersect, -Result, -Temp) */
/* Purpose:        Transforms minus/intersect-expressions into standard sql */
/*                 statements and creates a table using sql statements.     */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

do_minus(Table, _, [], Table) :- !.
do_minus(Table, TempType, [or(Or)-Coal-_|Rest], Result) :- 
        !,
        evaluate_or(Or, OrResult, TempType1),
        do_temporal_reduction(Coal, OrResult, TempType1, Res1),
        calculate_minus(Table, TempType, Res1, TempType1, Result1),
        do_minus(Result1, TempType, Rest, Result).

do_intersect(Table, TempType, [], Table, TempType) :- !.
do_intersect(Table, TempType, [or(Or)-Coal-_|Rest], Result, TempRes) :- 
        !,
        evaluate_or(Or, OrResult, TempType1),
        do_temporal_reduction(Coal, OrResult, TempType1, Res1),
        calculate_intersect(Table, TempType, Res1, TempType1, Result2, TempType2),
        do_intersect(Result2, TempType2, Rest, Result, TempRes).


evaluate_and(Table, MinusList, IntersectList, Result, TempRes) :-
        !,
        create_aux_table(Table, Result1, TempType),
        do_minus(Result1, TempType, MinusList, Result2),
        do_intersect(Result2, TempType, IntersectList, Result, TempRes).


/****************************************************************************/
/* Mode:           evaluate_or(+Or_List, -ResultTable, -TempType)           */
/* Purpose:        Transforms a union-expression into standard sql          */
/*                 statements and creates a table using sql statements.     */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

%% evaluate_or1(+Result, +TempType, +Or_Table_List)
evaluate_or1(_, _, []) :- !.
evaluate_or1(Result, TempType, [and(Table, Minus, Intersect)|Or_List]) :-
        !,
        evaluate_and(Table, Minus, Intersect, Result1, TempType),
        calculate_union(Result, Result1),
        evaluate_or1(Result, TempType, Or_List).
evaluate_or1(Result, TempType, [SelectExp|Or_List]) :-
        !,
        calculate_union(Result, SelectExp),
        evaluate_or1(Result, TempType, Or_List).

        
evaluate_or([and(Table, MinusList, IntersectList)|Or_List], Result, TempType) :-
        !,
        evaluate_and(Table, MinusList, IntersectList, Result, TempType),
        evaluate_or1(Result, TempType, Or_List).
evaluate_or([SelectExp|Or_List], Result, TempType) :-
        !,
        create_aux_table(SelectExp, Result, TempType),
        evaluate_or1(Result, TempType, Or_List).


/****************************************************************************/
/* Mode:           evaluate_union(+Table-Cols, +UnionList)                  */
/* Purpose:        Calculates Table union UnionList.                        */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

evaluate_union(_, []) :- !.
evaluate_union(Table-Cols, [SetOpStr|Rest]) :-
        evaluate_query(SetOpStr, _, SFW, DropTabs),
        calculate_union(Table-Cols, SFW),
        drop_aux_tables(DropTabs),
        evaluate_union(Table-Cols, Rest).


/****************************************************************************/
/* Mode:           evaluate_intersect(+Table-Cols, +TempType, +IntList,     */
/*                                    -Result-ResultCols)                   */
/* Purpose:        Calculates Table intersect IntList.                      */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

evaluate_intersect(Table, _, [], Table) :- !.
evaluate_intersect(Table-Cols, Temp, [SetOpStr|Rest], Result) :-
        evaluate_query(SetOpStr, ResType, SFW, Tabs),
        generate_table_name("aux", Table2),
        get_column_names(0, ResType, Cols2, _),
        sql_exec([], create(Table2, Cols2, SFW), _, _),
        drop_aux_tables(Tabs),
        calculate_intersect(Table-Cols, Temp, Table2-Cols2, Temp, T-C, _),
        evaluate_intersect(T-C, Temp, Rest, Result).


/****************************************************************************/
/* Mode:           evaluate_minus(+Table-Cols, +TempType, +MinusList, -Res) */
/* Purpose:        Calculates Table minus MinusList.                        */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

evaluate_minus(Res, _, [], Res) :- !.
evaluate_minus(Table-Cols, Temp, [SetOpStr|Rest], Result) :-
        evaluate_query(SetOpStr, ResType, SFW, DropTabs),
        generate_table_name("aux", Table2),
        get_column_names(0, ResType, Cols2, _),
        sql_exec([], create(Table2, Cols2, SFW), _, _),
        drop_aux_tables(DropTabs),
        calculate_minus(Table-Cols, Temp, Table2-Cols2, Temp, Result1),
        evaluate_minus(Result1, Temp, Rest, Result).


/****************************************************************************/
/* Mode:           evaluate_status(+Control)                                */
/* Purpose:        Transforms status control commands.                      */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

evaluate_status(status)                  :- 
	write_status.
evaluate_status(trace)                   :-  
	!, retractall(tracing(_)), assert(tracing(on)).
evaluate_status(notrace)                 :- 
	!, retractall(tracing(_)), assert(tracing(off)).

evaluate_status(calendar(Cal))           :- change_calendar(Cal).       
evaluate_status(remove_demo_tt)          :- retract_demo_tt.
evaluate_status(demo_tt(Chronons, Gran)) :- 
	retract_demo_tt, 
	granularity(G),
	check_granularity(G, Gran),
	get_inc(Gran, Inc), 
	set_demo_tt(Chronons, Inc),
	nl, write('Clock value updated'), nl, nl.

/****************************************************************************/
/* Mode:           evaluate_query(+SetOpStr, -ResType, -ResQuery, -DropTabs)*/
/* Purpose:        Transforms a list of union-, intersect and minus-expr    */
/*                 into standard sql statements and returns an SQL statement*/
/*                 which gets the results.                                  */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

%% unify(+Table, +Refs, +AllRefs)
unify(_, [], _) :- !.
unify(Table, [Ref=Table-C|Refs], AllRefs) :-
        member(Ref=C, AllRefs),
        unify(Table, Refs, AllRefs).


evaluate_query(or(Or)-_-AllRef-S-SRT-Ref-nored-_, SRT, 
               [sel([all]), from([Table2]), where([])], [Table2]) :-
        !,
        evaluate_or(Or, Table1-_, Time),
        unify(Table1, Ref, AllRef),
        create_aux_table([S-SRT, from([Table1]), where([])], Table2-_, Time),
        drop_aux_tables([Table1]).
evaluate_query(or(Or)-_-AllRef-S-SRT-Ref-Coal-_, SRT, 
               [seld([all]), from([Result]), where([])], [Result]) :-
        !,
        evaluate_or(Or, Table1-_, _),
        unify(Table1, Ref, AllRef),
        create_aux_table([S-SRT, from([Table1]), where([])], Table2-Cols2, Time),
        drop_aux_tables([Table1]),
        do_temporal_reduction(Coal, Table2-Cols2, Time, Result-_).
evaluate_query(union([U|Rest])-Time-Coal-_, ResType, 
               [seld([all]), from([Result]), where([])], [Result]) :-
        !,
        evaluate_query(U, ResType, SFW, DropTabs),
        generate_table_name("aux", Table),
        get_column_names(0, ResType, Cols, _),
        sql_exec([], create(Table, Cols, SFW), _, _),
        drop_aux_tables(DropTabs),
        evaluate_union(Table-Cols, Rest),
        do_temporal_reduction(Coal, Table-Cols, Time, Result-_).
evaluate_query(intersect([I|Rest])-Time-Coal-_, ResType, 
               [seld([all]), from([Result]), where([])], [Result]) :-
        !,
        evaluate_query(I, ResType, SFW, DropTabs),
        generate_table_name("aux", Table1),
        get_column_names(0, ResType, Cols1, _),
        sql_exec([], create(Table1, Cols1, SFW), _, _),
        evaluate_intersect(Table1-Cols1, Time, Rest, Table2-Cols2),
        drop_aux_tables(DropTabs),
        do_temporal_reduction(Coal, Table2-Cols2, Time, Result-_).
evaluate_query(minus([M|Rest])-Time-Coal-_, ResType, 
               [seld([all]), from([Result]), where([])], [Result]) :-
        !,
        evaluate_query(M, ResType, SFW, DropTabs),
        generate_table_name("aux", Table),
        get_column_names(0, ResType, Cols, _),
        sql_exec([], create(Table, Cols, SFW), _, _),
        evaluate_minus(Table-Cols, Time, Rest, Result1),
        drop_aux_tables(DropTabs),
        do_temporal_reduction(Coal, Result1, Time, Result-_).


/****************************************************************************/
/* Mode:           evaluate_man(+Manipulation)                              */
/* Purpose:        Transforms data manipulation commands.                   */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

%% msg(+Operation, +Number)
msg(insert, N) :- nl, write(N), write(' tuple(s) inserted'), nl.
msg(delete, N) :- nl, (nonvar(N) -> write(N); true), write(' tuple(s) deleted'), nl.
msg(update, N) :- nl, (nonvar(N) -> write(N); true), write(' tuple(s) updated'), nl.

check(A, 'tt#$s_#$', abs(A-'tt#$s_#$')) :- !. 
check(A, 'tt#$e_#$', abs(A-'tt#$e_#$')) :- !. 
check(A, 'sts_#$', abs(A-'sts_#$'))     :- !. 
check(A, 'ste_#$', abs(A-'ste_#$'))     :- !. 
check(A, C, A-C).


%% delete-predicates
get_del_conditions(_, _, [], [], []) :- !.
get_del_conditions(A1, A2, [C1|Rest1], [C2|Rest2], [rel(eq, AC1, AC2)|Conds]) :-
	!,
	check(A1, C1, AC1), check(A2, C2, AC2),
        get_del_conditions(A1, A2, Rest1, Rest2, Conds).
get_del_conditions(A1, A2, [C1|Rest1], [C2|Rest2], [rel(eq, A1-C1, A2-C2)|Conds]) :-
        !, get_del_conditions(A1, A2, Rest1, Rest2, Conds).


%% update_tt(+Now, +Table, +MinusTable, +Where, -Nr)
update_tt(N, T, M, W, Nr) :-
        !, 
        sql_exec([], 
                 update(T-'a0', set(['a0'-'ste_#$'],[sel([N]),from([dual])]),
                        where([and([ex([sel([all]),from([M-'a1']),where([and(W)])]),
                                    rel(gr, abs('a0'-'ste_#$'), abs(N))])])),
                 Nr, _).


%% do_delete(+Time-Exp-TableType, +Now, +Table, +MinusCols, 
%%           +ManipSel, +ManipCol, +InsSelect, +UpdateCondCol, -Nr),
do_delete(snapshot-[]-snapshot, _, Table, _, _, _, _, TCC-MCC-M, Nr) :-
	get_del_conditions('a0', 'a1', TCC, MCC, W),
	sql_exec([], delete(Table-'a0', 
                     where([and([ex([sel([all]),from([M-'a1']),where([and(W)])])])])),
		 Nr, _).
do_delete(snapshot-[]-valid, _, Table, _, _, _, _, TCC-MCC-M, Nr) :-
	get_del_conditions('a0', 'a1', TCC, MCC, W),
	sql_exec([], delete(Table-'a0',
                            where([ex([sel([all]),from([M-'a1']),where([and(W)])])])),
		 Nr, _).
do_delete(snapshot-_-valid, _, Table, MiC-Minus, MaS, MaC, _, TCC-QCC-Q, Nr) :-
	!,
        get_del_conditions('a0', 'a1', TCC, QCC, W),
        generate_table_name("manip", Manip),
	sql_exec([], 
	         create(Manip, MaC, [MaS, from([Table-'a0']), 
                        where([ex([sel([all]),from([Q-'a1']),where([and(W)])])])]), 
                 _, _),	
	sql_exec([], delete(Table-'a0', 
                            where([ex([sel([all]),from([Q-'a1']),where([and(W)])])])),
		 Nr, _),
        get_select('a0', valid, MaC, Select, Start, End),
        get_equal_conditions('a0', 'a1', MaC, valid, MiC, valid, Conds),
        temp_minus(Start, End, Manip, Minus, Select, Conds),
	sql_exec([], insert(Table, [sel([all]), from([Manip])]), _, _),
	sql_exec([], tdrop(Manip), _, _).
%do_delete(valid-[]-valid, _, Table, MiC-Minus, MaS, MaC, _, TCC-QCC-Q, Nr) :-
do_delete(valid-_-valid, _, Table, MiC-Minus, MaS, MaC, _, TCC-QCC-Q, Nr) :-
	!,
        get_del_conditions('a0', 'a1', TCC, QCC, W),
        generate_table_name("manip", Manip),
	sql_exec([], 
	         create(Manip, MaC, [MaS, from([Table-'a0']), 
                        where([ex([sel([all]),from([Q-'a1']),where([and(W)])])])]), 
                 _, _),	
	sql_exec([], delete(Table-'a0',
                            where([ex([sel([all]),from([Q-'a1']),where([and(W)])])])),
		 Nr, _),
        get_select('a0', valid, MaC, Select, Start, End),
        get_equal_conditions('a0', 'a1', MaC, valid, MiC, valid, Conds),
        temp_minus(Start, End, Manip, Minus, Select, Conds),
	sql_exec([], insert(Table, [sel([all]), from([Manip])]), _, _),
	sql_exec([], tdrop(Manip), _, _).
do_delete(snapshot-[]-system, Now, Table, _, _, _, _, TCC-MCC-Minus, Nr) :-
	!,
	get_del_conditions('a0', 'a1', TCC, MCC, W),
	update_tt(Now, Table, Minus, W, Nr).
do_delete(snapshot-[]-bitemporal, Now, Table, _, _, _, _, TCC-MCC-Minus, Nr) :-
	!,
	get_del_conditions('a0', 'a1', TCC, MCC, W),
	update_tt(Now, Table, Minus, W, Nr).

do_delete(_-_-bitemporal, Now, Table, MiC-Minus, MaS, MaC, IS, TCC-QCC-Q, Nr) :-
	!,
        get_del_conditions('a0', 'a1', TCC, QCC, W),
        generate_table_name("manip", Manip),
	sql_exec([], 
	         create(Manip, MaC, [MaS, from([Table-'a0']), 
                        where([ex([sel([all]),from([Q-'a1']),where([and(W)])])])]), 
                 _, _),	
	update_tt(Now, Table, Q, W, Nr),
        get_select('a0', valid, MaC, Select, Start, End),
        get_equal_conditions('a0', 'a1', MaC, valid, MiC, valid, Conds),
        temp_minus(Start, End, Manip, Minus, Select, Conds),
	sql_exec([], insert(Table, [sel(IS), from([Manip-'a0'])]), _, _),
	sql_exec([], tdrop(Manip), _, _).


%% update-predicates
get_set_columns([], _, []) :- !.
get_set_columns([CL|Cols], RefList, [Ref-noref|Rest]) :-
        member(CL-Ref-_-_, RefList),
        get_set_columns(Cols, RefList, Rest).

%% remove_updated(+Delete, -NrUpdates) :-
remove_updated(delete(Now, Tab, T-E-TT, MinCol, ManipSel, ManipCol, InsSel, UCC), N) :-
	do_delete(T-E-TT, Now, Tab, MinCol, ManipSel, ManipCol, InsSel, UCC, N),
        add_rollback_data(Tab, TT).


evaluate_man(insert(Table-TT, Cols, Values))              :- 
        !, sql_exec([], insert(Table, Cols, Values), N, _),
        add_rollback_data(Table, TT),
        msg(insert, N).
evaluate_man(insert(Table-TT, Values))                    :- 
        !, sql_exec([], insert(Table, Values), N, _),
        add_rollback_data(Table, TT),
        msg(insert, N).
evaluate_man(delete(Now, Tab, Time-Exp-TT, MinCol, ManipSel, ManipCol, InsSel, UCC)) :-
        !, 
	do_delete(Time-Exp-TT, Now, Tab, MinCol, ManipSel, ManipCol, InsSel, UCC, N),
        add_rollback_data(Tab, TT),
        msg(delete, N). 
evaluate_man(update(T, CL, Ins, val(VL), Del)) :-
        !,
        evaluate_query(Ins, ResType, [_, from([InsTab]), _], DropTables),
        get_set_columns(CL, ResType, CL1),
        sql_exec([], update(InsTab, set(CL1, [sel(VL), from([dual])]), where([])), _, _),
        remove_updated(Del, N),
        sql_exec([], insert(T, [sel([all]), from([InsTab]), where([])]), _, _),
        drop_aux_tables(DropTables),
        msg(update, N).
evaluate_man(update(T, CL, Vars, Ins, Val, Del)) :-
        !,
        evaluate_query(Val, _, ValQuery, DropTables1),
        check_update_value(Vars, ValQuery, DropTables1, VL),
        evaluate_query(Ins, ResType, [_, from([InsTab]), _], DropTables2),
        get_set_columns(CL, ResType, CL1),
        sql_exec([], update(InsTab, set(CL1, [sel(VL), from([dual])]), where([])), _, _),
        remove_updated(Del, N),
        sql_exec([], insert(T, [sel([all]), from([InsTab]), where([])]), _, _),
        drop_aux_tables(DropTables1),
        drop_aux_tables(DropTables2),
        msg(update, N).
        

/****************************************************************************/
/* Mode:           evaluate_def(+Definition)                                */
/* Purpose:        Transforms table and view creation and deletion commands.*/
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

%% add_referential_integrity_constraints(Table, ReferencedAttr)
add_referential_integrity_constraints(_, []) :- !.
add_referential_integrity_constraints(Tab, [Col-ref(RefTab-RefCol-TimeType)|Refs]) :-
	!,
	insert_ref_int(Tab, Col, RefTab, RefCol, TimeType),
	add_referential_integrity_constraints(Tab, Refs).


evaluate_def(create_table(TableName, Cols+Refs+Key+Checks, [], TimeType)) :-
        !,
	add_referential_integrity_constraints(TableName, Refs),
	add_key(TableName, Key),
	add_table_column_constraints(TableName, Checks),
        sql_exec([], create(TableName, Cols), _, _),
        add_meta_data(TableName, Cols, TimeType),
        nl, write('Table created'), nl, nl.
evaluate_def(create_table(TableName, Cols-Data, Query, TimeType)) :-
        !,
        sql_exec([], create(TableName, Cols, Query), _, _),
        add_meta_data(TableName, Data, TimeType),
        add_rollback_data(TableName, TimeType),
        nl, write('Table created'), nl, nl.

evaluate_def(droptable(TableName)) :-
        !,
        sql_exec([], tdrop(TableName), _, _),
        delete_meta_data(TableName),
	delete_ref_int(TableName),
	delete_key(TableName),
	delete_table_column_constraints(TableName),
        (retract(update_table(TableName)), !; true),
        nl, write('Table dropped'), nl, nl.


evaluate_def(create_view(View, ColumnTypes, Query, Cols, TT)) :-
        !,
        add_view_meta_data(View, Query, Cols, ColumnTypes, TT),
        nl, write('View created'), nl, nl.

evaluate_def(dropview(Name)) :-
        delete_view_meta_data(Name),
        nl, write('View dropped'), nl, nl.


evaluate_def(assertion(Name, Type, Query, _)) :-
	!,
	add_assertion(Name, Query, Type),
	nl, write('Assertion added'), nl, nl.

evaluate_def(dropassertion(Name)) :-
	delete_assertion(Name), !,
        nl, write('Assertion dropped'), nl, nl.	
evaluate_def(dropassertion(Name)) :-
	generate_error_flag(assertion_not_exists, Name).


evaluate_def(alter(Table, check(Query))) :- 
        !,
	add_table_column_constraints(Table, [Query]),
	nl, write('Table altered'), nl, nl.
evaluate_def(alter(Table, TableTimeType, Cols, Select, Types, Where, DropCols)) :- 
        !,
        generate_table_name("altercopy", Aux), 
        sql_exec([], create(Aux, Cols, [sel(Select), from([Table]), where(Where)]), 
                 _, _),
        sql_exec([], tdrop(Table), _, _),
        sql_exec([], create(Table, [sel([all]), from([Aux])]), _, _),
        sql_exec([], tdrop(Aux), _, _),
        delete_meta_data(Table),
	delete_key_attributes(Table, DropCols),
        add_meta_data(Table, Types, TableTimeType),
        add_rollback_data(Table, TableTimeType),
        nl, write('Table altered'), nl, nl.

/****************************************************************************/
/* Mode:           evaluate_ctrl(+Control)                                  */
/* Purpose:        Evaluates a control command.                             */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

generate_error(F,A) :- generate_error_flag(F,A).
generate_error(_,_).


evaluate_ctrl(ctrl(check)) :-
        get_tt_date(now, ST),
	check_referential_integrity,
	check_primary_key(ST),
	check_assertions,
	check_table_column_constraints,
	!,
	nl, write('No integrity constraints violated'), nl, nl.	
evaluate_ctrl(ctrl(check)) :- !.
evaluate_ctrl(ctrl(commit)) :- 
        get_tt_date(now, ST),
	check_referential_integrity,
	check_primary_key(ST),
	check_assertions,
	check_table_column_constraints,
        !,
        update_rollback_data(ST),
        sql_exec([], lit("commit"), _, _),
        nl, write('Commit complete'), nl, nl.
evaluate_ctrl(ctrl(commit)) :-
	retract_update_table, 
	sql_exec([], lit("rollback"), _, _),
        nl, write('Rollback'), nl, nl.

evaluate_ctrl(ctrl(rollback)) :- 
        !, sql_exec([], lit("rollback"), _, _),
        retract_update_table,
        nl, write('Rollback complete'), nl, nl.

evaluate_ctrl(ctrl(open(UserList))) :- 
	(clause(db_is_open,true) ->
	   generate_error(db_already_open,_)
           ;
	   assert(emulator(on)),
	   assert(tracing(off)),
	   assert(calendar(fiscal)),
	   (db_open(UserList) ->
  	      (assert(db_is_open), nl, write('Database opened'), nl, nl);
  	      generate_error_flag(db_not_exists, UserList))).

evaluate_ctrl(ctrl(close)) :- 
	db_close,
	retractall(db_is_open),
	retractall(calendar(_)),
	nl, write('Database closed'), nl, nl.

evaluate_ctrl(ctrl(batch(I,O))) :- 
	atom_chars(I1,I),
	atom_chars(O1,O),
	see(I1), tell(O1).

evaluate_ctrl(ctrl(eof)) :- 
	seen, told.

/****************************************************************************/
/* Mode:           evaluate(+SetOpStruct, -ResType, -ResQuery, -DropTables) */
/* Purpose:        Transforms a list of union-, intersect and minus-expr    */
/*                 into standard sql statements and returns an SQL statement*/
/*                 which gets the results.                                  */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

evaluate(StatusControl, [], [], []) :-
        evaluate_status(StatusControl), !.
evaluate(Manipulation, [], [], []) :-
        evaluate_man(Manipulation), !.
evaluate(DataDefinition, [], [], []) :-
        evaluate_def(DataDefinition), !.
evaluate(SetOpStruct, ResType, ResQuery, DropTables) :-
        evaluate_query(SetOpStruct, ResType, ResQuery, DropTables), !.
evaluate(Ctrl, [], [], []) :- 
        evaluate_ctrl(Ctrl), !.
evaluate(_, [], [], []) :- !. 
