/******************************************************************************/
/* Module:      time.pl                                                       */
/* Project:     TimeDB 1.04                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SICStus Prolog                                                */
/* Machine:     SPARC/Solaris, PowerMac/MacOS                                 */
/* Export:                                                                    */
/*              set_demo_tt/2 (evaluate.pl,timeDB.pl)                         */
/*              retract_demo_tt/0 (evaluate.pl)                               */
/*              get_tt_date/2 (translate.pl,evaluate.pl,fiscal_calendar.pl,   */
/*                             decimal_calendar.pl,sql3_calendar.pl)          */
/*              tt_to_internal/4 (translate.pl)                               */
/* Import:                                                                    */
/*              generate_error_flag/2 (errors.pl)                             */
/*              my_sql_sel/3 (interface.pl)                                   */
/*              fiscal_calendar_start_year/1 (fiscal_calendar.pl)             */
/*              fiscal_min_number/1 (fiscal_calendar.pl)                      */
/*              fiscal_max_number/1 (fiscal_calendar.pl)                      */
/******************************************************************************/

foreign_file('$TIMEDB_HOME/timer.o',[get_system_time]).
foreign(get_system_time,c,
        get_system_time(-integer,-string,-integer,-integer,-integer,
                        -integer,-integer)).

:- load_foreign_files(['$TIMEDB_HOME/timer.o'], []).


month('Jan',1).   month('Feb',2).    month('Mar',3).    month('Apr',4).
month('May',5).   month('Jun',6).    month('Jul',7).    month('Aug',8).
month('Sep',9).   month('Oct',10).   month('Nov',11).   month('Dec',12).

/****************************************************************************/
/* Mode:           scan_time(-Year, -Month, -Day,                           */
/*                           -Hour, -Min, -Sec, -MilliSec)                  */
/* Purpose:        Scan transaction time string.                            */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

scan_year(N, N)   --> "/", !.
scan_year(N1, N3) --> [N], {N2 is N1 * 10 + (N-0'0)}, scan_year(N2, N3).

scan_month(N, N)   --> "/", !.
scan_month(N1, N3) --> [N], {N2 is N1 * 10 + (N-0'0)}, scan_month(N2, N3).

scan_day(N, N)   --> "~", !.
scan_day(N1, N3) --> [N], {N2 is N1 * 10 + (N-0'0)}, scan_day(N2, N3).

scan_hour(N, N)   --> ":", !.
scan_hour(N1, N3) --> [N], {N2 is N1 * 10 + (N-0'0)}, scan_hour(N2, N3).

scan_minute(N, N)   --> ":", !.
scan_minute(N1, N3) --> [N], {N2 is N1 * 10 + (N-0'0)}, scan_minute(N2, N3).

scan_second(N1, N3) --> [N], !, {N2 is N1 * 10 + (N-0'0)}, scan_second(N2, N3).
scan_second(N, N)   --> [].

scan_time(Year, Month, Day, Hour, Min, Sec, 0) -->
	scan_year(0, Year), scan_month(0, Month), scan_day(0, Day),
	scan_hour(0, Hour), scan_minute(0, Min), scan_second(0, Sec).


/****************************************************************************/
/* Mode:           set_demo_tt(+Chronons, +Increment)                       */
/* Purpose:        Sets transaction time for demo purposes.                 */
/* Example:                                                                 */
/* Sideeffects:    Assertion of fact demo/2.                                */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

set_demo_tt(Chronons, Inc) :- 
	retract_demo_tt,
	assert(demo(Chronons, Inc)).
	

/****************************************************************************/
/* Mode:           retract_demo_tt                                          */
/* Purpose:        Retracts demo-transaction time.                          */
/* Example:                                                                 */
/* Sideeffects:    Retraction of fact demo/2.                               */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

retract_demo_tt :-
	retract(demo(_, _)),
	!,
	retract_demo_tt.
retract_demo_tt.	

/****************************************************************************/
/* Mode:           get_tt(-Year, -Month, -Day, -Hour, -Min, -Sec, -MilliSec)*/
/* Purpose:        Gets transaction time.                                   */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

get_tt(Year, Month, Day, Hour, Min, Sec, MilliSec) :-
	emulator(on),
	prolog(sicstus),
	!,
	get_system_time(Year, MonthStr, Day, Hour, Min, Sec, MilliSec),
	month(MonthStr, Month).
get_tt(Year, Month, Day, Hour, Min, Sec, MilliSec) :-
	emulator(on),
	prolog(swi),
	!,
	get_time(X),
	convert_time(X,Year,Month,Day,Hour,Min,Sec,MilliSec).
get_tt(Year, Month, Day, Hour, Min, Sec, MilliSec) :-
	oracle(on),
	my_sql_sel([], "select to_char(sysdate,'YYYY/MM/DD~hh24:mi:ss') from dual", 
	           [var(Date, s)]),
	scan_time(Year, Month, Day, Hour, Min, Sec, MilliSec, Date, []).


/****************************************************************************/
/* Mode:           tt_to_internal(+Start, +End, -STS, -STE)                 */
/* Purpose:        Translates transaction time to internal format.          */
/*                 Note that STS is always negative (placeholder for end of */
/*                 transaction which allows use of temporal predicates      */
/*                 during transactions)!                                    */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

get_tt_date(now, Chronons) :-
	retract(demo(Chronons, Inc)), 
	!, 	
	NewChronons is Chronons + Inc,
	assert(demo(NewChronons, Inc)).	
get_tt_date(now, Now) :-
	get_tt(Year, Month, Day, Hour, Min, Sec, _),
	!,
	fiscal_calendar_start_year(Y), 
	Now is ((((((((((Year-Y)*12)+Month)*31)+Day)*24)+Hour)*60)+Min)*60)+Sec.
get_tt_date(now, _) :- 
	!,
	generate_error_flag(demo_clock, _).
get_tt_date(beginning, Min) :- !, fiscal_min_number(Min).
get_tt_date(forever, Max)   :- !, fiscal_max_number(Max).


tt_to_internal(Start, End, cst(NegSTS, number), cst(STE, number)) :-
	get_tt_date(Start, STS),
	NegSTS is -1 * STS,
	get_tt_date(End, STE).
