/******************************************************************************/
/* Module:      parser.pl                                                     */
/* Project:     Oracle Emulator                                               */
/* Author:      Michael Boehlen                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Export:                                                                    */
/*              read_lines/1 (oracle.pl)                                      */
/*              scan/2 (oracle.pl)                                            */
/*              tokenize/3 (oracle.pl)                                        */
/*              parse/2 (oracle.pl)                                           */
/*              u_expr/3 (norm_check.pl,sqlmsg.pl)                            */
/* Import:                                                                    */
/*              bf/3 (oracle.pl)                                              */
/*              error/1 (sqlmsg.pl)                                           */
/******************************************************************************/

%% read_lines(-String)
read_lines(T) :-
  prompt(_,'SQL> '),
  get0(X),
  (X=10 -> 
     read_lines(T);
     prompt(_,'     '),
     readln(X,T)).

readln(In,T) :-
  get0(X),
  ((In=0';, X=10) ->
     T=[];
     readln(X,T2),
     T=[In|T2]).

/******************************************************************************/
/* Mode:                scan(-TokenList) 	                              */
/* Purpose:             prompts for a string on the screen and tokenizes it   */
/* Example:             scan(-TokenList)                		      */
/* Sideeffects:         none                                                  */
/* Call:                exits always                                          */
/* Redo:                fails always                                          */
/******************************************************************************/

%% scan(+StmtString,-TokenList)
scan(Str,Tks) :- tokenize(Tks,Str,[]), !.

%% token(-Tokens,+In,+Out).
token(T)	--> "/*", skip_comment, token(T).
token(gq)	--> ">=", !.
token(lq)	--> "<=", !.
token(nq)	--> "<>", !.
token(gr)	--> ">", !.
token(ls)	--> "<", !.
token(eq)	--> "=", !.
token(add)	--> "+", !.
token(mul)	--> "*", !.
token(sub)	--> "-", !.
token(div)	--> "/", !.
token(lpar)	--> "(", !.
token(rpar)	--> ")", !.
token(pt)	--> ".", !.
token(comma)	--> ",", !.
token(colon)	--> ":", !.
token(str(Str))	--> [0''], !, scan_str(Str).
token(T)        -->
  char(X), !,
  scan_ident(Chrs),
  {atom_chars(A,[X|Chrs]), (reserved(A) -> T=A; T=id(A))}.
token(num(N))	--> 
  num(X), !, 
  scan_num(Chrs), 
  {number_chars(N,[X|Chrs])}.
token(T)	--> [_], token(T).
token([])	--> [].

skip_comment	--> "*/", !.
skip_comment	--> [_], skip_comment.

char(X)         --> [X], {X>=0'a,X=<0'z}, !.
char(Y)         --> [X], {X>=0'A, X=<0'Z, !, Y is X+0'a-0'A}.
char(0'#)       --> [0'#], !.
char(0'$)       --> [0'$], !.
char(0'_)       --> [0'_], !.

num(X)	--> [X], {X>=0'0,X=<0'9}.

%% tokenize(-TokenList,+String_In,+String_Out).
tokenize(R) --> token(T), tokenize2(T,R).
tokenize2([],[]) 	--> !, [].
tokenize2(T,[T|Tks])	--> token(T1), tokenize2(T1,Tks).

reserved(all).          reserved(and).          reserved(asc).
reserved(as).           reserved(avg).          reserved(by).
reserved(count).        reserved(create).       reserved(delete).
reserved(desc).         reserved(distinct).     reserved(drop).
reserved(exists).       reserved(from).         reserved(greatest).
reserved(group).        reserved(having).       reserved(in).
reserved(index).        reserved(intersect).    reserved(insert).
reserved(least).        reserved(max).          reserved(min).
reserved(minus).        reserved(null).         reserved(not).
reserved(nvl).          reserved(or).           reserved(order).
reserved(select).       reserved(set).          reserved(sum).
reserved(table).        reserved(update).       reserved(union).
reserved(values).       reserved(view).         reserved(where).
reserved(number).       reserved(char).         reserved(into).
reserved(integer).      reserved(long).         reserved(to_number).
reserved(float).	reserved(trunc).	reserved(sequence).
reserved(increment).    reserved(start).        reserved(with).
reserved(abs).          reserved(varchar).

%% scan_str(-String)
scan_str([0''|R])	--> [0'',0''], !, scan_str(R).
scan_str([])		--> [0''], !.
scan_str([X|R])		--> [X], scan_str(R).

%% scan_ident(-String)
scan_ident([X|R])	--> char(X), !, scan_ident(R).
scan_ident([X|R])	--> num(X), !, scan_ident(R).
scan_ident([])		--> [].

%% scan_num(-String)
scan_num([X|R])		--> num(X), !, scan_num(R).
scan_num([0'.|R])	--> [0'.], !, scan_num2(R).
scan_num([])		--> [].
scan_num2([X|R])	--> num(X), !, scan_num2(R).
scan_num2([])		--> [].

/******************************************************************************/
/* Mode:                parse(-TokenList) 	                              */
/* Purpose:             parses a token list and builds a perse tree	      */
/* Example:             parse(-ParseTree,+TokenList_In,+TokenList_Out)        */
/* Sideeffects:         none                                                  */
/* Call:                exits always                                          */
/* Redo:                fails always                                          */
/******************************************************************************/

parse(ParseTree,TokenList) :- parse(ParseTree,TokenList,[]), !.
parse(_,_) :- error(ill_cmd), fail.

p_chk(X) --> [X], !.
p_chk(X) --> [Y], {error(exp_for(X,Y)), fail}.

%% DDL (Data Definition Language)
%% ------------------------------

%% parse(-Tree,+TokenList,+TokenList).
parse(create_table(T,Cs,Q)) --> 
  [create, table], !, 
  p_chk(id(T)), 
  p_table_def(Cs,Q).
parse(create_view(V,Cs,query(S,Q,[]))) -->
  [create, view], !, 
  p_chk(id(V)), 
  p_col_names(Cs),
  p_chk(as), 
  parse(query(S,Q,[])), !.
parse(create_sequence(N,S,I)) -->
  [create, sequence], !, 
  p_chk(id(N)),
  p_seq_incr(I),
  p_seq_start(S).
parse(_) --> [create, Y], !, {error(exp_for('TABLE/VIEW',Y)), fail}, [_].
parse(drop_table(T)) --> [drop, table], !, [id(T)].
parse(drop_view(V)) --> [drop, view], !, [id(V)].
parse(drop_sequence(S)) --> [drop, sequence], !, [id(S)].
parse(_) --> [drop, Y], !, {error(exp_for('TABLE/VIEW',Y)), fail}, [_].

%% p_table_def(-ColumnNames,-QueryStmt)
p_table_def(Cs,query(S,Q,[])) -->
  p_col_names(Cs), !,
  p_chk(as),
  parse(query(S,Q,[])).
p_table_def(Cs,[]) --> [lpar], p_col_defs(Cs), !, p_chk(rpar).

p_seq_incr(I) -->
  [increment, by], !,
  p_chk(num(I)).
p_seq_incr(1) --> [].

p_seq_start(S) -->
  [start, with], !,
  p_chk(num(S)).
p_seq_start(1) --> [].

%% DML (Data Manipulation Language)
%% --------------------------------

parse(delete(T-A,W)) --> 
  [delete], !, p_chk(from), p_chk(id(T)), p_alias(A), 
  p_where(W).
parse(insert(T,Cs,V)) --> 
  [insert], !, p_chk(into), p_chk(id(T)), 
  p_col_names(Cs),
  p_insert(V).
parse(update(T-A,CL,query(S,Q,[]),W)) -->
  [update], !,
  p_chk(id(T)), p_alias(A),
  p_chk(set),
  p_chk(lpar),
  p_exprList(CL),
  p_chk(rpar),
  p_chk(eq),
  p_chk(lpar),
  parse(query(S,Q,[])),
  p_chk(rpar),
  p_where(W).

p_insert(query(S,sel(all,S,[dual-dual],true),[])) -->
  [values], !,
  p_chk(lpar),
  p_selList(S),
  p_chk(rpar).
p_insert(query(S,Q,[])) --> parse(query(S,Q,[])).

%% SELECT statements (queries)
%% ---------------------------

parse(query(S,Out,O)) -->
  parse1(S,Sel),
  parse2(_,Sel,Out),
  p_orderby(O).
parse1(S,Out) -->
  [lpar], !,
  parse1(S,Q1),
  parse2(_,Q1,Out),
  p_chk(rpar).
parse1(S,sel(M,S,F,W)) -->
  p_select(sel(M,S,F,W)).
parse2(S,In,Out) -->
  [union], !,
  parse1(S,Sel),
  parse2(_,union(In,Sel),Out).
parse2(S,In,Out) -->
  [minus], !,
  parse1(S,Sel),
  parse2(_,minus(In,Sel),Out).
parse2(_,In,In) --> [].

p_select(sel(M,S,F,W)) --> 
  [select], p_mode(M), p_selList(S), 
  p_from(F),
  p_where(W), !.

p_selList([colS(SE,N)|R])  --> p_expr(SE), p_alias(N), p_selList2(R).
p_selList2([colS(SE,N)|R]) --> [comma], !, p_expr(SE), p_alias(N), p_selList2(R).
p_selList2([])             --> [].

p_from([T-A|Ts])     --> p_chk(from), p_chk(id(T)), p_alias(A), p_from2(Ts).
p_from2([T-A|Ts])    --> [comma], !, p_chk(id(T)), p_alias(A), p_from2(Ts).
p_from2([])	     --> [].

p_where(W)           --> [where], !, p_bool_expr(W).
p_where(true)        --> [].

p_bool_expr(Out)     --> p_bool_term(BT), p_bool_expr2(BT,Out).
p_bool_expr2(In,Out) --> [or], !, p_bool_term(BT), p_bool_expr2(or(In,BT),Out).
p_bool_expr2(In,In)  --> [].

p_bool_term(Out)     --> p_bool_fac(BF), p_bool_term2(BF,Out).
p_bool_term2(In,Out) --> [and], !, p_bool_fac(BF), p_bool_term2(and(In,BF),Out).
p_bool_term2(In,In)  --> [].

p_bool_fac(not(F))   --> [not], !, p_bool_fac(F).
p_bool_fac(ex(Q))    --> [exists], !, p_subquery(Q).
p_bool_fac(in(EL,Q)) -->
  [lpar],
  p_exprList(EL),
  [rpar, in], !,
  p_subquery(Q).
p_bool_fac(F)        --> p_expr(E), !, p_bool_fac2(E,F).
p_bool_fac2(E1,rel(Rel,E1,E2)) 	-->
  [Rel], {memberchk(Rel,[eq,nq,lq,ls,gq,gr])}, !, p_expr(E2).
p_bool_fac2(E,in([E],Q))	--> [in], !, p_subquery(Q).
p_bool_fac2(E,not(in([E],Q)))	--> [not], !, p_chk(in), p_subquery(Q).
p_bool_fac2(E,E)        	--> [].

p_subquery(query(S,Q,[])) --> p_chk(lpar), parse(query(S,Q,[])), p_chk(rpar).

p_orderby([SqlExp-Mode|O]) -->
  [order], !,
  p_chk(by),
  p_expr(SqlExp),
  p_orderby_mode(Mode),
  p_orderby2(O).
p_orderby([]) --> [].
p_orderby2([SqlExp-Mode|O]) -->
  [comma], !,
  p_expr(SqlExp),
  p_orderby_mode(Mode),
  p_orderby2(O).
p_orderby2([]) --> [].

p_orderby_mode(desc)  -->  [desc], !.
p_orderby_mode(asc)   -->  [asc], !.
p_orderby_mode(asc)   -->  [].

%% MISCELLANEOUS
%% -------------

p_expr(Res)     --> p_term(T), p_expr2(T,Res).
p_expr2(In,Out) -->
  [Op], {memberchk(Op,[add,sub])}, !, p_term(T), p_expr2(bin(Op,In,T),Out).
p_expr2(In,In)  --> [].

p_term(Res)     --> p_fac(F), p_term2(F,Res).
p_term2(In,Out) -->
  [Op], {memberchk(Op,[mul,div])}, !, p_fac(F), p_term2(bin(Op,In,F),Out).
p_term2(In,In)  --> [].

p_fac(un(Op,F))   --> [Op], {memberchk(Op,[add,sub])}, !, p_fac(F).
p_fac(all)        --> [mul], !.
p_fac(input(N,_)) --> [colon], !, p_chk(num(N)).
p_fac(num(N))     --> [num(N)], !.
p_fac(str(S))     --> [str(S)], !.
p_fac(C)          --> p_col(C), !.
p_fac(gst(L))     --> [greatest], !, p_chk(lpar), p_exprList(L), p_chk(rpar).
p_fac(lst(L))     --> [least], !, p_chk(lpar), p_exprList(L), p_chk(rpar).
p_fac(to_num(E))  --> [to_number], !, p_chk(lpar), p_expr(E), p_chk(rpar).
p_fac(trunc(E))   --> [trunc], !, p_chk(lpar), p_expr(E), p_chk(rpar).
p_fac(abs(E))     --> [abs], !, p_chk(lpar), p_expr(E), p_chk(rpar).
p_fac(F)          --> [lpar], !, p_bool_expr(F), p_chk(rpar).
p_fac(agg(A,M,E)) -->
  [A], {memberchk(A,[avg,max,min,sum,count])}, !,
  p_chk(lpar), p_mode(M), p_expr(E), p_chk(rpar).

p_exprList([E|EL])	--> p_expr(E), p_exprList2(EL).
p_exprList2([E|EL])	--> [comma], !, p_expr(E), p_exprList2(EL).
p_exprList2([])		--> [].

p_alias(A) --> [id(A)], !.
p_alias(_) --> [].

p_mode(distinct) --> [distinct], !.
p_mode(all) --> [all], !.
p_mode(all) --> [].

p_type(char) --> [varchar, lpar, num(_), rpar], !.
p_type(char) --> [varchar], !.
p_type(char) --> [char, lpar, num(_), rpar], !.
p_type(char) --> [char], !.
p_type(char) --> [long], !.
p_type(num)  --> [number, lpar, num(_), rpar], !.
p_type(num)  --> [number], !.
p_type(num)  --> [integer], !.
p_type(num)  --> [float], !.

p_col(colR(_-A,_,N))   --> [id(A), pt, id(N)], !.
p_col(colR(_-A,_,all)) --> [id(A), pt, mul], !.
p_col(colR(_-_,_,N))   --> [id(N)].

p_col_names([C|Cs])  --> [lpar, id(C)], !, p_col_names2(Cs), [rpar].
p_col_names([])      --> [].
p_col_names2([C|Cs]) --> [comma, id(C)], !, p_col_names2(Cs).
p_col_names2([])     --> [].

p_col_defs([C|Cs])  --> p_col_def(C), p_col_defs2(Cs).
p_col_defs2([C|Cs]) --> [comma], !, p_col_def(C), p_col_defs2(Cs).
p_col_defs2([])     --> [].

p_col_def(colD(N,T,Null)) --> p_chk(id(N)), p_type(T), p_null(Null).

p_null(not_null) --> [not], !, [null].
p_null(null)     --> [null], !.
p_null(null)     --> [].


%% Transactions
%% ------------

parse(commit)         --> [id(commit)].
parse(rollback(RBP))  --> [id(rollback)], p_RBP(RBP).
parse(savepoint(RBP)) --> [id(savepoint)], p_chk(id(RBP)).

p_RBP(RBP)            --> [id(to)], !, p_chk(id(RBP)).
p_RBP('$$work$$')     --> [].

%% Additional commands (sqlplus)
%% -----------------------------

parse(quit) --> [id(quit)].
parse(quit) --> [id(exit)].

/******************************************************************************/

u_atom(A)              --> {atom_chars(A,Chrs)}, bf(Chrs).
u_number(N)            --> {number_chars(N,Chrs)}, bf(Chrs).

u_mode(distinct)       --> "DISTINCT ".
u_mode(all)            --> "".

u_expr(bin(add,L,R))   --> !, u_expr(L), "+", u_term(R).
u_expr(bin(sub,L,R))   --> !, u_expr(L), "-", u_term(R).
u_expr(A)              --> u_term(A).

u_term(bin(mul,L,R))   --> !, u_term(L), "*", u_fac(R).
u_term(bin(div,L,R))   --> !, u_term(L), "/", u_fac(R).
u_term(A)              --> u_fac(A).

u_fac(un(add,F))       --> !, "+", u_fac(F).
u_fac(un(sub,F))       --> !, "-", u_fac(F).
u_fac(all)             --> !, "*".
u_fac(input(N,_))      --> !, u_number(N).
u_fac(trunc(E))        --> !, "TRUNC(", u_expr(E), ")".
u_fac(abs(E))          --> !, "ABS(", u_expr(E), ")".
u_fac(to_num(E))       --> !, "TO_NUMBER(", u_expr(E), ")".
u_fac(num(N))          --> !, u_number(N).
u_fac(str(S))          --> !, bf(S).
u_fac(colR(_-A,_,N))   --> !, u_atom(A), ".", u_atom(N).
u_fac(gst(L))          --> !, "GREATEST(", u_exprList(L), ")".
u_fac(lst(L))          --> !, "LEAST(", u_exprList(L), ")".
u_fac(agg(A,M,E))      --> !, u_atom(A),"(",  u_mode(M), u_expr(E), ")".
u_fac(E)               --> "(", u_expr(E), ")".

u_exprList([H|T])      --> u_expr(H), u_exprList2(T).
u_exprList2([H|T])     --> ",", u_expr(H), u_exprList2(T).
u_exprList2([])        --> "".

/******************************************************************************/
/* .									      */
/******************************************************************************/
