/******************************************************************************/
/* Module:      check.pl                                                      */
/* Project:     TimeDB 1.03                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Date:        December 12, 1995                                             */
/* Export:                                                                    */
/*              get_table_alias/4 (translate.pl)                              */
/*              get_column_length/2 (translate.pl,meta_data.pl)               */
/*              get_type/2 (meta_data.pl)                                     */
/*              check_temp_compatibility/2 (evaluate.pl)                      */
/*              check_db_open/1 (timeDB.pl)                                   */
/*              check_time_type/4 (translate.pl)                              */
/*              check_ambiguity/3 (translate.pl)                              */
/*              check_granularity/2 (evaluate.pl)                             */
/*              check_time_compatibility/3 (translate.pl)                     */
/*              check_temporal_exp/3 (translate.pl)                           */
/*              check_del_compatibility/2 (translate.pl)                      */
/*              check_alter/3 (translate.pl)                                  */
/*              check_length/2 (translate.pl)                                 */
/*              check_reference/3 (translate.pl)                              */
/*              check_still_referenced/1 (translate.pl)                       */
/*              check_column_names/1 (translate.pl)                           */
/*              check_create_compatibility/5 (translate.pl)                   */
/*              check_update_columns/4 (translate.pl)                         */
/*              check_update_values/4 (translate.pl)                          */
/*              check_update_value/4 (evaluate.pl)                            */
/*              check_insert_query_compatibility/2 (translate.pl)             */
/*              check_insert_values_compatibility/2 (translate.pl)            */
/*              check_user_defined_valid/2 (translate.pl)                     */
/*              check_union_compatibility/2 (translate.pl)                    */
/*              check_union_time_compatibility/2 (translate.pl)               */
/*              check_column_constraint_number/3 (translate.pl)               */
/*              check_cond_type_compatibility/3 (translate.pl)                */
/*              check_exp_type/7 (translate.pl)                               */
/* Import:                                                                    */
/*              drop_aux_tables/1 (basics.pl)                                 */
/*              get_expression/3 (basics.pl)                                  */
/*              generate_interval_col/3 (basics.pl)                           */
/*              get_interval_name/3 (basics.pl)                               */
/*              generate_error_flag/2 (errors.pl)                             */
/*              sql_sel/3 (meta.pl)                                           */
/*              get_table_type/2 (meta_data.pl)                               */
/*              get_table_infos/3 (meta_data.pl)                              */
/*              table_exists/1 (meta_data.pl)                                 */
/*              get_ref_int/2 (meta_data.pl)                                  */
/******************************************************************************/

% get_table_alias(+Ref, +From, -Table, -Alias)
get_table_alias(Ref, From, Ref, Alias) :- member(Ref-Alias-_, From), !.
get_table_alias(Ref, From, Table, Ref) :- member(Table-Ref-_, From), !.

%% get_column_length(?Type, ?Length)
get_column_length(char, 15)     :- !. % 30
get_column_length(s(Len), Len)  :- !.
get_column_length(number, 10)   :- !. % 16
get_column_length(integer, 10)  :- !. % 16
get_column_length(interval, 45) :- !.
get_column_length(event, 15)    :- !. % 25
get_column_length(span, 15)     :- !. % 20

%% get_type(?Type, ?TypeString)
get_type(number, "number")     :- !.
get_type(integer, "number")    :- !.
get_type(char, "char")         :- !.
get_type(s(_), "char")         :- !.
get_type(interval, "interval") :- !.
get_type(event, "event")       :- !.
get_type(span, "span")         :- !.


check_single_column(Type, [_-_-Type-_]) :- !.
check_single_column(Type, [_-Type-_]) :- !.
check_single_column(Type, [valid-interval-_, _-_-Type-_]) :- !.
check_single_column(Type, [valid-interval-_, _-Type-_]) :- !.
check_single_column(Type, [system-interval-_, _-_-Type-_]) :- !.
check_single_column(Type, [system-interval-_, _-Type-_]) :- !.
check_single_column(Type, [valid-interval-_, system-interval-_, _-_-Type-_]) :- !.
check_single_column(Type, [valid-interval-_, system-interval-_, _-Type-_]) :- !.


%% check_temp_compatibility(+TempType1, +TempType2)
check_temp_compatibility(TempType, TempType)    :- !.
check_temp_compatibility(_, TempType)           :- !,
        generate_error_flag(subquery_incompatible, TempType).


/****************************************************************************/
/* Mode:           check_db_open                                            */
/* Purpose:        Checks if a databse is open.                             */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_db_open(data_control(open(_))) :- !.
check_db_open(_)                     :- clause(db_is_open, true), !.
check_db_open(_)                     :- generate_error_flag(no_db_open, _).


/****************************************************************************/
/* Mode:           check_time_type(+Modus,+TimeType1,+TimeType2,-TimeType)  */
/* Purpose:        Checks if tables in from clause are temporally           */
/*                 compatible.                                              */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_time_type(_, Type, [], Type)                      :- !.
check_time_type(snapshot, _, _, snapshot)               :- !.
check_time_type(valid, bitemporal, bitemporal, valid)   :- !.
check_time_type(valid, valid, bitemporal, valid)        :- !.
check_time_type(valid, bitemporal, valid, valid)        :- !.
check_time_type(system, bitemporal, bitemporal, system) :- !.
check_time_type(system, system, bitemporal, system)     :- !.
check_time_type(system, bitemporal, system, system)     :- !.
check_time_type(Type, Type, Type, Type)                 :- !.
check_time_type(_, T1, T2, _)             :- 
	generate_error_flag(tables_incompatible, T1-T2).


/****************************************************************************/
/* Mode:           check_ambiguity(+ViewName, +Col, +RestColumns)           */
/* Purpose:        Checks if a column-name in the select-clause of a        */
/*                 derived table only occurs once.                          */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_ambiguity(View, Col, RestColumns) :-
	member(Col-_-_-_, RestColumns),
	!,
	generate_error_flag(column_name_ambiguous, View-Col). 
check_ambiguity(_, _, _).

/****************************************************************************/
/* Mode:           check_granularity(+CalGran, +AddGran)                    */
/* Purpose:        Checks if a granularity for the demo-clock is compatible */
/*                 with the calendar used.                                  */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_granularity(second, _) :- !.
check_granularity(minute, G) :- G \== second, !.
check_granularity(hour, G)   :- G \== second, G \== minute, !.
check_granularity(day, G)    :- G \== second, G \== minute, G \== hour, !.
check_granularity(month, G)  :- G \== second, G \== minute, G \== hour, G \== day, !.
check_granularity(year, G)   :- G \== second, G \== minute, G \== hour, 
                                G \== day, G \== month,  !.
check_granularity(_, none)   :- !.
check_granularity(_, G)      :- generate_error_flag(invalid_granularity, G).


/****************************************************************************/
/* Mode:           check_time_compatibility(+TUC, +TableType, +QueryType)   */
/* Purpose:        Checks if insert-table is temporally consistent with a   */
/*                 query or constant values.                                */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_time_compatibility(tuc-_-_, _, _)  :- !.
check_time_compatibility(_, bitemporal, valid)  :- !.
check_time_compatibility(_, valid, valid)       :- !.
check_time_compatibility(_, system, snapshot)   :- !.
check_time_compatibility(_, snapshot, snapshot) :- !.
check_time_compatibility(_, _, TimeType)        :-
        generate_error_flag(temp_incompatible, TimeType).


/****************************************************************************/
/* Mode:           check_temporal_exp(+Expression, +Table, -TempExpression) */
/* Purpose:        Checks if a user defined time is correct.                */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_temporal_exp([], _, []) :- !.
check_temporal_exp(Exp, Table, NewExp) :-
	check_exp_type(Exp, [], [Table]+[], NewExp, Type, _, _),
	Type=interval,
	!.
check_temporal_exp(Exp, _, _) :-
	generate_error_flag(interval_type_expected, Exp). 


/****************************************************************************/
/* Mode:           check_del_compatibility(+Time-ValidExp, +TableTime)      */
/* Purpose:        Checks if a table is temporally consistent with a delete */
/*                 statement.                                               */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_del_compatibility(snapshot-[], snapshot) :- !.
check_del_compatibility(snapshot-[], system)   :- !.
check_del_compatibility(_, valid)              :- !.
check_del_compatibility(_, bitemporal)         :- !.
check_del_compatibility(_, TimeType)           :-
        generate_error_flag(temp_incompatible, TimeType).


/****************************************************************************/
/* Mode:           check_alter(+TimeType, +Action, +MetaInfo)               */
/* Purpose:        Checks if a table alteration is possible.                */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_alter(snapshot, add(valid), _)      :- !.
check_alter(system, add(valid), _)        :- !.
check_alter(snapshot, add(system), _)     :- !.
check_alter(valid, add(system), _)        :- !.
check_alter(_, add(Column-_-_), MetaInfo) :- 
        \+ member(_-Column-_-_-_, MetaInfo),
        generate_interval_col(Column, S, E),
        \+ member(_-S-_-_-_, MetaInfo), \+ member(_-E-_-_-_, MetaInfo), !.
check_alter(_, add(_), _) :- generate_error_flag(alter_column_exists, _).

check_alter(bitemporal, drop(valid), _)  :- !.
check_alter(valid, drop(valid), _)       :- !.
check_alter(bitemporal, drop(system), _) :- !.
check_alter(system, drop(system), _)     :- !.
check_alter(_, drop(Column), MetaInfo)   :- 
	member(Table-Column-_-_-_, MetaInfo), !,
	check_column_still_referenced(Table, Column).
check_alter(_, drop(Column), MetaInfo)   :-
        generate_interval_col(Column, S, E),
        member(_-S-_-_-_, MetaInfo), member(_-E-_-_-_, MetaInfo), !.
check_alter(_, drop(_), _) :- generate_error_flag(alter_column_not_exists, _).


/****************************************************************************/
/* Mode:           check_length(+TableType, +NewMetaInfo)                   */
/* Purpose:        Checks if a table alteration does not lead to a table    */
/*                 without columns.                                         */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_length(snapshot, MetaInfo)   :- length(MetaInfo, N), N > 0, !.
check_length(valid, MetaInfo)      :- length(MetaInfo, N), N > 2, !.
check_length(system, MetaInfo)     :- length(MetaInfo, N), N > 2, !.
check_length(bitemporal, MetaInfo) :- length(MetaInfo, N), N > 4, !.
check_length(_, _)                 :- generate_error_flag(alter_impossible, _).


/****************************************************************************/
/* Mode:           check_reference(+Type, +Refential_Integrity_Refs)        */
/* Purpose:        Checks if referenced column exists.                      */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_ref_time(_, snapshot, _, _)                :- !.
check_ref_time(_, valid, valid, valid)           :- !.
check_ref_time(_, valid, valid, bitemporal)      :- !.
check_ref_time(_, valid, bitemporal, valid)      :- !.
check_ref_time(_, valid, bitemporal, bitemporal) :- !.
check_ref_time(Table, _, _, _)                   :- 
	generate_error_flag(ri_incompatible, Table).

check_ref_type(Col, _, interval) :- !, generate_error_flag(reference_to_interval, Col).
check_ref_type(_, Type, Type)    :- !.
check_ref_type(_, s(_), char)    :- !.
check_ref_type(_, _, Type)       :- 
	generate_error_flag(referenced_col_incompatible, Type).

check_reference(TableTimeType, UserDefType, ref(Table-Col-ReferencingTimeType)) :-
	table_exists(Table), 
	!,
	get_table_type(Table, RefTableTimeType),
	check_ref_time(Table, ReferencingTimeType, TableTimeType, RefTableTimeType),
	get_table_infos(Table, [], MetaInfo),
	generate_interval_col(Col, S, E),
	((member(_-Col-Type-_-_, MetaInfo);
	  (member(_-S-Type-_-_, MetaInfo), member(_-E-Type-_-_, MetaInfo))) -> 
	         true;
	         generate_error_flag(column_not_exists, Table-Col)),
	check_ref_type(Col, UserDefType, Type).
check_reference(_, _, ref(Table-_-_)) :-
	generate_error_flag(table_not_exist1, Table). 	


/****************************************************************************/
/* Mode:           check_still_referenced(+Table)                           */
/* Purpose:        Checks if there are referential integrity constraints on */
/*                 columns of table.                                        */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_still_referenced(Table) :-
	get_ref_int(Table, RefCols),
	length(RefCols, 0),
	!.
check_still_referenced(Table) :-
	generate_error_flag(still_referenced_columns, Table).	

/****************************************************************************/
/* Mode:           check_column_still_referenced(+Table, +Column)           */
/* Purpose:        Checks if there are referential integrity constraints on */
/*                 column of table.                                         */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_column_still_referenced(Table, Col) :-
	get_ref_int(Table, RefCols),
	\+ member(Col, RefCols),
	!.
check_column_still_referenced(_, Col) :-
	generate_error_flag(column_still_referenced, Col).	

/****************************************************************************/
/* Mode:           check_column_names(+Columns)                             */
/* Purpose:        Checks if the column names of a create table are unique. */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_column_names([]) :- !.
check_column_names([C-_-_|Cols]) :-
	\+ member(C-_-_, Cols),
	!,
	check_column_names(Cols).
check_column_names([C-_-_|_]) :-
	generate_error_flag(duplicate_col_names, C).


/****************************************************************************/
/* Mode:           check_create_compatibility(+UserCol, +ResCol, +ResTypes, */
/*                                            -ColNames, -ColTypes)         */
/* Purpose:        Checks if CREATE-columns are consistent with the         */
/*                 the subquery.                                            */
/*                 Format ResTypes: ColName-Type-Null (all columns)         */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_named_expression(_-_)        :- !. 
check_named_expression(C)          :- atom(C), !. 
check_named_expression(expr(_, _)) :- !. 
check_named_expression(Exp)        :- 
        get_expression(Exp, ExpStr, []), atom_chars(Exp1, ExpStr),
        generate_error_flag(no_expression_allowed, Exp1).


check_unique_col_names([]) :- !.
check_unique_col_names([C|Cols]) :-
	\+ member(C, Cols),
	!,
	check_unique_col_names(Cols).
check_unique_col_names([C|_]) :-
	generate_error_flag(duplicate_col_names, C).


check_create_compatibility1([], [], [], []) :- !.
check_create_compatibility1(['vts_#$', 'vte_#$'|Cols],
                            ['vts_#$'-interval-N, 'vte_#$'-interval-N|ColTypes], 
                            ['vts_#$', 'vte_#$'|NewCols], 
                            ['vts_#$'-interval-N, 'vte_#$'-interval-N|NewTypes]) :-
        !,
        check_create_compatibility1(Cols, ColTypes, NewCols, NewTypes).
check_create_compatibility1([C|Cols], [S-interval-N, E-interval-N|ColTypes], 
                           [S1, E1|NewCols], [S1-interval-N, E1-interval-N|NewTypes]) :-
        get_interval_name(_, S, E),
        !,
        check_named_expression(C),
        generate_interval_col(C, S1, E1),
        check_create_compatibility1(Cols, ColTypes, NewCols, NewTypes).
check_create_compatibility1([C|Cols], [_-Type-Null|ColTypes], 
                           [C|NewCols], [C-Type-Null|NewTypes]) :-
        !,
        check_named_expression(C),
        check_create_compatibility1(Cols, ColTypes, NewCols, NewTypes).

check_create_compatibility([], ResCols, ResTypes, ResCols, ResTypes) :- 
	!, check_unique_col_names(ResCols).
check_create_compatibility(Cols, _, ResTypes, NewCols, NewTypes) :-
	check_unique_col_names(Cols),
        check_create_compatibility1(Cols, ResTypes, NewCols, NewTypes).
check_create_compatibility(_, _, _, _, _) :-
        generate_error_flag(columns_incompatible, _).

/****************************************************************************/
/* Mode:           check_update_columns(+Cols, +ResType, +MetaInfo, -Vars)  */
/* Purpose:        Checks if updated columns exist and if types of subquery */
/*                 and referenced columns correspond.                       */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_update_columns([], [], _, []) :- !.
check_update_columns([C|CL], [_-_-T-_|Res], MI, [var(_, T)|Vars]) :-
        member(_-C-T-_-_, MI), !, 
        check_update_columns(CL, Res, MI, Vars).
check_update_columns([Col|_], _, _, _) :- 
        generate_error_flag(invalid_update_column, Col).
check_update_columns([], [Col-_-_-_|_], _, _) :- 
        generate_error_flag(too_many_update_columns, Col).


/****************************************************************************/
/* Mode:           check_update_values(+Values, +MetaInfo, -Cols, -ValList) */
/* Purpose:        Checks if updated columns exist and if types correspond. */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_update_values([], _, [], []) :- !.
check_update_values([C-V-number|Values], MetaInfo, [C|Cols], [V|VL]) :-
        member(_-C-number-_-_, MetaInfo),
        !, 
        check_update_values(Values, MetaInfo, Cols, VL).
check_update_values([C-V-event|Values], MetaInfo, [C|Cols], [V|VL]) :-
        member(_-C-event-_-_, MetaInfo),
        !, 
        check_update_values(Values, MetaInfo, Cols, VL).
check_update_values([C-V-span|Values], MetaInfo, [C|Cols], [V|VL]) :-
        member(_-C-span-_-_, MetaInfo),
        !, 
        check_update_values(Values, MetaInfo, Cols, VL).
check_update_values([C-V-char|Values], MetaInfo, [C|Cols], [V|VL]) :-
        member(_-C-char-_-_, MetaInfo),
        !, 
        check_update_values(Values, MetaInfo, Cols, VL).
check_update_values(_, _, _, _) :- generate_error_flag(invalid_set_values, _).


/****************************************************************************/
/* Mode:           check_update_value(+Vars, +Query, +DropTabs, -ValList)   */
/* Purpose:        Checks if update-subquery returns only one tupel and     */
/*                 returns its values.                                      */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

get_values1([], []) :- !.
get_values1([var(V, number)|Vars], [cst(N, number)|Vals]) :- 
        !, number_chars(N, V), get_values1(Vars, Vals).
get_values1([var(V, s)|Vars], [cst(A, s)|Vals]) :- 
        !, atom_chars(A, V), get_values1(Vars, Vals).
get_values1([var(V, char)|Vars], [cst(V, char)|Vals]) :- 
        !, get_values1(Vars, Vals).

get_values(Query, Vars, Vals) :- 
        !, sql_sel([], Query, Vars),
        get_values1(Vars, Vals).

check_update_value(Vars, Query, _, ValList) :-
        setof(Vals, get_values(Query, Vars, Vals), [ValList]), !.
check_update_value(_, _, DropTabs, _) :-
        drop_aux_tables(DropTabs),
        generate_error_flag(single_value_expected, _).
        
/****************************************************************************/
/* Mode:           check_insert_query_compatibility(+MetaInfo, +QueryTypes) */
/* Purpose:        Checks if INSERT-columns are consistent with the         */
/*                 subquery.                                                */
/*                 Format QueryType: ColName-Type-Null (all columns)        */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_insert_query_compatibility([], []) :- !.
check_insert_query_compatibility([_-_-Type-_-_|MetaInfo], [_-Type-_|QueryTypes]) :-
        !,
        check_insert_query_compatibility(MetaInfo, QueryTypes).
check_insert_query_compatibility(_, _) :- generate_error_flag(columns_incompatible, _).


/****************************************************************************/
/* Mode:           check_insert_values_compatibility(+MetaInfo, +ValTypes)  */
/* Purpose:        Checks if INSERT-columns are consistent with the         */
/*                 values.                                                  */
/*                 Format ValType: AttrName-Type-Len (output-like)          */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_insert_values_compatibility([], []) :- !.
check_insert_values_compatibility([_-_-interval-_-_, _-_-interval-_-_|MetaInfo], 
                                  [_-interval-_|ValTypes]) :-
        !,
        check_insert_values_compatibility(MetaInfo, ValTypes).
check_insert_values_compatibility([_-_-Type-_-_|MetaInfo], [_-Type-_|ValTypes]) :-
        !,
        check_insert_values_compatibility(MetaInfo, ValTypes).
check_insert_values_compatibility(_, _) :- generate_error_flag(columns_incompatible, _).


/****************************************************************************/
/* Mode:           check_user_defined_valid(+Time, +UserDefTime)            */
/* Purpose:        Checks if user can define valid/system time or if query  */
/*                 itself contains this time already.                       */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_user_defined_valid(snapshot, valid)    :- !.
check_user_defined_valid(system, valid)      :- !.
check_user_defined_valid(_, valid)        :- 
        generate_error_flag(user_def_time, valid).


/****************************************************************************/
/* Mode:           check_single_occurence(+Attr, +From, +MetaInfo)          */
/* Purpose:        Checks if attribute is ambigous or not.                  */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_single_occurence(A, From, MetaInfo) :-
        findall(A, (member(T-A-_-_-_, MetaInfo), member(T-_-_, From)), [_]).

/****************************************************************************/
/* Mode:           check_union_compatibility(+ResType1, +ResType2)          */
/* Purpose:        Checks if two relations are union compatible.            */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_union_compatibility([], []) :- !.
check_union_compatibility([valid-interval-_|RT1], [valid-interval-_|RT2]) :-
        check_union_compatibility(RT1, RT2).
check_union_compatibility([system-interval-_|RT1], [system-interval-_|RT2]) :-
        check_union_compatibility(RT1, RT2).
check_union_compatibility([_-_-_-Type-_|RT1], [_-_-_-Type-_|RT2]) :-
        check_union_compatibility(RT1, RT2).
check_union_compatibility([_-_-Type-_|RT1], [_-_-Type-_|RT2]) :-
        check_union_compatibility(RT1, RT2).


/****************************************************************************/
/* Mode:           check_union_time_compatibility(+EvalTime, +SFWTime)      */
/* Purpose:        Checks if query and set-operation are temporally         */
/*                 compatible.                                              */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_union_time_compatibility(snapshot, _)       :- !.
check_union_time_compatibility(valid, valid)      :- !.
check_union_time_compatibility(valid, bitemporal) :- !.
check_union_time_compatibility(system, system)      :- !.
check_union_time_compatibility(system, bitemporal) :- !.
check_union_time_compatibility(validsystem, bitemporal) :- !.
check_union_time_compatibility(systemvalid, bitemporal) :- !.
        

/****************************************************************************/
/* Mode:           check_column_constraint_number(+ColName, +ConstraintType,*/
/*                                                +Constraints)             */
/* Purpose:        Checks if there is more than one column constraint of    */
/*                 type ConstType defined in ConstList.                     */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_column_constraint_number(ColName, ConstraintType, Constraints) :-
	findall(ConstraintType, member(ConstraintType, Constraints), List),
	length(List, L),
	L > 1,
	!,
	generate_error_flag(too_many_constraints, ColName).
check_column_constraint_number(_, _, _).


/*****************************************************************************/
/* Mode:           check_type_compatibility(+Op, +Type1, +Type2, -Type, -Len)*/
/* Purpose:        Checks if two expressions are type compatible.            */
/* Example:                                                                  */
/* Sideeffects:    None                                                      */
/* Call:           Exits always                                              */
/* Redo:           Fails always                                              */
/*****************************************************************************/

check_type_compatibility(neg, span, _, span, _) :- !.

check_type_compatibility('+', span, span, span, Length) :-
        !, get_column_length(span, Length).
check_type_compatibility('-', span, span, span, Length) :-
        !, get_column_length(span, Length).

check_type_compatibility('+', event, span, event, Length) :-
        !, get_column_length(event, Length).
check_type_compatibility('-', event, span, event, Length) :-
        !, get_column_length(event, Length).
check_type_compatibility('+', span, event, event, Length) :-
        !, get_column_length(event, Length).

check_type_compatibility('-', event, event, span, Length) :- 
        !, get_column_length(span, Length).
check_type_compatibility('*', span, number, span, Length) :-
        !, get_column_length(span, Length).
check_type_compatibility('*', number, span, span, Length) :-
        !, get_column_length(span, Length).
check_type_compatibility('/', span, number, span, Length) :-
        !, get_column_length(span, Length).

check_type_compatibility('+', span, interval, interval, Length) :-
        !, get_column_length(interval, Length).
check_type_compatibility('+', interval, span, interval, Length) :-
        !, get_column_length(interval, Length).
check_type_compatibility('-', interval, span, interval, Length) :-
        !, get_column_length(interval, Length).

check_type_compatibility('/', span, span, number, Length) :-
        !, get_column_length(number, Length).

check_type_compatibility(neg, number, _, number, _)           :- !.
check_type_compatibility('+', number, number, number, Length) :-
        !, get_column_length(number, Length).
check_type_compatibility('-', number, number, number, Length) :-
        !, get_column_length(number, Length).

check_type_compatibility('*', number, number, number, Length) :-
        !, get_column_length(number, Length).
check_type_compatibility('/', number, number, number, Length) :-
        !, get_column_length(number, Length).
check_type_compatibility(Op, Type1, Type2, _, _) :- 
        get_type(Type1, Str1), get_type(Type2, Str2), 
        generate_error_flag(incompatible_types, [Op, Str1, Str2]).


/*****************************************************************************/
/* Mode:           check_cond_type_compatibility(+Op, +Type1, +Type2)        */
/* Purpose:        Checks if two expressions are type compatible.            */
/* Example:                                                                  */
/* Sideeffects:    None                                                      */
/* Call:           Exits always                                              */
/* Redo:           Fails always                                              */
/*****************************************************************************/

check_cond_type_compatibility(precedes, event, event)       :- !.
check_cond_type_compatibility(precedes, event, interval)    :- !.
check_cond_type_compatibility(precedes, interval, event)    :- !.
check_cond_type_compatibility(precedes, interval, interval) :- !.
check_cond_type_compatibility(overlaps, event, interval)    :- !.
check_cond_type_compatibility(overlaps, interval, event)    :- !.
check_cond_type_compatibility(overlaps, interval, interval) :- !.
check_cond_type_compatibility(meets, interval, interval)    :- !.
check_cond_type_compatibility(contains, interval, interval) :- !.
check_cond_type_compatibility(eq, Type, Type)               :- !.
check_cond_type_compatibility(ls, number, number)           :- !.
check_cond_type_compatibility(ls, span, span)               :- !.
check_cond_type_compatibility(gr, number, number)           :- !.
check_cond_type_compatibility(gr, span, span)               :- !.
check_cond_type_compatibility(gq, number, number)           :- !.
check_cond_type_compatibility(gq, span, span)               :- !.
check_cond_type_compatibility(lq, number, number)           :- !.
check_cond_type_compatibility(lq, span, span)               :- !.
check_cond_type_compatibility(nq, number, number)           :- !.
check_cond_type_compatibility(nq, char, char)               :- !.
check_cond_type_compatibility(nq, span, span)               :- !.
check_cond_type_compatibility(Op, Type1, Type2)             :- 
        get_type(Type1, Str1), get_type(Type2, Str2), 
        generate_error_flag(incompatible_types_in_cond, [Op, Str1, Str2]).


/****************************************************************************/
/* Mode:           check_exp_type(+Exp, +MetaTable, +From,                  */
/*                                -NewExp -Type, -Length, -RefList)         */
/* Purpose:        Checks if expression is type compatible and if referenced*/
/*                 columns exist.                                           */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

check_interval_addition(span, Span, interval, interval(S, E), 
                        interval(add(Span, S), add(Span, E))) :- !.
check_interval_addition(interval, interval(S, E), span, Span,
                        interval(add(Span, S), add(Span, E))) :- !.
check_interval_addition(_, N1, _, N2, add(N1, N2))            :- !.

check_interval_subtraction(interval, interval(S, E), span, Span,
                           interval(sub(S, Span), sub(E, Span))) :- !.
check_interval_subtraction(_, N1, _, N2, sub(N1, N2))            :- !.


check_exp_type(neg(Exp), MetaTable, Tables, neg(NewExp), Type, Length, RefList) :-
        !,
        check_exp_type1(Exp, MetaTable, Tables, NewExp, Type1, Length, RefList),
        check_type_compatibility(neg, Type1, _, Type, _).
check_exp_type(Exp, MetaTable, Tables, NewExp, Type, Length, RefList) :- 
        !,
        check_exp_type1(Exp, MetaTable, Tables, NewExp, Type, Length, RefList).

check_exp_type1(add(Term, Exp), MetaTable, Tables, Add, Type, Len, RefList) :-
        !,
        check_term_type(Term, MetaTable, Tables, NT, Type1, _, RefList1),
        check_exp_type1(Exp, MetaTable, Tables, NE, Type2, _, RefList2),
        append(RefList1, RefList2, RefList),
        check_type_compatibility('+', Type1, Type2, Type, Len),
        check_interval_addition(Type1, NT, Type2, NE, Add). 
check_exp_type1(sub(Term, Exp), MetaTable, Tables, Sub, Type, Len, RefList) :-
        !,
        check_term_type(Term, MetaTable, Tables, NT, Type1, _, RefList1),
        check_exp_type1(Exp, MetaTable, Tables, NE, Type2, _, RefList2),
        append(RefList1, RefList2, RefList),
        check_type_compatibility('-', Type1, Type2, Type, Len),
        check_interval_subtraction(Type1, NT, Type2, NE, Sub). 
check_exp_type1(Term, MetaTable, Tables, NewTerm, Type, Len, RefList) :-
        !,
        check_term_type(Term, MetaTable, Tables, NewTerm, Type, Len, RefList).
        
check_term_type(quo(Fact, Term), MetaTable, Tabs, quo(NF, NT), Type, Len, RefList) :-
        !,      
        check_faktor_type(Fact, MetaTable, Tabs, NF, Type1, _, RefList1),
        check_term_type(Term, MetaTable, Tabs, NT, Type2, _, RefList2),
        append(RefList1, RefList2, RefList),
        check_type_compatibility('/', Type1, Type2, Type, Len).
check_term_type(mul(Fact, Term), MetaTable, Tabs, mul(NF, NT), Type, Len, RefList) :-
        !,
        check_faktor_type(Fact, MetaTable, Tabs, NF, Type1, _, RefList1),
        check_term_type(Term, MetaTable, Tabs, NT, Type2, _, RefList2),
        append(RefList1, RefList2, RefList),
        check_type_compatibility('*', Type1, Type2, Type, Len).
check_term_type(Fact, MetaTable, Tabs, NewFact, Type, Len, RefList) :-  
        !,
        check_faktor_type(Fact, MetaTable, Tabs, NewFact, Type, Len, RefList).

% Constant
check_faktor_type(cst(C, interval), _, _, cst(C, interval), event, Len, []) :- 
        !,
        get_column_length(event, Len).
check_faktor_type(cst(C, Type), _, _, cst(C, Type), Type, Len, []) :- 
        !,
        get_column_length(Type, Len).

% Attribute
check_faktor_type(A, MetaTable, F+JT, interval(R-S, R-E), interval, Len,
                  [Al-S1-interval-Len-ID=R-S, Al-E1-interval-Len-ID=R-E]) :-
        atom(A),
        generate_interval_col(A, S1, E1),
        append(F, JT, Tables),
        check_single_occurence(S1, Tables, MetaTable),
        check_single_occurence(E1, Tables, MetaTable), 
	member(T-S1-interval-Len-ID, MetaTable),
        !,
        get_table_alias(T, Tables, _, Al),
        member(T-E1-interval-Len-ID, MetaTable).
check_faktor_type(A, MetaTable, F+JT, R-C, Type, Len, [Al-A-Type-Len-ID=R-C]) :-
        atom(A),
        append(F, JT, Tables),
        check_single_occurence(A, Tables, MetaTable),
        member(T-A-Type-Len-ID, MetaTable),
        !,
        get_table_alias(T, Tables, _, Al).
check_faktor_type(A, _, _, _, _, _, _) :-
        atom(A),
        !,
        generate_error_flag(invalid_column, []-A).

% Table-Attribute
check_faktor_type(T-A, MetaTable, F+JT, interval(R-S, R-E), interval, Len,
                  [Al-S1-interval-Len-ID=R-S, Al-E1-interval-Len-ID=R-E]) :-
        append(F, JT, Tables),
        get_table_alias(T, Tables, Tab, Al),
        generate_interval_col(A, S1, E1),
        member(Tab-S1-interval-Len-ID, MetaTable),
        !,
        member(Tab-E1-interval-Len-ID, MetaTable).
check_faktor_type(T-A, MetaTable, F+JT, R-C, Type, Len, [Al-A-Type-Len-ID=R-C]) :-
        append(F, JT, Tables),
        get_table_alias(T, Tables, Tab, Al),
        member(Tab-A-Type-Len-ID, MetaTable),
        !.
check_faktor_type(T-A, _, _, _, _, _, _) :-
        !,
        generate_error_flag(invalid_column, T-A).

% Functional references
check_faktor_type(vtime(R), MetaTable, F+JT, interval(A-S, A-E), Type, Len,
                  [Al-'vts_#$'-Type-Len-ID=A-S, Al-'vte_#$'-Type-Len-ID=A-E]) :-
        append(F, JT, Tables),
        get_table_alias(R, Tables, Tab, Al),
        member(Tab-'vts_#$'-Type-Len-ID, MetaTable),
        member(Tab-'vte_#$'-Type-Len-ID, MetaTable),
        !.
check_faktor_type(vtime(R), _, _, _, _, _, _) :-
        !, generate_error_flag(table_not_exist, R).

check_faktor_type(valid(R), MetaTable, F+JT, interval(A-S, A-E), Type, Len,
                  [Al-'vts_#$'-Type-Len-ID=A-S, Al-'vte_#$'-Type-Len-ID=A-E]) :-
        append(F, JT, Tables),
        get_table_alias(R, Tables, Tab, Al),
        member(Tab-'vts_#$'-Type-Len-ID, MetaTable),
        member(Tab-'vte_#$'-Type-Len-ID, MetaTable),
        !.
check_faktor_type(valid(R), _, _, _, _, _, _) :-
        !, generate_error_flag(table_not_exist, R).

check_faktor_type(xtime(R), MetaTable, F+JT, interval(abs(A-S), abs(A-E)), Type, Len, 
                  [Al-'sts_#$'-Type-Len-ID=A-S, Al-'ste_#$'-Type-Len-ID=A-E]) :-
        append(F, JT, Tables),
        get_table_alias(R, Tables, Tab, Al),
        member(Tab-'sts_#$'-Type-Len-ID, MetaTable),
        member(Tab-'ste_#$'-Type-Len-ID, MetaTable),
        !.
check_faktor_type(xtime(R), _, _, _, _, _, _) :-
        !, generate_error_flag(table_not_exist, R).

check_faktor_type(transaction(R), MetaTable, F+JT, interval(abs(A-S), abs(A-E)), Type, 
                  Len, [Al-'sts_#$'-Type-Len-ID=A-S, Al-'ste_#$'-Type-Len-ID=A-E]) :-
        append(F, JT, Tables),
        get_table_alias(R, Tables, Tab, Al),
        member(Tab-'sts_#$'-Type-Len-ID, MetaTable),
        member(Tab-'ste_#$'-Type-Len-ID, MetaTable),
        !.
check_faktor_type(transaction(R), _, _, _, _, _, _) :-
        !, generate_error_flag(table_not_exist, R).

check_faktor_type(interval(S, E), MetaTable, F+JT, 
                  interval(NS, NE), interval, Len, RefList) :-
        check_exp_type(S, MetaTable, F+JT, NS, Type1, _, RefList1),
        Type1 == event,
        check_exp_type(E, MetaTable, F+JT, NE, Type2, _, RefList2),
        Type2 == event,
        !,
        append(RefList1, RefList2, RefList),
        get_column_length(interval, Len).
check_faktor_type(interval(S, E), _, _, _, _, _, _) :-
        !, generate_error_flag(event_type_expected, interval(S, E)).

check_faktor_type(begin(Exp), MetaTable, F+JT, S, event, Len, RefList) :-
        check_exp_type(Exp, MetaTable, F+JT, interval(S, _), Type, _, RefList),
        Type == interval,
        !,
        get_column_length(event, Len).
check_faktor_type(begin(Exp), _, _, _, _, _, _) :-
        !, generate_error_flag(interval_type_expected, begin(Exp)).

check_faktor_type(end(Exp), MetaTable, F+JT, E, event, Len, RefList) :-
        check_exp_type(Exp, MetaTable, F+JT, interval(_, E), Type, _, RefList),
        Type == interval,
        !,
        get_column_length(event, Len).
check_faktor_type(end(Exp),  _, _, _, _, _, _) :-
        !, generate_error_flag(interval_type_expected, end(Exp)).

check_faktor_type(first(E1, E2), Meta, F+JT, lst([NE1, NE2]), event, Len, RefList) :-
        check_exp_type(E1, Meta, F+JT, NE1, Type1, _, RefList1),
        Type1 == event,
        check_exp_type(E2, Meta, F+JT, NE2, Type2, _, RefList2),
        Type2 == event,
        !,
        get_column_length(event, Len),
        append(RefList1, RefList2, RefList).
check_faktor_type(first(E1, E2),  _, _, _, _, _, _) :-
        !, generate_error_flag(event_type_expected, first(E1, E2)).

check_faktor_type(last(E1, E2), Meta, F+JT, gst([NE1, NE2]), event, Len, RefList) :-
        check_exp_type(E1, Meta, F+JT, NE1, Type1, _, RefList1),
        Type1 == event,
        check_exp_type(E2, Meta, F+JT, NE2, Type2, _, RefList2),
        Type2 == event,
        !,
        get_column_length(event, Len),
        append(RefList1, RefList2, RefList).
check_faktor_type(last(E1, E2),  _, _, _, _, _, _) :-
        !, generate_error_flag(event_type_expected, last(E1, E2)).

check_faktor_type(span(Int), MetaTable, F+JT, sub(E, S), span, Len, Refs) :-
        check_exp_type(Int, MetaTable, F+JT, interval(S, E), Type, _, Refs),
        Type == interval,
        !,
        get_column_length(event, Len).
check_faktor_type(span(Int),  _, _, _, _, _, _) :-
        !, generate_error_flag(interval_type_expected, span(Int)).

check_faktor_type(lst([E1, E2]), MetaTab, F+JT, lst([NE1, NE2]), number, Len, Refs) :-
        check_exp_type(E1, MetaTab, F+JT, NE1, Type1, Len, RefList1),
        Type1 == number,
        check_exp_type(E2, MetaTab, F+JT, NE2, Type2, Len, RefList2),
        Type2 == number,
        !,
        append(RefList1, RefList2, Refs).
check_faktor_type(gst([E1, E2]), MetaTab, F+JT, gst([NE1, NE2]), number, Len, Refs) :-
        check_exp_type(E1, MetaTab, F+JT, NE1, Type1, Len, RefList1),
        Type1 == number,
        check_exp_type(E2, MetaTab, F+JT, NE2, Type2, Len, RefList2),
        Type2 == number,
        append(RefList1, RefList2, Refs).

check_faktor_type(intersect(I1, I2), MetaTab, F+JT, 
                  interval(gst([S1, S2]), lst([E1, E2])), interval, Len, Refs) :-
        check_exp_type(I1, MetaTab, F+JT, Int1, Type1, Len, Refs1),
        Type1 == interval, Int1 = interval(S1, E1),
        check_exp_type(I2, MetaTab, F+JT, Int2, Type2, Len, Refs2),
        Type2 == interval, Int2 = interval(S2, E2),
        append(Refs1, Refs2, Refs).
check_faktor_type(intersect(I1, I2),  _, _, _, _, _, _) :-
        !, generate_error_flag(interval_type_expected, intersect(I1, I2)).

check_faktor_type(abs(N), MetaTable, F+JT, abs(NN), Type, Length, Refs) :-
        check_exp_type(N, MetaTable, F+JT, NN, Type, Length, Refs),
        Type == number,
        !.
check_faktor_type(absolute(Span), MetaTable, F+JT, abs(NS), Type, Length, Refs) :-
        check_exp_type(Span, MetaTable, F+JT, NS, Type, Length, Refs),
        Type == span,
        !.
check_faktor_type(absolute(Span),  _, _, _, _, _, _) :-
        !, generate_error_flag(span_type_expected, absolute(Span)).

check_faktor_type(Faktor, MetaTable, F+JT, NewFaktor, Type, Length, Refs) :- 
        !,
        check_exp_type1(Faktor, MetaTable, F+JT, NewFaktor, Type, Length, Refs).

