/******************************************************************************/
/* Module:      constraints.pl                                                */
/* Project:     TimeDB 1.03                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Date:        December 12, 1995                                             */
/* Export:                                                                    */
/*              check_primary_key/1 (evaluate.pl)                             */
/*              check_referential_integrity/0 (evaluate.pl)                   */
/*              check_assertions/0 (evaluate.pl)                              */
/*              check_table_column_constraints/0 (evaluate.pl)                */
/* Import:                                                                    */
/*              remove_counter/0 (basics.pl)                                  */
/*              drop_aux_tables/1 (basics.pl)                                 */
/*              translate_query/5 (translate.pl)                              */
/*              translate/3 (translate.pl)                                    */
/*              create_derived_tables/2 (evaluate.pl)                         */
/*              evaluate/4 (evaluate.pl)                                      */
/*              get_parameters/4 (display.pl)                                 */
/*              sql_sel/3 (meta.pl)                                           */
/*              get_table_type/2 (meta_data.pl)                               */
/*              get_referenced_table/5 (meta_data.pl)                         */
/*              get_assertion/3 (meta_data.pl)                                */
/*              get_key/2 (meta_data.pl)                                      */
/*              get_table_column_constraint/2 (meta_data.pl)                  */
/******************************************************************************/

/*************************************************************************************/
/******************************** Primary Key ****************************************/
/*************************************************************************************/

%% generate_condition(+AttrList, -Where)
generate_condition([], []) :- !.
generate_condition([Attr|Rest], [rel(eq, a1-Attr, a2-Attr)|Conds]) :-
	generate_condition(Rest, Conds).


%% multiple_occurence(+Now, +Table, +AttrList, +Type)
multiple_occurence(_, _, [], _) :- !, fail.
multiple_occurence(_, Table, AttrList, snapshot) :-
	!,
	sql_sel([],
		[sel([count(all)]),
		 from([Table-a1]),
		 where([])],
		[var(C1, n(i))]),
	generate_condition(AttrList, Where), % nat. join on key-attributes
	sql_sel([],
		[sel([count(all)]),
		 from([Table-a1, Table-a2]),
		 where([and(Where)])],
		[var(C2, n(i))]),
	C1 < C2, % nat. self-join produces more tuples 
	nl, nl, 
	write('Error(s) found checking primary key of table * '),
	write(Table), write(' *'), nl, 
	!. 
multiple_occurence(Now, Table, AttrList, system) :-
	!,
	sql_sel([],
		[sel([count(all)]),
		 from([Table-a1]),
		 where([rel(gr, abs(a1-'ste_#$'), cst(Now, n(_)))])],
		[var(C1, n(i))]),
	generate_condition(AttrList, Where), % nat. join on key-attributes
	sql_sel([],
		[sel([count(all)]),
		 from([Table-a1, Table-a2]),
		 where([and([rel(gr, abs(a1-'ste_#$'), cst(Now, n(_))),
	                     rel(gr, abs(a2-'ste_#$'), cst(Now, n(_))) |Where])])],
		[var(C2, n(i))]),
	C1 < C2, % nat. self-join produces more tuples 
	nl, nl, 
	write('Error(s) found checking primary key of table * '),
	write(Table), write(' *'), nl, 
	!. 
multiple_occurence(_, Table, AttrList, valid) :-
	!,
	sql_sel([],
		[sel([count(all)]),
		 from([Table-a1]),
		 where([])],
		[var(C1, n(i))]),
	generate_condition(AttrList, Where), % nat. join on key-attributes
	sql_sel([],
		[sel([count(all)]),
		 from([Table-a1, Table-a2]),
		 where([and([rel(ls, a1-'vts_#$', a2-'vte_#$'),
	                     rel(gr, a1-'vte_#$', a2-'vts_#$') |Where])])],
		[var(C2, n(i))]),
	C1 < C2, % nat. self-join produces more tuples
	nl, nl, 
	write('Error(s) found checking primary key of table * '), 
	write(Table), write(' *'), nl, 
	!.
multiple_occurence(Now, Table, AttrList, bitemporal) :-
	!,
	sql_sel([],
		[sel([count(all)]),
		 from([Table-a1]),
		 where([rel(gr, abs(a1-'ste_#$'), cst(Now, n(_)))])],
		[var(C1, n(i))]),
	generate_condition(AttrList, Where), % nat. join on key-attributes
	sql_sel([],
		[sel([count(all)]),
		 from([Table-a1, Table-a2]),
		 where([and([rel(gr, abs(a1-'ste_#$'), cst(Now, n(_))),
	                     rel(gr, abs(a2-'ste_#$'), cst(Now, n(_))),
	                     rel(ls, a1-'vts_#$', a2-'vte_#$'),
	                     rel(gr, a1-'vte_#$', a2-'vts_#$') |Where])])],
		[var(C2, n(i))]),
	C1 < C2, % nat. self-join produces more tuples
	nl, nl, 
	write('Error(s) found checking primary key of table * '), 
	write(Table), write(' *'), nl, 
	!.


check_primary_key(Now) :-
	(update_table(Table); check_table(Table)),
	get_key(Table, AttrList),
	get_table_type(Table, Type),
	multiple_occurence(Now, Table, AttrList, Type),
	!, fail.
check_primary_key(_).


/*************************************************************************************/
/*************************** Referential Integrity ***********************************/
/*************************************************************************************/

%% show_constraint(+TimeType, +Table, +Col, +RefTable, +RefCol)
show_constraint(_, Table, Col, RefTable, RefCol) :-
	nl, nl, 
	write('Error(s) found checking referential integrity * '),
	write(Table), write('.'), write(Col), write(' references '),
	write(RefTable), write('.'), write(RefCol), write(' *'), nl, 
	!.
/*
show_constraint(valid, Table, Col, RefTable, RefCol) :-
	nl, nl, 
	write('Error(s) found checking referential integrity * '),
	write(Table), write('.'), write(Col), write(' valid references '),
	write(RefTable), write('.'), write(RefCol), write(' *'), nl, 
	!.
*/


%% get_result(+ResType, +ResultQuery, -N)
get_result(ResType, ResultQuery, N) :- 
	get_parameters(ResType, _, OutputVars, _),
	findall(OutputVars, sql_sel([], ResultQuery, OutputVars), OutputList),
	length(OutputList, N), 
	!.


%% do_query(+TableType, +Table, +Col, +RefTable, +RefCol, -N)
do_query(snapshot, Table, Col, RefTable, RefCol, N) :-
	!,
	translate(query(sfw([snapshot([all]),
	                     from([Table-a0]),
                             where(not(exists(
                                query(sfw([snapshot([all]),
                                           from([RefTable-a1]),
                                           where(rel(eq, a0-Col, a1-RefCol))],
                        tuc-nored-snapshot)))))],tuc-nored-snapshot)), 
                  DerivedTables, SetOpStruct),
	create_derived_tables(DerivedTables, DropTables1),
	evaluate(SetOpStruct, ResType, ResultQuery, DropTables2),
	get_result(ResType, ResultQuery, N),
	drop_aux_tables(DropTables1),
	drop_aux_tables(DropTables2),
	remove_counter.
do_query(valid, Table, Col, RefTable, RefCol, N) :-
	!,
	translate(query(sfw([valid([all]),
	                     from([Table-a0]),
                             where(not(exists(
                                query(sfw([valid([all]),
                                           from([RefTable-a1]),
                                           where(rel(eq, a0-Col, a1-RefCol))],
                        tt-nored-valid)))))],tt-nored-valid)), 
                  DerivedTables, SetOpStruct),
	create_derived_tables(DerivedTables, DropTables1),
	evaluate(SetOpStruct, ResType, ResultQuery, DropTables2),
	get_result(ResType, ResultQuery, N),
	drop_aux_tables(DropTables1),
	drop_aux_tables(DropTables2),
	remove_counter.


do_check(TT, Origin, Col, RefTable, RefCol) :-
	do_query(TT, Origin, Col, RefTable, RefCol, N),
	N > 0,
	!,
	show_constraint(TT, Origin, Col, RefTable, RefCol).

check_referential_integrity :-
	get_referenced_table(Origin, Col, RefTable, RefCol, TT),
	do_check(TT, Origin, Col, RefTable, RefCol),
	!, fail.
check_referential_integrity.



/*************************************************************************************/
/********************************* ASSERTIONS ****************************************/
/*************************************************************************************/

check_result(exists, N)    :- N = 0, !.
check_result(notexists, N) :- N > 0.


do_assertion(Query, N) :-
	translate_query(Query, [], [], DerivedTables, SetOpStruct),		
	create_derived_tables(DerivedTables, DropTables1),
	evaluate(SetOpStruct, ResType, ResultQuery, DropTables2),
	get_result(ResType, ResultQuery, N),
	drop_aux_tables(DropTables1),
	drop_aux_tables(DropTables2),
	remove_counter, 
	!.

check_assertions :-
	get_assertion(Name, Query, Type),
	do_assertion(Query, N),
	check_result(Type, N),
	!,
	nl,
	write('Error(s) found checking assertion * '), write(Name), write(' *'), nl,
	fail. 
check_assertions.		



/*************************************************************************************/
/************************ TABLE / COLUMN CONSTRAINTS *********************************/
/*************************************************************************************/

check_table_column_constraints :-
	get_table_column_constraint(TableName, Query),
	do_assertion(Query, N),
	check_result(notexists, N),
	!,
	nl,
	write('Error(s) found checking table-/column-constraint of table * '), 
	write(TableName), write(' *'), nl,
	fail. 
check_table_column_constraints.
