/******************************************************************************/
/* Module:      decimal_calendar.pl                                           */
/* Project:     TimeDB 1.03                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Date:        December 12, 1995                                             */
/* Export:                                                                    */
/*              decimal_granularity/1 (calendar.pl)                           */
/*              decimal_get_inc/2 (calendar.pl)                               */
/*              decimal_to_internal/3 (calendar.pl)                           */
/*              internal_to_decimal/2 (calendar.pl)                           */
/*              decimal_span_to_internal/4 (calendar.pl)                      */
/*              internal_to_decimal_span/2 (calendar.pl)                      */
/*              decimal_min_number/1 (calendar.pl)                            */
/*              decimal_max_number/1 (calendar.pl)                            */
/* Import:                                                                    */
/*              get_tt_date/2 (time.pl)                                       */
/******************************************************************************/

decimal_granularity(year).
decimal_get_inc(year, 1).

/********************************************************************************/
/* Mode:	decimal_to_internal(-Years)		                        */
/* Purpose:	Parses a date-string and converts it to an integer representing */
/*              amount of years gone by since 0                                 */
/* Example:								        */
/* Sideeffects:	None								*/
/* Call:        Exists always							*/
/* Redo:        Fails always							*/
/********************************************************************************/

decimal_const(Now)       --> [now], !, {get_tt_date(now, Now)}. 
decimal_const(Now)       --> [present], {get_tt_date(now, Now)}. 
decimal_const(BEGINNING) --> [beginning], {decimal_min_number(BEGINNING), !}. 
decimal_const(FOREVER)   --> [forever], {decimal_max_number(FOREVER), !}. 

decimal_from_year(Year)      --> [num(Year)], !.


decimal_to_internal(Years) --> 
	decimal_const(Years), !.
decimal_to_internal(Years) --> 
	decimal_from_year(Years),
	!,
 	{decimal_max_number(MaxYears), Years < MaxYears}.


/**************************************************************************************/
/* Mode:	internal_to_decimal(+Internal, -Date_Literal)			      */
/* Purpose:	Converts an integer representing amount of years (Internal) gone by   */
/*              since 0 into a date-string                                            */
/* Example:									      */
/* Sideeffects:	None								      */
/* Call:        Exists always							      */
/* Redo:        Fails always							      */
/**************************************************************************************/

internal_to_decimal(Internal, "forever") :-
        number_chars(I, Internal), 
        decimal_max_number(I1),
	I >= I1,
        !.
internal_to_decimal(Internal, "beginning") :-
        number_chars(I, Internal), 
        decimal_min_number(I1),
	I =< I1,
        !.
internal_to_decimal(Internal, Internal) :-
	!.


/**************************************************************************************/
/* Mode:	decimal_span_to_internal(-Seconds)				      */
/* Purpose:	Parses a duration-string and converts it to an integer representing   */
/*              amount of years                                                       */
/* Example:									      */
/* Sideeffects:	None								      */
/* Call:        Exists always							      */
/* Redo:        Fails always							      */
/**************************************************************************************/

decimal_span_to_internal(year, Chronons) --> [num(Chronons)].


/**************************************************************************************/
/* Mode:	internal_to_decimal_span(+Span, -SpanStr)       		      */
/* Purpose:	Converts an integer representing duration of seconds (Internal)       */
/* Example:									      */
/* Sideeffects:	None								      */
/* Call:        Exists always							      */
/* Redo:        Fails always							      */
/**************************************************************************************/

internal_to_decimal_span(Years, SpanStr) :- 
	!, 
	number_chars(Years, YearsStr),
	append(YearsStr, " year", SpanStr).


/**************************************************************************************/
/* Constants used in translate.pl (-inf, +inf)                                        */
/**************************************************************************************/

decimal_min_number(0).
decimal_max_number(1000000000000).

