/******************************************************************************/
/* Module:      meta_data.pl                                                  */
/* Project:     TimeDB 1.03                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Date:        December 12, 1995                                             */
/* Export:                                                                    */
/*              get_table_type/2 (check.pl,translate.pl,constraints.pl)       */
/*              get_table_infos/3 (check.pl,translate.pl)                     */
/*              table_exists/1 (check.pl,translate.pl)                        */
/*              view_exists/1 (translate.pl)                                  */
/*              add_meta_data/3 (evaluate.pl)                                 */
/*              delete_meta_data/1 (evaluate.pl)                              */
/*              add_view_meta_data/5 (evaluate.pl)                            */
/*              get_view_meta_data/3 (translate.pl)                           */
/*              delete_view_meta_data/1 (evaluate.pl)                         */
/*              insert_ref_int/5 (evaluate.pl)                                */
/*              delete_ref_int/1 (evaluate.pl)                                */
/*              get_ref_int/2 (check.pl)                                      */
/*              get_referenced_table/5 (constraints.pl)                       */
/*              add_assertion/3 (evaluate.pl)                                 */
/*              get_assertion/3 (constraints.pl)                              */
/*              delete_assertion/1 (evaluate.pl)                              */
/*              add_key/2 (evaluate.pl)                                       */
/*              get_key/2 (constraints.pl)                                    */
/*              delete_key/1 (evaluate.pl)                                    */
/*              delete_key_attributes/2 (evaluate.pl)                         */
/*              add_table_column_constraints/2 (evaluate.pl)                  */
/*              get_table_column_constraint/2 (constraints.pl)                */
/*              delete_table_column_constraints/1 (evaluate.pl)               */
/* Import:                                                                    */
/*              read_from_chars/2 (swi.pl)                                    */
/*              format_to_chars/3 (swi.pl)                                    */
/*              upper_case/2 (scanner.pl)                                     */
/*              lower_case/2 (scanner.pl)                                     */
/*              generate_error_flag/2 (errors.pl)                             */
/*              get_column_length/2 (check.pl)                                */
/*              get_type/2 (check.pl)                                         */
/*              sql_exec/4 (meta.pl)                                          */
/*              sql_sel/3 (meta.pl)                                           */
/******************************************************************************/

adjust_string(String, OutString) :- atom(String), !, 
	atom_chars(String, String1), adjust_string(String1, OutString).
adjust_string(String, OutString) :- oracle(on), !, upper_case(String, OutString).
adjust_string(String, String).


double_quotes([], []) :- !.
double_quotes([0''|Rest1], [0'', 0''|Rest2]) :- !, double_quotes(Rest1, Rest2).
double_quotes([C|Rest1], [C|Rest2]) :- double_quotes(Rest1, Rest2).

single_quotes([], []) :- !.
single_quotes([0'', 0''|Rest1], [0''|Rest2]) :- !, single_quotes(Rest1, Rest2).
single_quotes([C|Rest1], [C|Rest2]) :- single_quotes(Rest1, Rest2).


/****************************************************************************/
/* Mode:	insert_table_type(+TableName, +TableType)                   */
/* Purpose:	Adds TYPE-info of new table to meta-table table_types       */
/****************************************************************************/

insert_table_type(TableName, TableType) :-
	!,
	atom_chars(TableName, NameStr),
	atom_chars(TableType, TypeStr),
	sql_exec([],
	   insert(table_types, 
                  [cst(NameStr, s), cst(TypeStr, s)]), 
           _, _).


/****************************************************************************/
/* Mode:	delete_table_type(+TableName)                               */
/* Purpose:	Deletes TYPE-info of table TableName in table_types.        */
/****************************************************************************/

delete_table_type(TableName) :-
	atom_chars(TableName, Name1),
	adjust_string(Name1, Name),
	sql_exec([],
	         delete(table_types-"a1",
	         where([rel(eq, "a1"-"table_name", cst(Name, s))])),
	         _, _).


/****************************************************************************/
/* Mode:	get_table_type(+TableName, -TableType)                      */
/* Purpose:	Gets TYPE-info of table from meta-table table_types.        */
/****************************************************************************/

get_table_type(TableName, TableType) :-
	atom_chars(TableName, Name1),
	adjust_string(Name1, Name),
	sql_sel([],
		[sel([table_type-noref]),
		 from([table_types]),
		 where([and([rel(eq, table_name-noref, cst(Name, s))])])],
		[var(Type, s)]),
	lower_case(Type, Type1),
	atom_chars(TableType, Type1).

	
/****************************************************************************/
/* Mode:	create_emulator_table_types                                 */
/* Purpose:	Creates meta-table table_types for emulator.                */
/****************************************************************************/

create_emulator_table_types :-
	!,
	sql_exec([],
	  create(table_types,
	     ["table_name", "table_type"],
	     [seld(["a1"-"table_name", cst("snapshot", s(10))]),
             from(["user_tab_columns"-"a1"])]),
	_, _),
	nl, 
	write('*************************************'), nl,
	write('New metatable <<table_types>> created'), nl,
	write('*************************************'), nl.


/****************************************************************************/
/* Mode:	get_table_infos(+LoadTables, +OldMetaInfo, -NewMetaInfo)    */
/* Purpose:	Holt die Informationen aus 'table_view_schemes'	            */
/****************************************************************************/

sort_columns([], []) :- !.
sort_columns([T-'vte_#$'-interval-L-ID, T-'vts_#$'-interval-L-ID|MetaInfo], 
	     [T-'vts_#$'-interval-L-ID, T-'vte_#$'-interval-L-ID|NewMetaInfo]) :-
	!,
	sort_columns(MetaInfo, NewMetaInfo).
sort_columns([T-'ste_#$'-interval-L-ID, T-'sts_#$'-interval-L-ID|MetaInfo],
	     [T-'sts_#$'-interval-L-ID, T-'ste_#$'-interval-L-ID|NewMetaInfo]) :-
	!,
	sort_columns(MetaInfo, NewMetaInfo).
sort_columns([Col|MetaInfo], [Col|NewMetaInfo]) :- 
	sort_columns(MetaInfo, NewMetaInfo).
	

get_table_infos(Table, MetaInfo, MetaInfo) :- 
	member(Table-_-_-_-_, MetaInfo),
	!.
get_table_infos(Table1, MetaInfo, NewMetaInfo) :-
	adjust_string(Table1, TableStr), 
	findall(Table-Column-Type-Length-ID,
	     get_table_infos1(TableStr, Table, Column, Type, Length, ID),
	     MetaInfo1),
	MetaInfo1 \== [],
	!,
	sort_columns(MetaInfo1, MetaInfo2),
	append(MetaInfo2, MetaInfo, NewMetaInfo).
get_table_infos(Table, _, _) :-
	generate_error_flag(table_not_exist1, Table).

%% number_type(+Type, -New_Type)
number_type("num", "number") :- !.
number_type(Type, Type).

%% get_table_infos1(+Table, -Table, -Column, -DataType, -DataLen, -ID)
get_table_infos1(Table, TableName, Column, DataType, Length, ID) :-
	sql_sel([],
		[sel([table_name-noref, column_name-noref, 
                      data_type-noref, column_length-noref, column_id-noref]),
		 from([table_view_schemes]),
		 where([and([rel(eq, table_name-noref, cst(Table, s))])]),
	         group([]), having([]), 
	         order([column_id-noref])],
		[var(T, s), var(C, s), var(D, s), var(L, s), var(ID2, s)]),
	lower_case(T, T1), lower_case(C, C1), lower_case(D, D1), 
	lower_case(L, L1), lower_case(ID2, ID1),
	atom_chars(TableName, T1), atom_chars(Column, C1),
	number_type(D1, D2), atom_chars(DataType, D2),
	number_chars(Length, L1),
	number_chars(ID, ID1).	

/****************************************************************************/
/* Mode:	table_exists(+Name)     	                            */
/* Purpose:	Checks in meta-table table_view_schemes if a user-          */
/*              defined table Name exists.                                  */
/****************************************************************************/

table_exists(Name1) :-
	adjust_string(Name1, Name),
	sql_sel([],
	         [sel([cst(1,(n(i)))]),
	          from(["table_view_schemes"-"a1"]),
		  where([rel(eq, "a1"-"table_name", cst(Name, s))])],
		  [var(_,n(i))]),
	!.

/****************************************************************************/
/* Mode:	view_exists(+Name)       	                            */
/* Purpose:	Checks in meta-table views if a view Name exists.           */
/****************************************************************************/

view_exists(Name1) :-
	adjust_string(Name1, Name),
	sql_sel([],
	         [sel([cst(1,(n(i)))]),
	          from(["views"-"a1"]),
	          where([rel(eq, "a1"-"name", cst(Name, s))])],
	         [var(_,n(i))]),
	!.

/****************************************************************************/
/* Mode:	insert_table(+Name, +ID, +ColumnList)                       */
/* Purpose:	Adds meta-info into table_view_schemes.                     */
/****************************************************************************/

expression_to_column(_-Column, Column) :- !.
expression_to_column(Column, Column) :- atom(Column), !.


insert_table(_, _, []) :- !.
insert_table(Name, ID, [Col1-interval-Null, Col2-interval-Null|Col_Type_List]) :-
	!,
	atom_chars(Col1, ColStr1),
	atom_chars(Col2, ColStr2),
	atom_chars(Null, NullStr),
	get_column_length(interval, Length),
	NewID is ID + 1, 
	sql_exec([],
	   insert(table_view_schemes, 
                  [cst(Name, s), cst(ColStr1, s), cst("interval", s), 
                   cst(NullStr, s), cst(Length, n(i)), cst(NewID, n(i))]), 
           _, _),
	sql_exec([],
	   insert(table_view_schemes, 
                  [cst(Name, s), cst(ColStr2, s), cst("interval", s), 
                   cst(NullStr, s), cst(Length, n(i)), cst(NewID, n(i))]), 
           _, _),
	insert_table(Name, NewID, Col_Type_List).
insert_table(Name, ID, [Exp-Type-Null|Col_Type_List]) :-
	!,
	expression_to_column(Exp, Column),
	atom_chars(Column, ColStr),
	atom_chars(Null, NullStr),
	get_column_length(Type, Length),
	get_type(Type, TypeStr),
	NewID is ID + 1, 
	sql_exec([],
	   insert(table_view_schemes, 
                  [cst(Name, s), cst(ColStr, s), cst(TypeStr, s), 
                   cst(NullStr, s), cst(Length, n(i)), cst(NewID, n(i))]), 
           _, _),
	insert_table(Name, NewID, Col_Type_List).


/****************************************************************************/
/* Mode:	add_meta_data(+Name, +Columns, Type)                        */
/* Purpose:	Adds meta-info of table Name to table_view_schemes and      */
/*              table_types.                                                */
/****************************************************************************/

add_meta_data(TableName, Columns, TableType) :-
	atom_chars(TableName, Name),
	\+ table_exists(Name),
	!,
	insert_table(Name, 0, Columns),
	insert_table_type(TableName, TableType).


/****************************************************************************/
/* Mode:	delete_meta_data(+Name)                                     */
/* Purpose:	Deletes meta-info of table Name in table_view_schemes.      */
/****************************************************************************/

delete_meta_data(TableName) :-
	atom_chars(TableName, Name1),
	adjust_string(Name1, Name),
	sql_exec([],
	         delete(table_view_schemes-"a1",
	         where([rel(eq, "a1"-"table_name", cst(Name, s))])),
	         N, _),
	N \== 0,
	!,
	delete_meta_data(TableName).
delete_meta_data(TableName) :- delete_table_type(TableName).
	
/****************************************************************************/
/* Mode:	add_view_meta_data(+View, +Query, +ColumnTypes, +Time, +Now)*/
/* Purpose:	Adds meta-info of view Name to table_view_schemes, views    */
/*              and table_types.                                            */
/****************************************************************************/

add_view_meta_data(View, Query, Columns, ColumnTypes, TT) :-
	add_meta_data(View, ColumnTypes, TT),
	atom_chars(View, ViewStr),
	format_to_chars("~k.", [Query], QueryStr1),
	format_to_chars("~k.", [Columns], ColsStr1),
	double_quotes(QueryStr1, QueryStr),
	double_quotes(ColsStr1, ColsStr),
	sql_exec([],
	   insert(views, [cst(ViewStr, s), cst(ColsStr, s), cst(QueryStr, long)]), 
	          _, _).
	
	
/****************************************************************************/
/* Mode:	get_view_meta_data(+View, -Query)                           */
/* Purpose:	Gets query of view View.                                    */
/****************************************************************************/

get_view_meta_data(View, Query, Cols) :-
	atom_chars(View, ViewStr),
	sql_sel([],
	        [sel([a1-calcs, a1-cols]),
	         from([views-a1]),
	         where([rel(eq, a1-name, cst(ViewStr, s))])],
		[var(QueryStr1, long), var(ColsStr1, s)]),
	single_quotes(QueryStr1, QueryStr),
	single_quotes(ColsStr1, ColsStr),
	read_from_chars(QueryStr, Query),
	read_from_chars(ColsStr, Cols).
	

/****************************************************************************/
/* Mode:	delete_view_meta_data(+View)                                */
/* Purpose:	Deletes meta-info of view Name to table_view_schemes, views */
/*              and table_types.                                            */
/****************************************************************************/

delete_view_meta_data(View) :-
	delete_meta_data(View),
	atom_chars(View, Name1),
	adjust_string(Name1, Name),
	sql_exec([],
	         delete(views, where([rel(eq, name-noref, cst(Name, s))])),
	         _, _).
	
/****************************************************************************/
/* Mode:	insert_ref_int(+Table, +Col, +RefTable, +RefCol, +TimeType) */
/* Purpose:	Adds referential integrity constraint to metatable refint.  */
/****************************************************************************/

insert_ref_int(Table, Col, RefTable, RefCol, TimeType) :-
	!,
	atom_chars(Table, TableStr),
	atom_chars(Col, ColStr),
	atom_chars(RefTable, RefTableStr),
	atom_chars(RefCol, RefColStr),
	atom_chars(TimeType, TimeTypeStr),
	sql_exec([],
	   insert(refint, 
                  [cst(TableStr, s), cst(ColStr, s), 
                   cst(RefTableStr, s), cst(RefColStr, s), cst(TimeTypeStr, s)]), 
           _, _).

/****************************************************************************/
/* Mode:	delete_ref_int(+Table)                                      */
/* Purpose:	Deletes all referential integrity constraints of Table.     */
/****************************************************************************/

delete_ref_int(Table) :-
	atom_chars(Table, TabStr),
	adjust_string(TabStr, Name),
	sql_exec([],
	         delete(refint, where([rel(eq, origin-noref, cst(Name, s))])),
	         _, _).

/****************************************************************************/
/* Mode:	get_ref_int(+Table, -ReferencedColumns)                     */
/* Purpose:	Gets all columns of a table which are referenced.           */
/****************************************************************************/

get_reference(Table, RefCol) :-
	atom_chars(Table, TableStr),
	sql_sel([],
	        [sel([a1-refcol]),
	         from([refint-a1]),
	         where([rel(eq, a1-reftable, cst(TableStr, s))])],
		[var(RefColStr1, s)]),
	lower_case(RefColStr1, RefColStr),
	atom_chars(RefCol, RefColStr).
	
get_ref_int(Table, RefCols) :-
	findall(Col, get_reference(Table, Col), RefCols). 

/****************************************************************************/
/* Mode:	get_ref_data(+Table,                                        */
/*                           -Origin, -Col, -RefTable, -RefCol, -TimeType)  */
/* Purpose:	Gets data on a referential integrity constraint.            */
/****************************************************************************/

get_ref_data(Origin, Origin, Col, RefTable, RefCol, TimeType) :-
	atom_chars(Origin, Name1),
	adjust_string(Name1, Name),
	sql_sel([],
		[sel([col-noref, reftable-noref, refcol-noref, timetype-noref]),
		 from([refint]),
		 where([and([rel(eq, origin-noref, cst(Name, s))])])],
		[var(C, s), var(RT, s), var(RC, s), var(TT, s)]),
	lower_case(C, Col1), atom_chars(Col, Col1),
	lower_case(RT, RefTab1), atom_chars(RefTable, RefTab1),
	lower_case(RC, RefCol1), atom_chars(RefCol, RefCol1),
	lower_case(TT, TimeType1), atom_chars(TimeType, TimeType1).
get_ref_data(RefTable, Origin, Col, RefTable, RefCol, TimeType) :-
	atom_chars(RefTable, Name1),
	adjust_string(Name1, Name),
	sql_sel([],
		[sel([origin-noref, col-noref, refcol-noref, timetype-noref]),
		 from([refint]),
		 where([and([rel(eq, reftable-noref, cst(Name, s))])])],
		[var(OR, s), var(C, s), var(RC, s), var(TT, s)]),
	lower_case(OR, Origin1), atom_chars(Origin, Origin1),
	lower_case(C, Col1), atom_chars(Col, Col1),
	lower_case(RC, RefCol1), atom_chars(RefCol, RefCol1),
	lower_case(TT, TimeType1), atom_chars(TimeType, TimeType1).


/****************************************************************************/
/* Mode:	get_referenced_table(-Tab, -Col, -RefTab, -RefCol, -TT)     */
/* Purpose:	Gets an entry out of meta-table refint.                     */
/****************************************************************************/

get_referenced_table(Table, Col, RefTable, RefCol, TimeType) :-
	sql_sel([],
	        [sel([all]),
	         from([refint-a1]),
	         where([])],
		[var(T, s), var(C, s), var(RT, s), var(RC, s), var(TT, s)]),
	lower_case(T, TabStr),
	atom_chars(Table, TabStr),
	lower_case(C, ColStr),
	atom_chars(Col, ColStr),
	lower_case(RT, RefTabStr),
	atom_chars(RefTable, RefTabStr),
	lower_case(RC, RefColStr),
	atom_chars(RefCol, RefColStr),
	lower_case(TT, TimeTypeStr),
	atom_chars(TimeType, TimeTypeStr).
	

/****************************************************************************/
/* Mode:	add_assertion(+Name, +Query, +Type)                         */
/* Purpose:	Adds a new assertion to metatable assertions.               */
/****************************************************************************/

add_assertion(Name, Query, Type) :-
	atom_chars(Name, NameStr),
	atom_chars(Type, TypeStr),
	format_to_chars("~k.", [Query], QueryStr1),
	double_quotes(QueryStr1, QueryStr),
	sql_exec([],
	         insert(assertions, 
	                [cst(NameStr, s), cst(TypeStr, s), cst(QueryStr, long)]), 
	          _, _).


/****************************************************************************/
/* Mode:	get_assertion(-Name, -Query, -Type)                         */
/* Purpose:	Gets an assertion out of metatable assertions.              */
/****************************************************************************/

get_assertion(Name, Query, Type) :-
	sql_sel([],
	        [sel([a1-name, a1-type, a1-query]),
	         from([assertions-a1]),
	         where([])],
		[var(NameStr, s), var(TypeStr, s), var(QueryStr1, long)]),
	single_quotes(QueryStr1, QueryStr),
	read_from_chars(QueryStr, Query),
	atom_chars(Name, NameStr),
	atom_chars(Type, TypeStr).


/****************************************************************************/
/* Mode:	delete_assertion(+Name)                                     */
/* Purpose:	Deletes an assertion.                                       */
/****************************************************************************/

delete_assertion(Name) :-
	atom_chars(Name, NameStr1),
	adjust_string(NameStr1, NameStr),
	sql_exec([],
	         delete(assertions, where([rel(eq, name-noref, cst(NameStr, s))])),
	         _, _).


/****************************************************************************/
/* Mode:	add_key(+TableName, +AttributeList)                         */
/* Purpose:	Adds a new primary key to metatable keys.                   */
/****************************************************************************/

add_key(_, []) :- !.
add_key(Name, [Attr|Rest]) :-
	!,
	atom_chars(Name, NameStr),
	atom_chars(Attr, AttrStr),
	sql_exec([],
	         insert(keys, 
	                [cst(NameStr, s), cst(AttrStr, s)]), 
	          _, _),
	add_key(Name, Rest).


/****************************************************************************/
/* Mode:	get_key(+TableName, -AttributeList)                         */
/* Purpose:	Gets a primary key out of metatable keys.                   */
/****************************************************************************/

get_key_attribute(Name, Attr) :-
	atom_chars(Name, NameStr),
	sql_sel([],
	        [sel([a1-attribute]),
	         from([keys-a1]),
	         where([rel(eq, a1-tablename, cst(NameStr, s))])],
		[var(AttrStr, s)]),
	atom_chars(Attr, AttrStr).

get_key(Table, AttributeList) :-
	findall(Attr, get_key_attribute(Table, Attr), AttributeList). 


/****************************************************************************/
/* Mode:	delete_key(+TableName)                                      */
/* Purpose:	Deletes a primary key of metatable keys.                    */
/****************************************************************************/

delete_key(TableName) :-
	atom_chars(TableName, NameStr1),
	adjust_string(NameStr1, NameStr),
	sql_exec([],
	         delete(keys, where([rel(eq, tablename-noref, cst(NameStr, s))])),
	         _, _).

/****************************************************************************/
/* Mode:	delete_key_attributes(+TableName, +AttrList)                */
/* Purpose:	Deletes primary key attributes in metatable keys.           */
/****************************************************************************/

%% get_condition(+AttrList, -Where)
get_condition([], []) :- !.
get_condition([Attr|Rest], [rel(eq, attribute-noref, cst(AttrStr, s))|Conds]) :-
	atom_chars(Attr, AttrStr1),
	adjust_string(AttrStr1, AttrStr),
	get_condition(Rest, Conds).


delete_key_attributes(_, []) :- !.
delete_key_attributes(TableName, Attr) :-
	get_condition(Attr, Conds),
	atom_chars(TableName, NameStr1),
	adjust_string(NameStr1, NameStr),
	sql_exec([],
	         delete(keys, where([and([rel(eq, tablename-noref, cst(NameStr, s)),
	                                  or(Conds)])])),
	         _, _).


/****************************************************************************/
/* Mode:	add_table_column_constraints(+TableName, +QueryList)        */
/* Purpose:	Adds new table column constraints to metatable              */
/*              table_column_constraints.                                   */
/****************************************************************************/

add_table_column_constraint(Name, Query) :-
	atom_chars(Name, NameStr),
	format_to_chars("~k.", [Query], QueryStr1),
	double_quotes(QueryStr1, QueryStr),
	sql_exec([],
	         insert(table_column_constraints, 
	                [cst(NameStr, s), cst(QueryStr, long)]), 
	          _, _).


add_table_column_constraints(_, []) :- !.
add_table_column_constraints(Name, [Query|QueryList]) :-
	!,
	add_table_column_constraint(Name, Query),
	add_table_column_constraints(Name, QueryList).


/****************************************************************************/
/* Mode:	get_table_column_constraints(-TableName, -QueryList)        */
/* Purpose:	Gets all table column constraint of a table out of          */
/*              metatable table_column_constraints.                         */
/****************************************************************************/

get_table_column_constraint(TableName, Query) :-
	sql_sel([],
	        [sel([a1-table_name, a1-query]),
	         from([table_column_constraints-a1]),
	         where([])],
		[var(TableNameStr, s), var(QueryStr1, long)]),
	single_quotes(QueryStr1, QueryStr),
	read_from_chars(QueryStr, Query),
	atom_chars(TableName, TableNameStr).


/****************************************************************************/
/* Mode:	delete_table_column_constraints(+TableName)                 */
/* Purpose:	Deletes all table column constraints of a table.            */
/****************************************************************************/

delete_table_column_constraints(TableName) :-
	atom_chars(TableName, TableNameStr1),
	adjust_string(TableNameStr1, TableNameStr),
	sql_exec([],
	         delete(table_column_constraints, 
                        where([rel(eq, table_name-noref, cst(TableNameStr, s))])),
	         _, _).
