/******************************************************************************/
/* Module:      norm_check.pl                                                 */
/* Project:     Oracle Emulator                                               */
/* Author:      Michael Boehlen                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Export:                                                                    */
/*              norm_check/2 (oracle.pl)                                      */
/*              type/2 (eval.pl)                                              */
/* Import:                                                                    */
/*              metadb/1 (oracle.pl)                                          */
/*              u_expr/3 (parser.pl)                                          */
/*              error/1 (sqlmsg.pl)                                           */
/******************************************************************************/

/******************************************************************************/
/* Mode:                norm_check(+SQLCmd,-SQLCmd) 		              */
/* Purpose:             nomalizes and checks the user input		      */
/* Example:             norm_check(Tree,Tree1)                		      */
/* Sideeffects:         error message/flag if input id not a legal SQL-Stmt   */
/* Call:                exits if input is a legal SQL-Stmt                    */
/* Redo:                fails 		            		              */
/******************************************************************************/

norm_check(SQLCmd,SQLCmd1) :- n_cmd(SQLCmd,SQLCmd1).

%% n_cmd(+ParseTree,-ParseTree)
n_cmd(create_table(T,Cs,Q),create_table(T,Cs,Q1)) :- !,
  atom_chars(T,TStr),
  (db:user_catalog(TStr,_) -> error(obj_ex(T)), fail; true),
  n_cmd(Q,Q1).
n_cmd(create_view(V,Cs,Q),create_view(V,Cs,Q1)) :- !,
  atom_chars(V,VStr),
  (db:user_catalog(VStr,_) -> error(obj_ex(V)), fail; true),
  n_cmd(Q,Q1).
n_cmd(create_sequence(N,S,I),create_sequence(N,S,I)) :- !,
  atom_chars(N,SStr),
  (db:user_catalog(SStr,_) -> error(obj_ex(N)), fail; true).
n_cmd(insert(T,Cs,Q),insert(T,Cs,Q1)) :- !,
  atom_chars(T,TStr),
  (metadb(TStr) -> error(meta_obj(T)), fail; true),
  (db:user_catalog(TStr,_) -> true; error(no_obj(T)), fail),
  n_cmd(Q,Q1).
n_cmd(delete(T-A,W),delete(T-A,Q1)) :- !,
  n_from([T-A]),
  atom_chars(T,TStr),
  (metadb(TStr) -> error(meta_obj(T)), fail; true),
  n_query(sel(_,[colS(all,_)],[T-A],W),[],Q1,_).
n_cmd(drop_table(T),drop_table(T)) :- !,
  atom_chars(T,TStr),
  (metadb(TStr) -> error(meta_obj(T)), fail; true), 
  (db:user_catalog(TStr,"table") -> true; error(no_obj(T)), fail).
n_cmd(drop_view(V),drop_view(V)) :- !,
  atom_chars(V,VStr),
  (metadb(VStr) -> error(meta_obj(V)), fail; true), 
  (db:user_catalog(VStr,"view") -> true; error(no_obj(V)), fail).
n_cmd(drop_sequence(S),drop_sequence(S)) :- !,
  atom_chars(S,SStr),
  (metadb(SStr) -> error(meta_obj(S)), fail; true), 
  (db:user_catalog(SStr,"sequence") -> true; error(no_obj(S)), fail).
n_cmd(query(_,Q,O),query(S1,Q1,O1)) :- !,
  n_query(Q,[],Q1,S1),
  n_orderby(O,Q1,S1,O1).
n_cmd(update(T-A,CL,query(S,Q,[]),W1),
      update(T-A,CL,set(query(S1,Q1,[])),which(Q2))) :- !,
  atom_chars(T,TStr),
  (metadb(TStr) -> error(meta_obj(T)); true), 
  (db:user_catalog(TStr,_) -> true; error(no_obj(T)), fail),
  n_from([T-A]),
  n_query(Q,[[T-A]],Q1,S1),
  n_colList(CL,[[T-A]],S),
  n_query(sel(all,[colS(all,_)],[T-A],W1),[],Q2,_).
n_cmd(X,X).

n_colList([],_,_) :- !.
n_colList([colR(R-A,Pos,N)|T],Fs,[SE|SET]) :- !,
  chk_colName(Fs,colR(R-A,Pos,N),colR(R-A,Pos,N)),
  type(colR(R-A,Pos,N),Type),
  type(SE,Type),
  n_colList(T,Fs,SET).
n_colList(colR(R-A,Pos,N),Fs,[SE|_]) :- !,
  chk_colName(Fs,colR(R-A,Pos,N),colR(R-A,Pos,N)),
  type(colR(R-A,Pos,N),Type),
  type(SE,Type).
n_colList(_,_,_) :- error(ill_upd), fail.

%% n_from(*FromList)
n_from([]).
n_from([R-A|T]) :- 
  atom_chars(R,RStr),
  (db:user_catalog(RStr,Kind), Kind\=="sequence" ->
     (R=A; true);
     error(no_obj(R)), fail), !,
  n_from(T).

%% n_query(+ParseTree,+FromLists,-ParseTree,-SelectList)
n_query(union(Q1,Q2),Fs,union(Q3,Q4),S1) :-
  n_query(Q1,Fs,Q3,S1),
  n_query(Q2,Fs,Q4,S2),
  chk_setop(S1,S2).
n_query(minus(Q1,Q2),Fs,minus(Q3,Q4),S1) :-
  n_query(Q1,Fs,Q3,S1),
  n_query(Q2,Fs,Q4,S2),
  chk_setop(S1,S2).
n_query(inter(Q1,Q2),Fs,inter(Q3,Q4),S1) :-
  n_query(Q1,Fs,Q3,S1),
  n_query(Q2,Fs,Q4,S2),
  chk_setop(S1,S2).
n_query(sel(AD,S,F,W),Fs,sel(AD,S1,F,W1,A-Sx-T),S1) :- !,
  n_from(F),
  replace_star(S,F,S2),
  n_select(S2,[F],S1,ai(A-A,Sx-Sx,T-T),ai(A-[],Sx-[],T-[])),
  n_expr(W,[F|Fs],bool,W1,_,_).

%% n_select(+SelectList,+FromLists,-SelectList,+AggInfo,-AggInfo)
n_select([],_,[],AI,AI).
n_select([colS(E,Alias)|T],Fs,[colS(E1,Alias)|T1],AI,AI1) :-
  n_expr(E,Fs,_,E1,AI,AI2),
  (var(Alias) ->
     (E=colR(_,_,Alias) -> true; u_expr(E,A,[]), atom_chars(Alias,A));
     true),
  n_select(T,Fs,T1,AI2,AI1).

%% n_orderby(+OrderbyList,+Query,+TopSelect,-NewOrderbyList)
n_orderby(O,sel(_,S1,F,_,_),S1,O1) :- !, n_orderby2(O,S1,[F],O1).
n_orderby(O,_,S1,O1) :- !, n_orderby2(O,S1,[],O1).
n_orderby2([],_,_,[]).
n_orderby2([num(N)-AD|O],S1,Fs,[num(N)-AD|O1]) :- !, n_orderby2(O,S1,Fs,O1).
n_orderby2([Expr-AD|O],S1,Fs,[Expr1-AD|O1]) :-
  n_expr(Expr,Fs,_,Expr1,_,_),
  n_orderby2(O,S1,Fs,O1).

%% n_expr(+ParseTree,+FromLists,+-Type,-ParseTree,+AggInfo,-AggInfo)
n_expr(true,_,bool,true,AI,AI) :- !.
n_expr(str(S),_,char,str(S),AI,AI) :- !.
n_expr(num(N),_,num,num(N),AI,AI) :- !.
n_expr(colR(R-A,Pos,N),Fs,T,ColRef,AI,AI) :- !,
  chk_colName(Fs,colR(R-A,Pos,N),ColRef),
  type(ColRef,T).
n_expr(agg(count,_,all),_,num,agg(X),
       ai(A-[count|AT],S-[colS(num(1),_)|ST],T-[X|TT]),
       ai(A-AT,S-ST,T-TT)) :- !.
n_expr(agg(Agg,DA,E),Fs,Type,agg(X),
       ai(A-AT,S-ST,T-TT),ai(A-AT1,S-ST1,T-TT1)) :- !,
  n_expr(E,Fs,_,E1,
         ai(A-AT,S-ST,T-TT),ai(A-[Agg|AT1],S-[colS(E1,_)|ST1],T-[X|TT1])),
  type(agg(Agg,DA,E),Type).
n_expr(bin(Op,E1,E2),Fs,num,bin(Op,E3,E4),AI,AI1) :- !,
  n_expr(E1,Fs,num,E3,AI,AI2),
  n_expr(E2,Fs,num,E4,AI2,AI1).
n_expr(gst(EL),Fs,num,gst(EL1),AI,AI1) :- !, n_exprlist(EL,Fs,num,EL1,AI,AI1).
n_expr(lst(EL),Fs,num,lst(EL1),AI,AI1) :- !, n_exprlist(EL,Fs,num,EL1,AI,AI1).
n_expr(to_num(E),Fs,num,to_num(E1),AI,AI1) :- !, n_expr(E,Fs,char,E1,AI,AI1).
n_expr(trunc(E),Fs,num,trunc(E1),AI,AI1) :- !, n_expr(E,Fs,num,E1,AI,AI1).
n_expr(abs(E),Fs,num,abs(E1),AI,AI1) :- !, n_expr(E,Fs,num,E1,AI,AI1).
n_expr(rel(Rel,E1,E2),Fs,bool,rel(Rel,E3,E4),AI,AI1) :- !,
  n_expr(E1,Fs,T,E3,AI,AI2),
  n_expr(E2,Fs,T,E4,AI2,AI1).
n_expr(un(Op,E1),Fs,num,un(Op,E2),AI,AI1) :- !,
  n_expr(E1,Fs,num,E2,AI,AI1).
n_expr(and(E1,E2),Fs,bool,and(E3,E4),AI,AI1) :- !,
  n_expr(E1,Fs,bool,E3,AI,AI2),
  n_expr(E2,Fs,bool,E4,AI2,AI1).
n_expr(or(E1,E2),Fs,bool,or(E3,E4),AI,AI1) :- !,
  n_expr(E1,Fs,bool,E3,AI,AI2),
  n_expr(E2,Fs,bool,E4,AI2,AI1).
n_expr(not(E1),Fs,bool,not(E2),AI,AI1) :- !,
  n_expr(E1,Fs,bool,E2,AI,AI1).
n_expr(ex(query(_,Q,[])),Fs,bool,ex(query(S1,Q1,[])),AI,AI) :- !,
  n_query(Q,Fs,Q1,S1).
n_expr(in(EL,query(_,Q,[])),Fs,bool,in(EL1,query(S1,Q1,[])),AI,AI) :- !,
  n_query(Q,Fs,Q1,S1),
  n_exprlist(EL,Fs,_,EL1,_,_).
n_expr(input(N,T1),_,T2,input(N,T1),AI,AI) :- !, type(input(N,T1),T2).
n_expr(X,_,_,_,_,_) :- error(unknown_expr(X)), fail.

%% n_exprlist(+ExprList,+FromLists,+Type,-ExprList,+AggInfo,-AggInfo)
n_exprlist([],_,_,[],AI,AI) :- !.
n_exprlist([E|T],Fs,Type,[E1|T1],AI,AI1) :- !,
  copy_term(Type,Type2),
  n_expr(E,Fs,Type2,E1,AI,AI2),
  n_exprlist(T,Fs,Type,T1,AI2,AI1).

%% replace_star(+SelectList,+FromList,-SelectList)
replace_star([],_,[]).
replace_star([colS(all,_)],F,S2) :- !,
  findall(colS(colR(R-A,_,all),_), member(R-A,F), S1),
  replace_star2(S1,F,S2).
replace_star(S,F,S1) :- replace_star2(S,F,S1).
replace_star2([],_,[]).
replace_star2([colS(colR(R-A,_,all),_)|SR],F,S3) :- !,
  memberchk(R-A,F),
  atom_chars(R,RStr),
  findall(colS(colR(R-A,Pos,N),N), 
          (db:user_tab_columns(RStr,NStr,_,_,Pos), atom_chars(N,NStr)), 
          S1),
  replace_star2(SR,F,S2),
  append(S1,S2,S3).
replace_star2([C|SR],F,[C|SR1]) :- replace_star2(SR,F,SR1).

/******************************************************************************/
/* Mode:                type(+SQLExpr,?Type) 				      */
/* Purpose:             determines/checks the type of an SQL expression	      */
/* Example:             type(bin(add,num(1),str(abc)),num)                    */
/* Sideeffects:         error message/flag, if no type can be inferred	      */
/* Call:                exits if type is ok 		          	      */
/* Redo:                fails            				      */
/******************************************************************************/

%% type(+SQLExpr,?Type)
type(num(_),num) :- !.
type(str(_),char) :- !.
type(colR(R-A,Pos,N),T) :-
  atom_chars(R,RStr),
  atom_chars(N,NStr),
  db:user_tab_columns(RStr,NStr,TStr,_,Pos), !,
  (atom_chars(T,TStr) -> true; error(type_err(colR(R-A,Pos,N),T)), fail).
type(un(_,A),num) :- !, type(A,num).
type(bin(_,A,B),num) :- !, type(A,num), type(B,num).
type(rel(_,A,B),bool) :- !, type(A,T), type(B,T).
type(gst(EL),num) :- !, type(EL,num).
type(lst(EL),num) :- !, type(EL,num).
type(to_num(E),num) :- !, type(E,char).
type(trunc(E),num) :- !, type(E,num).
type(abs(E),num) :- !, type(E,num).
type(input(_,T),T) :- !.
type(true,bool) :- !.
type(agg(max,_,E),T) :- !, type(E,T).
type(agg(min,_,E),T) :- !, type(E,T).
type(agg(count,_,_),num) :- !.
type(agg(sum,_,E),num) :- !, type(E,num).
type(agg(avg,_,E),num) :- !, type(E,num).
type(colS(E,_),T) :- !, type(E,T).
type(nextval(_),num) :- !.
type(currval(_),num) :- !.
type([],_) :- !.
type([E|EL],T) :- !, type(E,T), type(EL,T).
type(E,T) :- error(type_err(E,T)), fail.

%% chk_setop(+SelectList,+SelectList)
chk_setop([],[]) :- !.
chk_setop([colS(SE1,_)|SR1],[colS(SE2,_)|SR2]) :- !,
  type(SE1,T1),
  type(SE2,T2),
  (T1==T2 -> chk_setop(SR1,SR2); error(incTypes_in_setop), fail).
chk_setop(_,_) :- error(ill_setop), fail.

%% chk_colName(+FromLists,+colRef,-colRef)
chk_colName([],colR(_-S,_,nextval),nextval(SStr)) :- !,
  atom_chars(S,SStr),
  (db:user_sequences(SStr) -> true; error(seq_unknown(S)), fail).
chk_colName([],colR(_-S,_,currval),currval(SStr)) :- !,
  atom_chars(S,SStr),
  (db:user_sequences(SStr) -> 
    (db:sequence(SStr,_,_,def) -> true; error(seq_undef(S)), fail);
    error(seq_unknown(S)), fail).
chk_colName([],colR(_,_,N),_) :- error(ill_colN(N)), fail.
chk_colName([H|T],colR(R-A,Pos,N),ColRef) :-
  atom_chars(N,NStr), 
  findall([R-A,Pos,N],
          ( member(R-A,H),
            db:user_tab_columns(RStr,NStr,_,_,Pos),
            atom_chars(R,RStr)),
          Res),
  (Res=[]            -> chk_colName(T,colR(R-A,Pos,N),ColRef);
   Res=[[R-A,Pos,N]] -> ColRef=colR(R-A,Pos,N);
                        error(ambig_colN(N)), fail).

/******************************************************************************/
/* .									      */
/******************************************************************************/
