/******************************************************************************/
/* Module:      oracle.pl                                                     */
/* Project:     Oracle Emulator                                               */
/* Author:      Michael Boehlen                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Export:                                                                    */
/*              abolish/1 (eval.pl)                                           */
/*              bf/3 (parser.pl)                                              */
/*              set_of/3 (eval.pl)                                            */
/*              metadb/1 (norm_check.pl)                                      */
/* Import:                                                                    */
/*              read_lines/1 (parser.pl)                                      */
/*              scan/2 (parser.pl)                                            */
/*              tokenize/3 (parser.pl)                                        */
/*              parse/2 (parser.pl)                                           */
/*              norm_check/2 (norm_check.pl)                                  */
/*              compile/3 (eval.pl)                                           */
/*              exec_query/4 (eval.pl)                                        */
/*              exec_stmt/3 (eval.pl)                                         */
/*              msg/1 (sqlmsg.pl)                                             */
/******************************************************************************/

:- module(oracle,[sql/0, sql/1,
                  sql_open_db/1,
                  sql_close_db/0,
                  sql_open_cursor/1,
                  sql_compile_statement/2,
                  sql_close_cursor/1,
                  sql_create_column_string/2,
                  sql_create_column_integer/2,
                  sql_create_column_float/2,
                  sql_get_column_string/4,
                  sql_get_column_integer/4,
                  sql_get_column_float/4,
                  sql_put_var_string/3,
                  sql_put_var_integer/3,
                  sql_put_var_float/3,
                  sql_fetch_row/2,
                  sql_exec_cursor/3,
                  sql_commit/0,
                  sql_rollback/0]).

:- ensure_loaded([flags,parser,norm_check,metadb,eval,sqlmsg]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% SWI-Prolog %%%%%%%%%%%%%%%%%%%%%%

sum_list(L,S) :- sum_list2(L,0,S).
sum_list2([],S,S).
sum_list2([H|T],S0,S) :- S1 is H+S0, sum_list2(T,S1,S).

max_list([H|T],M) :- max_list2(T,H,M).
max_list2([],M,M).
max_list2([H|T],X,M) :- H=<X, !, max_list2(T,X,M).
max_list2([H|T],_,M) :- max_list2(T,H,M).

min_list([H|T],M) :- min_list2(T,H,M).
min_list2([],M,M).
min_list2([H|T],X,M) :- H>=X, !, min_list2(T,X,M).
min_list2([H|T],_,M) :- min_list2(T,H,M).

abolish(P/A) :- abolish(P,A).

format_to_chars(F,A,L) :- sformat(S,F,A), string_to_list(S,L).
format_to_chars(F,A,L,R) :- sformat(S,F,A), string_to_list(S,X), append(X,R,L).

bf(X,I,O) :- append(X,O,I).

:- redefine_system_predicate(select(_,_,_)).
select(X,L,L1) :- system:select(L,X,L1).

set_of(T,G,R) :- findall(T,G,R1), set_of2(R1,R).
set_of2([],[]).
set_of2([H|T],[H|T2]) :- \+memberchk(H,T), !, set_of2(T,T2).
set_of2([H|T],T2) :- memberchk(H,T), set_of2(T,T2).

/******************************************************************************/
/* Mode:                sql 	                          		      */
/* Purpose:             starts sql main loop and prompts for user commands    */
/* Example:             sql		                		      */
/* Sideeffects:         none                                                  */
/* Call:                exits always                                          */
/* Redo:                fails always                                          */
/******************************************************************************/

sql :-
  assert(verbose),
  nl,
  repeat,
    retractall(err(_)),
    read_lines(StmtStr),
    scan(StmtStr,Tks),
    parse(Tree,Tks), \+err(_),
    interpret(Tree,StmtStr),
    Tree==quit,
  retractall(verbose),
  format("Disconnected from PROLOG RDBMS V2.0 Interpreter - Production~n",[]).

sql(Login) :-
  atom_chars(Login,LoginChrs),
  tokenize([id(DBName)|_],LoginChrs,[]),
  atom_chars(DBName,DBNameChrs),
  assert(db(DBNameChrs)),
  load_from_file(DBNameChrs),
  asserta(undo_log(savepoint('$$work$$'))),
  sql,
  store_to_file(DBNameChrs),
  abolish(undo_log/1),
  retract(db(DBNameChrs)).

%% interpret(+ParseTree,+StmtStr)
interpret(quit,_) :- !.
interpret(query(S,Q,O),StmtStr) :- !,
  norm_check(query(S,Q,O),query(S1,Q1,O1)), \+err(_),
  compile(query(S1,Q1,O1),_,PrologClause),
  exec_query(PrologClause,StmtStr,Res,Num),
  displ_result(Res,S1),
  msg(selRec(Num)).
interpret(Cmd,StmtStr) :-
  norm_check(Cmd,Cmd1), \+err(_),
  compile(Cmd1,_,PrologClause),
  exec_stmt(PrologClause,StmtStr,_).

/******************************************************************************/

%% calc_format(+Tabel,-StrTabel,+LengthList,-LengthList)
calc_format([],[],X,X).
calc_format([_-Tupel|R],[StrTupel|StrR],LengthIn,LengthOut) :-
  calc_format2(Tupel,StrTupel,LengthIn,LengthTmp),
  calc_format(R,StrR,LengthTmp,LengthOut).
calc_format2([],[],L,L).
calc_format2([A|T],[Str|StrT],[L|InR],[L1|OutR]) :-
  (integer(A)  -> number_chars(A,Str);
   float(A)    -> format_to_chars("~3f",A,Str);
   is_list(A)  -> Str=A),
  length(Str,Lx),
  (Lx+2>L -> L1 is Lx+2; L1=L),
  calc_format2(T,StrT,InR,OutR).

%% calc_formatStr(+LengthList,-FormatStr)
calc_formatStr([],[0'~,0'n]).
calc_formatStr([H|T],FormOut) :-
  calc_formatStr(T,FormTmp),
  number_chars(H,Chrs),
  append([0'~,0't,0'~,0's,0'~|Chrs],[0'+|FormTmp],FormOut).

%% displ_result(+Table)
displ_result([],_) :- format("~nno tuples selected~n~n",[]).
displ_result([_-Tupel|Table],S) :-
  findall(4,member(_,Tupel),LengthIn),
  findall(Chrs, (member(colS(_,Col),S),atom_chars(Col,Chrs)), Names),
  calc_format([_-Names,_-Tupel|Table],[NameStrs|StrTable],LengthIn,LengthOut),
  calc_formatStr(LengthOut,FormStr), nl,
  sum_list([-2|LengthOut],LengthTot),
  format(FormStr,NameStrs),
  format("  ~*c~n",[LengthTot,0'-]),
  (member(X,StrTable), format(FormStr,X), fail; nl).

/******************************************************************************/

%% store_to_file(+FileNameChrs)
store_to_file(FChrs) :-
  append("$ORACLE_EMULATOR/DBS/",FChrs,AbsFChrs),
  atom_chars(AbsF,AbsFChrs),
  tell(AbsF),
  (db:num_cols(String,Arity),
     format(":- dynamic '~s'/~d.~n", [String,Arity]),
     fail;
     true),
  db:listing,
  told.
  
%% load_from_file(+FileNameChrs)
load_from_file(FChrs) :-
  append("$ORACLE_EMULATOR/DBS/",FChrs,AbsFChrs),
  atom_chars(AbsF,AbsFChrs),
  (exists_file(AbsF) ->
    (db:num_cols(String,Arity),
       atom_chars(Pred,String),
       Pred\==num_cols,
       db:abolish(Pred,Arity),
       fail;
       true),
     db:abolish(num_cols,2),
     db:consult(AbsF);
     true).

/*****************************************************************************/

%% sql_open_cursor(-Cursor)
sql_open_cursor(C2) :-
  cursor(C,_,_,_,_),
  \+ (cursor(C1,_,_,_,_), C1>C), !,
  C2 is C+1.
sql_open_cursor(0).

%% sql_compile_statement(+Cursor,-SQLString)
sql_compile_statement(Cur,StmtStr) :-
  scan(StmtStr,Tks),
  parse(Tree,Tks),
  norm_check(Tree,Tree1),
  compile(Tree1,Cur,Stmt),
  assert(cursor(Cur,StmtStr,Stmt,none,none)).

%% sql_create_column_XXX(+Cursor,+VariableNr).
sql_create_column_string(_,_).
sql_create_column_integer(_,_).
sql_create_column_float(_,_).
 
%% sql_put_var_XXX(+Cursor,+VariableNr,+Value).
sql_put_var_string(Cur,Nr,Str) :-
  retractall(in(Cur,Nr,_)),
  assert(in(Cur,Nr,Str)).
sql_put_var_integer(Cur,Nr,Int) :-
  retractall(in(Cur,Nr,_)),
  assert(in(Cur,Nr,Int)).
sql_put_var_float(Cur,Nr,Float) :-
  retractall(in(Cur,Nr,_)),
  assert(in(Cur,Nr,Float)).

%% sql_fetch_row(+Cursor,-Success).
sql_fetch_row(Cur,0) :-
  retract(cursor(Cur,StmtStr,CompiledStmt,_,[_-R|Rs])),
  assert(cursor(Cur,StmtStr,CompiledStmt,R,Rs)), !.
sql_fetch_row(_,1).

%% sql_get_column_XXX(+Cursor,+ColumnNr,-Value,-ReturnCode).
sql_get_column_string(Cur,Nr,Str,0) :-
  cursor(Cur,_,_,R,_),
  nth0(Nr,R,X),
  (number(X) -> number_chars(X,Str); Str=X).
sql_get_column_integer(Cur,Nr,Int,0) :-
  cursor(Cur,_,_,R,_),
  nth0(Nr,R,Int).
sql_get_column_float(Cur,Nr,Float,0) :-
  cursor(Cur,_,_,R,_),
  nth0(Nr,R,Float).

%% sql_exec_cursor(+Cursor,-NumberOfRows,-ReturnCode).
sql_exec_cursor(Cur,Num,ErrCode) :-
  retract(cursor(Cur,StmtStr,CompiledStmt,_,_)),
  (functor(CompiledStmt,query,_) ->
    exec_query(CompiledStmt,StmtStr,Res,Num);
    exec_stmt(CompiledStmt,StmtStr,Num),
    Res=none), !,
  assert(cursor(Cur,StmtStr,CompiledStmt,none,Res)),
  (retract(err(ErrCode)) -> true; ErrCode=0).

%% sql_close_cursor(+Cursor).
sql_close_cursor(Cur) :-
  retract(cursor(Cur,_,_,_,_)),
  retractall(in(Cur,_,_)).

%% sql_open_db(+LoginStr)
sql_open_db(LoginStr) :-
  tokenize([id(DBName)|_],LoginStr,[]),
  atom_chars(DBName,DBNameChrs),
  assert(db(DBNameChrs)),
  load_from_file(DBNameChrs),
  asserta(undo_log(savepoint('$$work$$'))).

%% sql_close_db.
sql_close_db :-
  retract(db(DBNameChrs)),
  store_to_file(DBNameChrs),
  abolish(undo_log/1).

sql_commit :- exec_stmt(commit,"commit",_).
sql_rollback :- exec_stmt(rollback('$$work$$'),"rollback",_).

/******************************************************************************/

metadb("dictionary").	metadb("dict_columns").	metadb("user_catalog").
metadb("user_tables").	metadb("user_views").	metadb("user_tab_columns").
metadb("dual").		metadb("dict").		metadb("cat").	
metadb("tabs").		metadb("cols").		metadb("user_sequences").
metadb("num_cols").	metadb("sequence").

/******************************************************************************/
/* .	                                                                      */
/******************************************************************************/
