/******************************************************************************/
/* Module:      basics.pl                                                     */
/* Project:     TimeDB 1.03                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Date:        December 12, 1995                                             */
/* Export:                                                                    */
/*              remove_counter/0 (constraints.pl,timeDB.pl)                   */
/*              drop_aux_tables/1 (check.pl,coalescing.pl,constraints.pl,     */
/*                                 evaluate.pl,timeDB.pl)                     */
/*              get_op/3 (errors.pl)                                          */
/*              get_expression/3 (errors.pl,check.pl,translate.pl,display.pl) */
/*              generate_alias/1 (parser.pl,translate.pl)                     */
/*              generate_table_name/2 (translate.pl,coalescing.pl,evaluate.pl)*/
/*              generate_interval_col/3 (parser.pl,check.pl,translate.pl,     */
/*                                       meta.pl)                             */
/*              get_interval_name/3 (check.pl,translate.pl)                   */
/*              write_status/0 (evaluate.pl)                                  */
/* Import:                                                                    */
/*              sql_exec/4 (meta.pl)                                          */
/*              internal_to_date/2 (calendar.pl)                              */
/******************************************************************************/

remove_counter :- 
	(retract(table_counter(_)), !; true),
	(retract(alias_counter(_)), !; true).



/****************************************************************************/
/* Mode:          drop_aux_tables(+DropTableList)                           */
/* Purpose:       Drops auxiliary tables in DropTableList.                  */
/* Used in:       chronosql.pl, coalescing.pl, evaluate.pl                  */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

drop_aux_tables([])           :-!.
drop_aux_tables([Table|Rest]) :- 
	sql_exec([], tdrop(Table), _, _), 
	drop_aux_tables(Rest).


/****************************************************************************/
/* Mode:          get_expression(+Exp, -Expression, [])                     */
/* Purpose:       Converts an expression in intermediate format to a string */
/* Used in:       display.pl, translate.pl                                  */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

get_op(precedes) --> " precedes ".	
get_op(overlaps) --> " overlaps ".
get_op(meets)    --> " meets ".	
get_op(contains) --> " contains ".
get_op(eq)       --> " = ".
get_op(nq)       --> " <> ".
get_op(ls)       --> " < ".
get_op(gr)       --> " > ".
get_op(gq)       --> " >= ".
get_op(lq)       --> " <= ".

get_condition(rel(Op, E1, E2))      --> 
	!, get_expression(E1), get_op(Op), get_expression(E2).
get_condition(between(Col, E1, E2)) --> 
	!, get_expression(Col), 
	" between ", get_expression(E1), " and ", get_expression(E2).

get_expression(neg(Exp)) --> !, "-", get_expression1(Exp).
get_expression(Exp)      --> !, get_expression1(Exp).

get_expression1(add(Term, Exp)) --> !, get_term(Term), "+", get_expression1(Exp).
get_expression1(sub(Term, Exp)) --> !, get_term(Term), "-", get_expression1(Exp).
get_expression1(Term)           --> !, get_term(Term).

get_term(quo(Faktor, Term)) -->	!, get_faktor(Faktor), "/", get_term(Term).
get_term(mul(Faktor, Term)) --> !, get_faktor(Faktor), "*", get_term(Term).
get_term(Faktor)            --> !, get_faktor(Faktor).

get_faktor(cst(C, char))         --> !, C.
get_faktor(cst(C, interval))     --> !, get_event(C).
get_faktor(cst(N, event))        --> !, "|", get_event(N), "|".
get_faktor(cst(N, span))         --> !, "%", get_number(N), "%".
get_faktor(cst(N, _))            --> !, get_number(N).
get_faktor(expr(_, Alias))       --> !, get_string(Alias).
get_faktor(vtime(Ref))           --> !, "vtime(", get_string(Ref), ")".
get_faktor(xtime(Ref))           --> !, "xtime(", get_string(Ref), ")".
get_faktor(valid(Ref))           --> !, "valid(", get_string(Ref), ")".
get_faktor(transaction(Ref))     --> !, "transaction(", get_string(Ref), ")".
get_faktor(begin(Exp))           --> !, "begin(", get_expression(Exp), ")".
get_faktor(end(Exp))             --> !, "end(", get_expression(Exp), ")".
get_faktor(span(Span))           --> !, "span(", get_expression(Span), ")".
get_faktor(abs(Num))             --> !, "abs(", get_expression(Num), ")".
get_faktor(absolute(Span))       --> !, "absolute(", get_expression(Span), ")".
get_faktor(first(E1, E2))        --> !, 
	"first(", get_expression(E1), ",", get_expression(E2), ")".
get_faktor(last(E1, E2))         --> !, 
	"last(", get_expression(E1), ",", get_expression(E2), ")".
get_faktor(interval(Start, End)) --> !,
	"[", get_expression(Start), "-", get_expression(End), ")".
get_faktor(lst([E1, E2]))        --> !,
	"least(", get_expression(E1), ",", get_expression(E2), ")".
get_faktor(gst([E1, E2]))        --> !, 
	"greatest(", get_expression(E1), ",", get_expression(E2), ")".
get_faktor(intersect(E1, E2))    --> !, 
	"intersect(", get_expression(E1), ",", get_expression(E2), ")".
get_faktor(Ref-Attr)             --> !, get_string(Ref), ".", get_string(Attr).
get_faktor(Attr)                 --> {atom(Attr)}, !, get_string(Attr).
get_faktor(Faktor)               --> !, get_expression1(Faktor).

get_string(S) --> {atom_chars(S, Str)}, Str.
get_number(N) --> {number_chars(N, Str)}, Str.
get_event(I)  --> {number_chars(I, Str1), internal_to_date(Str1, Str)}, Str.


/****************************************************************************/
/* Mode:          generate_alias(-Alias)                                    */
/* Purpose:       Generates a new unused alias.                             */
/* Used in:       parser.pl, translate.pl                                   */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

generate_alias(Alias) :-
	(retract(alias_counter(N)); N=0),
	name(N, NumStr),
	append("a#$_", NumStr, AliasStr),
	name(Alias, AliasStr),
	!,
	N1 is N+1,
	assert(alias_counter(N1)).


/****************************************************************************/
/* Mode:          generate_table_name(+RootStr, -TableName)                 */
/* Purpose:       Generates a new unused table-name.                        */
/* Used in:       translate.pl, evaluate.pl, coalescing.pl                  */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

generate_table_name(RootStr, TableName) :-
	append(RootStr, "_#$", TableStr1),
	(retract(table_counter(N)); N=0),
	name(N, NumStr),
	append(TableStr1, NumStr, TableStr),
	name(TableName, TableStr),
	!,
	N1 is N+1,
	assert(table_counter(N1)).


/****************************************************************************/
/* Mode:          generate_interval_col(+Table-Column, -Start, -End)        */
/* Purpose:       Generates a two attribute names out of interval-column.   */
/* Used in:       check.pl, meta.pl, parser.pl, translate.pl                */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

generate_interval_col(Table-Col, Table-Start, Table-End) :-
	!,
	atom_chars(Col, ColStr),
	append(ColStr, "s_#$", ColStr1),
	append(ColStr, "e_#$", ColStr2),
	atom_chars(Start, ColStr1),
	atom_chars(End, ColStr2).
generate_interval_col(Col, Start, End) :-
	atom(Col),
	!,
	atom_chars(Col, ColStr),
	append(ColStr, "s_#$", ColStr1),
	append(ColStr, "e_#$", ColStr2),
	atom_chars(Start, ColStr1),
	atom_chars(End, ColStr2).

/****************************************************************************/
/* Mode:          get_interval_name(-Col, +Start, +End)                     */
/* Purpose:       Gets original attribute name out of interval-attributes.  */
/* Used in:       check.pl, meta.pl, parser.pl, translate.pl                */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

get_interval_name(Col, Start, End) :-
	atom(Start),
	atom(End), 
	!,
	atom_chars(Start, StartStr),
	atom_chars(End, EndStr),
	append(ColStr, "s_#$", StartStr),
	append(ColStr, "e_#$", EndStr),
	atom_chars(Col, ColStr).


/****************************************************************************/
/* Mode:          write_status                                              */
/* Purpose:       Writes status report to screen.                           */
/* Used in:       evaluate.pl                                               */
/* Example:                                                                 */
/* Sideeffects:   Output on screen.                                         */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

write_calendar :-
	calendar(Cal), !, write('Calendar : '), write(Cal), nl.
write_calendar :-
	write('Calendar : none '), nl.

write_emulator :- emulator(on), !, write('Evaluation : Emulator'), nl. 
write_emulator.

write_oracle :- oracle(on), !, write('Evaluation : Oracle RDBMS'), nl.
write_oracle.

write_tt :- demo(_, _), !, write('Transaction-Time : Demo'), nl.
write_tt :- write('Transaction-Time : Real'), nl.

write_tracing :- tracing(_), !, write('Tracing : on'), nl.
write_tracing.


write_status :-
	write_calendar,
	write_emulator,
	write_oracle,
	write_tt,
	nl.
