/******************************************************************************/
/* Module:      calendar.pl                                                   */
/* Project:     TimeDB 1.03                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Date:        December 12, 1995                                             */
/* Export:                                                                    */
/*              change_calendar/1 (evaluate.pl)                               */
/*              granularity/1 (parser.pl,evaluate.pl)                         */
/*              get_inc/2 (evaluate.pl)                                       */
/*              interval_to_internal/4 (parser.pl)                            */
/*              event_to_internal/3 (parser.pl)                               */
/*              span_to_int/3 (parser.pl)                                     */
/*              internal_to_date/2 (basics.pl,display.pl)                     */
/*              internal_to_span/2 (display.pl)                               */
/* Import:                                                                    */
/*              tokenize/3 (scanner.pl)                                       */
/*              generate_error_flag/2 (errors.pl)                             */
/*              fiscal_granularity/1 (fiscal_calendar.pl)                     */
/*              fiscal_get_inc/2 (fiscal_calendar.pl)                         */
/*              fiscal_to_internal/3 (fiscal_calendar.pl)                     */
/*              internal_to_fiscal/2 (fiscal_calendar.pl)                     */
/*              fiscal_span_to_internal/4 (fiscal_calendar.pl)                */
/*              internal_to_fiscal_span/2 (fiscal_calendar.pl)                */
/*              fiscal_min_number/1 (fiscal_calendar.pl)                      */
/*              fiscal_max_number/1 (fiscal_calendar.pl)                      */
/*              decimal_granularity/1 (decimal_calendar.pl)                   */
/*              decimal_get_inc/2 (decimal_calendar.pl)                       */
/*              decimal_to_internal/3 (decimal_calendar.pl)                   */
/*              internal_to_decimal/2 (decimal_calendar.pl)                   */
/*              decimal_span_to_internal/4 (decimal_calendar.pl)              */
/*              internal_to_decimal_span/2 (decimal_calendar.pl)              */
/*              decimal_min_number/1 (decimal_calendar.pl)                    */
/*              decimal_max_number/1 (decimal_calendar.pl)                    */
/*              sql3_granularity/1 (sql3_calendar.pl)                         */
/*              sql3_get_inc/2 (sql3_calendar.pl)                             */
/*              sql3_to_internal/3 (sql3_calendar.pl)                         */
/*              internal_to_sql3/2 (sql3_calendar.pl)                         */
/*              sql3_span_to_internal/4 (sql3_calendar.pl)                    */
/*              internal_to_sql3_span/2 (sql3_calendar.pl)                    */
/*              sql3_min_number/1 (sql3_calendar.pl)                          */
/*              sql3_max_number/1 (sql3_calendar.pl)                          */
/******************************************************************************/

:- ensure_loaded([fiscal_calendar]).
:- ensure_loaded([decimal_calendar]).
:- ensure_loaded([sql3_calendar]).

calendar_description(decimal) :-
	!,
	nl, nl,
	write('Decimal calendar: '),
	write('starting at 0, chronon : year'),
	nl, nl.

calendar_description(fiscal) :-
	!,
	nl, nl,
	write('Common fiscal calendar: '),
	write('starting 1900, 31 days per month, chronon : second'),
	nl, nl.

calendar_description(sql3) :-
	!,
	nl, nl,
	write('SQL3 calendar: '),
	write('starting 1900, 31 days per month, chronon : second'),
	nl, nl.


% change_calendar(+CalName)
change_calendar(CalName) :-
	calendar(C), 
	!, 
	retractall(calendar(C)),
	assert(calendar(CalName)),
	nl, write('Calendar changed!'), nl.
change_calendar(CalName) :-
	assert(calendar(CalName)),
	nl, write('Calendar changed!'), nl.


granularity(G) :- 
	calendar(fiscal),
	!,
	fiscal_granularity(G).
granularity(G) :- 
	calendar(sql3),
	!,
	sql3_granularity(G).
granularity(G) :- 
	calendar(decimal),
	!,
	decimal_granularity(G).


get_inc(none, 0) :- !.
get_inc(G, Inc)  :-
	calendar(fiscal),
	!,
	fiscal_get_inc(G, Inc).
get_inc(G, Inc)  :-
	calendar(sql3),
	!,
	sql3_get_inc(G, Inc).
get_inc(G, Inc)  :-
	calendar(decimal),
	!,
	decimal_get_inc(G, Inc).


/****************************************************************************/
/*************************** CALENDAR INTERFACE *****************************/
/****************************************************************************/

get_start([])       --> [-].
get_start([C|Rest]) --> [C], get_start(Rest).

get_end([C|Rest]) --> [C], get_end(Rest).
get_end([])       --> [].

get_start_parenthesis --> {calendar(sql3)}, ['['].
get_start_parenthesis --> [].

get_end_parenthesis --> {calendar(sql3)}, [')'].
get_end_parenthesis --> [].



%% interval_to_internal(-Start, -End)
interval_to_internal(Start, End) --> 
	[str(IntervalStr)], 
	{tokenize(TokenList1, IntervalStr, []),
	 get_start_parenthesis(TokenList1, TokenList),
 	 get_start(StartTokens, TokenList, TokenRest),
	 date_to_internal(Start, StartTokens, []),
 	 get_end(EndTokens, TokenRest, CloseBracket),
	 date_to_internal(End, EndTokens, []),
	 get_end_parenthesis(CloseBracket, [])}.

%% event_to_internal(-Event)
event_to_internal(Event) --> 
	[str(EventStr)], 
	{tokenize(TokenList, EventStr, []),
	 date_to_internal(Event, TokenList, [])}.

%% span_to_int(+Qualifier, +SpanStr, -Span)
span_to_int(Qual, SpanStr, Span) :-
	tokenize(TokenList, SpanStr, []),
	span_to_internal(Qual, Span, TokenList, []).

/****************************************************************************/

% date_to_internal(+Time)
date_to_internal(Time) --> 
	{calendar(fiscal)},
	!, 
	fiscal_to_internal(Time).
date_to_internal(Time) --> 
	{calendar(sql3)},
	!, 
	sql3_to_internal(Time).
date_to_internal(Time) --> 
	{calendar(decimal)},
	!, 
	decimal_to_internal(Time).
date_to_internal(_) --> 
	{generate_error_flag(no_calendar, _)},
	[_]. % Due to SWI-Bug

% span_to_internal(Qual, Value)
span_to_internal(Qual, Value) --> 
	{calendar(fiscal)},
	!,
	fiscal_span_to_internal(Qual, Value).
span_to_internal(Qual, Value) --> 
	{calendar(sql3)},
	!,
	sql3_span_to_internal(Qual, Value).
span_to_internal(Qual, Value) --> 
	{calendar(decimal)},
	!,
	decimal_span_to_internal(Qual, Value).
span_to_internal(_, _)     --> 
	{generate_error_flag(no_calendar, _)},
	[_]. % Due to SWI-Bug

/****************************************************************************/

% internal_to_date(+Internal, -Literal)
internal_to_date(Internal, Literal) :- 
	calendar(fiscal),
	!,
	internal_to_fiscal(Internal, Literal).
internal_to_date(Internal, Literal) :- 
	calendar(sql3),
	!,
	internal_to_sql3(Internal, Literal).
internal_to_date(Internal, Literal) :- 
	calendar(decimal),
	!,
	internal_to_decimal(Internal, Literal).
internal_to_date(_, _)              :- 
	generate_error_flag(no_calendar, _).

% internal_to_span(+SpanStr, -SpanStr)
internal_to_span(SpanStrIn, SpanStrOut) :-
	calendar(fiscal),
	!,
	internal_to_fiscal_span(SpanStrIn, SpanStrOut).
internal_to_span(SpanStrIn, SpanStrOut) :-
	calendar(sql3),
	!,
	internal_to_sql3_span(SpanStrIn, SpanStrOut).
internal_to_span(SpanStrIn, SpanStrOut) :-
	calendar(decimal),
	!,
	internal_to_decimal_span(SpanStrIn, SpanStrOut).
internal_to_span(_, _) :- 
	generate_error_flag(no_calendar, _).


/****************************************************************************/
/* Constants used in translate.pl (-inf, +inf)                              */
/****************************************************************************/

min_number(Min) :- 
	calendar(fiscal),
	!,
	fiscal_min_number(Min).
min_number(Min) :- 
	calendar(sql3),
	!,
	sql3_min_number(Min).
min_number(Min) :- 
	calendar(decimal),
	!,
	decimal_min_number(Min).
min_number(_) :- 
	generate_error_flag(500, _).


max_number(Max) :- 
	calendar(fiscal),
	!,
	fiscal_max_number(Max).
max_number(Max) :- 
	calendar(sql3),
	!,
	sql3_max_number(Max).
max_number(Max) :- 
	calendar(decimal),
	!,
	decimal_max_number(Max).
max_number(_) :- 
	generate_error_flag(500, _).

