/******************************************************************************/
/* Module:      fiscal_calendar.pl                                            */
/* Project:     TimeDB 1.03                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Date:        December 12, 1995                                             */
/* Export:                                                                    */
/*              fiscal_calendar_start_year/1 (time.pl)                        */
/*              fiscal_granularity/1 (calendar.pl)                            */
/*              fiscal_get_inc/2 (calendar.pl)                                */
/*              fiscal_to_internal/3 (calendar.pl)                            */
/*              internal_to_fiscal/2 (calendar.pl)                            */
/*              fiscal_span_to_internal/4 (calendar.pl)                       */
/*              internal_to_fiscal_span/2 (calendar.pl)                       */
/*              fiscal_min_number/1 (time.pl,calendar.pl)                     */
/*              fiscal_max_number/1 (time.pl,calendar.pl)                     */
/* Import:                                                                    */
/*              generate_error_flag/2 (errors.pl)                             */
/*              get_tt_date/2 (time.pl)                                       */
/******************************************************************************/

fiscal_calendar_start_year(0).

fiscal_granularity(second).

fiscal_get_inc(year, Inc)   :- Inc is 12*31*24*60*60.
fiscal_get_inc(month, Inc)  :- Inc is 31*24*60*60.
fiscal_get_inc(day, Inc)    :- Inc is 24*60*60.
fiscal_get_inc(hour, Inc)   :- Inc is 60*60.
fiscal_get_inc(minute, Inc) :- Inc is 60.
fiscal_get_inc(second, Inc) :- Inc is 1.

/****************************************************************************/
/* Mode:	fiscal_to_internal(-Seconds)				    */
/* Purpose:	Parses a date-string and converts it to an integer 	    */
/*              representing amount of seconds gone by since 1970/1/1       */
/* Example:								    */
/* Sideeffects:	None							    */
/* Call:        Exits always 						    */
/* Redo:        Fails always						    */
/****************************************************************************/

fiscal_now(Now) --> [now], !, {get_tt_date(now, Now)}.
fiscal_now(Now) --> [present], {get_tt_date(now, Now)}.

fiscal_beginning(BEGINNING) --> [beginning], {fiscal_min_number(BEGINNING), !}. 
fiscal_forever(FOREVER)     --> [forever], {fiscal_max_number(FOREVER), !}. 

from_year(Seconds) --> 
	[num(Year)],
	!,
	{fiscal_calendar_start_year(Y),
	 (Year >= Y, !; generate_error_flag(invalid_year, Y))}, 
	from_year2(Year, Seconds).
from_year2(Years, Seconds) --> 
	[/],
	!, 
	{fiscal_calendar_start_year(Y), Months is (Years-Y) * 12},
	from_month(Months, Seconds).
from_year2(Years, Seconds) --> 
	[],  
	{fiscal_calendar_start_year(Y), 
	 Seconds is (((Years-Y)*12 + 1) * 31 + 1) * 24 * 60 * 60}.


from_month(M1, Seconds) --> 
	[num(Month)],
	!,
	{Month >= 0, Month < 13, M is M1 + Month}, 
	from_month2(M, Seconds).
from_month2(Months, Seconds) --> 
	[/], 
	{Days is Months*31},
	from_day(Days, Seconds).
from_month2(Months, Seconds) --> 
	[], 
	{Seconds is ((Months * 31) + 1) * 24 * 60 * 60}.

from_day(D, Seconds) --> 
	[num(Day)], 
	!,
	{Day >= 0, Day =< 31, Days is D + Day}, 
	from_day2(Days, Seconds).
from_day2(Days, Seconds) --> 
	['~'],
	!, 
	{Hours is Days * 24},
	from_hours(Hours, Seconds).
from_day2(Days, Seconds) --> 
	[], 
	{Seconds is Days * 24 * 60 * 60}.

from_hours(H, Seconds) --> 
	[num(Hour)], 
	!,
	{Hour >= 0, Hour < 24, Hours is H + Hour}, 
	from_hours2(Hours, Seconds).
from_hours2(Hours, Seconds) --> 
	[:], 
	!,
	{Minutes is Hours * 60},
        from_minutes(Minutes, Seconds).
from_hours2(Hours, Seconds) --> 
	[], 
	{Seconds is Hours * 60 * 60}.

from_minutes(M, Seconds) --> 
	[num(Minute)], 
	!,
	{Minute >= 0, Minute < 60, Mins is M + Minute}, 
	from_minutes2(Mins, Seconds).
from_minutes2(Minutes, Seconds) --> 
	[:], 
	!,
	{Sec is Minutes * 60},
        from_seconds(Sec, Seconds).
from_minutes2(Minutes, Seconds) --> 
	[], 
	{Seconds is Minutes * 60}.

from_seconds(S, Seconds) --> 
	[num(Secs)], 
	{Secs >= 0, Secs < 60, Seconds is S+Secs}.


fiscal_to_internal(Seconds) --> 
	fiscal_now(Seconds),
        !.
fiscal_to_internal(Seconds) --> 
	fiscal_forever(Seconds),
        !.
fiscal_to_internal(Seconds) --> 
	fiscal_beginning(Seconds),
        !.
fiscal_to_internal(Seconds) --> 
	from_year(Seconds),
	!,
 	{fiscal_max_number(MaxSecs), Seconds < MaxSecs}.

/****************************************************************************/
/* Mode:	internal_to_fiscal(+Internal, -Date_Literal)		    */
/* Purpose:	Converts an integer representing amount of seconds 	    */
/*              gone by since start year into a date-string                 */
/* Example:								    */
/* Sideeffects:	None							    */
/* Call:        Exists always						    */
/* Redo:        Fails always			 	                    */
/****************************************************************************/

to_chars(I, D, [D, 0'0|Str]) :- I < 10, !, number_chars(I, Str).
to_chars(I, D, [D|Str])      :- number_chars(I, Str).

to_seconds(Internal, Date) :-
 	S is Internal mod 60, S == 0,
	!,
	Rest is Internal // 60,
	to_minutes(S, Rest, Date).
to_seconds(Internal, Date) :-
	!,
 	S is Internal mod 60,
	to_chars(S, 58, Sec),
	Rest is Internal // 60,
	to_minutes(S, Rest, Min),
	append(Min, Sec, Date).

to_minutes(0, Internal, Date) :-
 	M is Internal mod 60, M == 0,
	!,
	Rest is Internal // 60,
	to_hours(0, Rest, Date).
to_minutes(_, Internal, Date) :-
	!,
 	M is Internal mod 60,
	to_chars(M, 58, Min),
	Rest is Internal // 60,
	to_hours(1, Rest, Hrs),
	append(Hrs, Min, Date).

to_hours(0, Internal, Date) :-
 	H is Internal mod 24, H == 0,
	!,
	Rest is Internal // 24,
	to_days(0, Rest, Date).
to_hours(_, Internal, Date) :-
	!,
 	H is Internal mod 24,
	to_chars(H, 126, Hrs),
	Rest is Internal // 24,
	to_days(1, Rest, Days),
	append(Days, Hrs, Date).

to_days(0, Internal, Date) :-
 	D is Internal mod 31, D == 1,
	!,
	Rest is Internal // 31,
	to_months(0, Rest, Date).
to_days(_, Internal, Date) :-
 	D is Internal mod 31, D == 0,
	!,
	to_chars(31, 47, Days),
	Rest is (Internal-31) // 31,
	to_months(1, Rest, Months),
	append(Months, Days, Date).
to_days(_, Internal, Date) :-
	!,
 	D is Internal mod 31,
	to_chars(D, 47, Days),
	Rest is Internal // 31,
	to_months(1, Rest, Months),
	append(Months, Days, Date).

to_months(0, Internal, Date) :-
 	M is Internal mod 12, M == 1,
	!,
	Rest is Internal // 12,
	to_years(Rest, Date).
to_months(_, Internal, Date) :-
 	M is Internal mod 12, M == 0,
	!,
	to_chars(12, 47, Months),
	Rest is (Internal-12) // 12,
	to_years(Rest, Years),
	append(Years, Months, Date).
to_months(_, Internal, Date) :-
	!,
 	M is Internal mod 12,
	to_chars(M, 47, Months),
	Rest is Internal // 12,
	to_years(Rest, Years),
	append(Years, Months, Date).

to_years(Internal, Date) :-
	!,
	fiscal_calendar_start_year(Y), 
 	Year is Internal + Y,
	number_chars(Year, Date).


internal_to_fiscal(Internal, "forever") :-
        number_chars(I, Internal), 
        fiscal_max_number(I1), 
	I >= I1,
        !.
internal_to_fiscal(Internal, "eot") :-
        number_chars(I, Internal), I < 0, !.
internal_to_fiscal(Internal, "beginning") :-
        number_chars(I, Internal), 
        fiscal_min_number(I1), 
	I =< I1, I >= 0,
        !.
internal_to_fiscal(Internal, Literal) :-
	!,
        number_chars(I, Internal), 
        to_seconds(I, Literal).


/****************************************************************************/
/* Mode:	fiscal_span_to_internal(-Seconds)			    */
/* Purpose:	Parses a duration-string and converts it to an integer 	    */
/*              representing amount of seconds                              */
/* Example:								    */
/* Sideeffects:	None							    */
/* Call:        Exits always				                    */
/* Redo:        Fails always						    */
/****************************************************************************/

fiscal_chronons(year, Value, Chronons)   :- Chronons is Value *12*31*24*60*60.
fiscal_chronons(month, Value, Chronons)  :- Chronons is Value *31*24*60*60.
fiscal_chronons(day, Value, Chronons)    :- Chronons is Value *24*60*60.
fiscal_chronons(hour, Value, Chronons)   :- Chronons is Value *60*60.
fiscal_chronons(minute, Value, Chronons) :- Chronons is Value *60.
fiscal_chronons(second, Chronons, Chronons).
 
fiscal_span_to_internal(Q, Chronons) --> [num(V)], {fiscal_chronons(Q, V, Chronons)}. 


/*****************************************************************************/
/* Mode:	internal_to_fiscal_span(+Internal, -Span_Literal)            */
/* Purpose:	Converts an integer representing duration in seconds         */
/* Example:								     */
/* Sideeffects:	None							     */
/* Call:        Exists always						     */
/* Redo:        Fails always						     */
/*****************************************************************************/

internal_to_fiscal_span(Span, SpanStr) :- 
	!, 
	number_chars(Span, Str), 
	append(Str, " second", SpanStr).


/*****************************************************************************/
/* Constants used in translate.pl (-inf, +inf)                               */
/*****************************************************************************/

fiscal_min_number(0).
fiscal_max_number(1000000000000).
