/******************************************************************************/
/* Module:      interface.pl                                                  */
/* Project:     TimeDB 1.0                                                    */
/* Author:      Michael Boehlen                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Export:                                                                    */
/*              db_open/1 (evaluate.pl)                                       */
/*              db_close/0 (evaluate.pl)                                      */
/*              sql_exec/4 (basics.pl,coalescing.pl,evaluate.pl,meta_data.pl) */
/*              sql_sel/3 (check.pl,constraints.pl,meta_data.pl)              */
/*              my_sql_sel/3 (time.pl)                                        */
/* Import:                                                                    */
/*              generate_interval_col/3 (basics.pl)                           */
/******************************************************************************/

elem1([])      --> [0'|], !.
elem1([X|Y])   --> [X], elem1(Y).

elem(n(T))     --> {var(T)}, "n(n)|", !.
elem(n(i))     --> "n(i)|", !.
elem(n(f))     --> "n(f)|", !.
elem(t(i))     --> "t(i)|", !.
elem(s)        --> "s()|", !.
elem(unspec)   --> "u()|", !.
elem(Str)      --> elem1(Str).


%% listXstr(?List, ?StringIn, +StringOut)
listXstr([X|Y]) --> elem(X), !, listXstr(Y).
listXstr([])    --> "".

/*****************************************************************************/
/* Mode:              db_open(+Login)                                        */
/* Purpose:           connects to a database and open cursors for metatables */
/* Example:           db_open("scott/tiger")                                 */
/* Sideeffects:       connected to db an cursors allocated                   */
/* Call:              exits if connection to db is successfull               */
/* Redo:              fails always                                           */
/*****************************************************************************/

db_open(S) :-
  sql_open_db(S).

db_close :-
  retract(cursor(_,C)),
    sql_close_cursor(C),
    fail;
  sql_close_db.

/****************************************************************************/
/* Mode:            g_stat(+SQL_stat,-Chars,+Chars)                         */
/* Purpose:         generate sql statement                                  */
/* Example:         g_stat(open(multi,ded),Cs,[])                           */
/* Sideeffects:     none                                                    */
/* Call:            exits always                                            */
/* Redo:            fails always                                            */
/****************************************************************************/

g_stat(create(R,L))   --> "CREATE TABLE ", g_atom(R), "(", g_defs(L), ")".
g_stat(create(R,S))   --> "CREATE TABLE ", g_atom(R), " AS ", g_selstat(S).
g_stat(create(R,C,S)) --> "CREATE TABLE ", g_atom(R), "(", g_red_defs(C), ")",
			    " AS ", g_selstat(S).

g_stat(tdrop(R))      --> "DROP TABLE ", g_atom(R).
g_stat(vdrop(R))      --> "DROP VIEW ", g_atom(R).

g_stat(delete(P,W))   --> "DELETE FROM ", g_range(P), g_wherestat(W).

g_stat(insert(P,C,S)) --> "INSERT INTO ", g_atom(P), " (", g_collist(C), ") ",
                          g_selstat(S).
g_stat(insert(P,C,S)) --> "INSERT INTO ", g_atom(P), " (", g_collist(C), 
                               ") VALUES (", g_exprlist(S), ")".
g_stat(insert(P,S))   --> "INSERT INTO ", g_atom(P), " ", g_selstat(S).
g_stat(insert(P,S))   --> "INSERT INTO ", g_atom(P), " VALUES (", 
			       g_exprlist(S), ")".

g_stat(lit(S))		--> S.
g_stat(update(P,U,W))   --> "UPDATE ", g_range(P),
                            " SET ", g_update(U),
                            g_wherestat(W).
g_stat(X)               --> g_selstat(X).

g_selstat([S,F,W,G,H,O]) --> g_selectstat(S), g_fromstat(F), g_wherestat(W),
			     g_groupstat(G), g_havingstat(H), g_orderstat(O).
g_selstat([S,F])         --> g_selectstat(S), g_fromstat(F).
g_selstat([S,F,W])       --> g_selectstat(S), g_fromstat(F), g_wherestat(W).
g_selstat([S,F,W,G,H])   --> g_selectstat(S), g_fromstat(F), g_wherestat(W),
		             g_groupstat(G), g_havingstat(H).

g_selectstat(sel(S))  --> "SELECT ", g_exprlist(S).
g_selectstat(seld(S)) --> "SELECT DISTINCT ", g_exprlist(S).

g_fromstat(from(F)) --> " FROM ", g_from(F).

g_wherestat(where([])) --> "".
g_wherestat(where(W))  --> " WHERE ", g_condlist(W," OR ").

g_groupstat(group([])) --> "".
g_groupstat(group(G))  --> " GROUP BY ", g_exprlist(G).

g_havingstat(having([])) --> "".
g_havingstat(having(H))  --> " HAVING ", g_condlist(H," OR ").

g_orderstat(order([])) --> "".
g_orderstat(order(L))  --> " ORDER BY ", g_exprlist(L).


g_selstat([union(L),O])	    --> g_union(L), g_orderstat(O).
g_selstat([union(L)])	    --> g_union(L).
g_selstat([minus(L),O])	    --> g_minus(L), g_orderstat(O).
g_selstat([minus(L)])	    --> g_minus(L).
g_selstat([intersect(L),O]) --> g_intersect(L), g_orderstat(O).
g_selstat([intersect(L)])   --> g_intersect(L).

g_update(set(Cs,SQ))    --> "(", g_exprlist(Cs), ")=(", g_selstat(SQ), ")".

g_union([A])		--> !, g_selstat(A).
g_union([A|B])		--> "(", g_selstat(A), " UNION ", g_union2(B).
g_union2([A])		--> !, g_selstat(A), ")".
g_union2([A|B])		--> g_selstat(A), " UNION ", g_union2(B).

g_minus([A])		--> !, g_selstat(A).
g_minus([A|B])		--> "(", g_selstat(A), " MINUS ", g_minus2(B).
g_minus2([A])		--> !, g_selstat(A), ")".
g_minus2([A|B])		--> g_selstat(A), " MINUS ", g_minus2(B).

g_intersect([A])	--> !, g_selstat(A).
g_intersect([A|B])	--> "(", g_selstat(A), " INTERSECT ", g_intersect2(B).
g_intersect2([A])	--> !, g_selstat(A), ")".
g_intersect2([A|B])	--> g_selstat(A), " INTERSECT ", g_intersect2(B).

g_defs([C-T-null])      --> !, g_col(C), " ", g_type(T).
g_defs([C-T-notnull])   --> !, g_col(C), " ", g_type(T), " NOT NULL".
g_defs([C-T-null|B])    --> !, g_col(C), " ", g_type(T), ", ", g_defs(B).
g_defs([C-T-notnull|B]) --> !, g_col(C), " ", g_type(T), " NOT NULL, ", 
                            g_defs(B).
g_defs([C-T])           --> !, g_col(C), " ", g_type(T).
g_defs([C-T|B])         --> !, g_col(C), " ", g_type(T), ", ", g_defs(B).

g_red_defs([C-_-null])      --> !, g_col(C), " ".
g_red_defs([C-_-notnull])   --> !, g_col(C), " ", " NOT NULL".
g_red_defs([C-_-null|B])    --> !, g_col(C), ", ", g_red_defs(B).
g_red_defs([C-_-notnull|B]) --> !, g_col(C), " NOT NULL, ", g_red_defs(B).
g_red_defs(X)		    --> g_collist(X).

g_collist([C])          --> !, g_col(C).
g_collist([C|B])        --> g_col(C), ", ", g_collist(B).

% character strings sind in oracle auf 240 bytes begrenzt)
g_type(s(X))            --> 
	{Len is min(X, 240), number_chars(Len,Str)}, "CHAR(", Str, ")".
g_type(long)            --> "LONG".
g_type(s)               --> "CHAR(32)".
g_type(n(f))            --> "FLOAT".
g_type(n(i))            --> "INTEGER".
g_type(n(T))            --> {var(T)}, !, "NUMBER".
g_type(number)          --> "INTEGER".
g_type(interval)        --> "INTEGER".
g_type(event)           --> "INTEGER".
g_type(span)            --> "INTEGER".

g_condlist([X],_)	--> g_cond(X), !.
g_condlist([H|T],Op)	--> g_cond(H), Op, g_condlist(T,Op), !.

g_cond(rel(R,X,Y))        --> g_expr(X), g_rel(R), g_expr(Y).
g_cond(or(L))	  	  --> "(", g_condlist(L," OR "), ")".
g_cond(and(L))		  --> g_condlist(L," AND ").
g_cond(ex(S))   	  --> "EXISTS (", g_selstat(S), ")".
g_cond(notex(S))	  --> "NOT EXISTS (", g_selstat(S), ")".
g_cond(like(X,Y))	  --> g_expr(X), " LIKE ", g_expr(Y).
g_cond(notlike(X,Y))	  --> g_expr(X), " NOT LIKE ", g_expr(Y).
g_cond(between(X,Y,Z))	  --> g_expr(X), " BETWEEN ", 
                              g_expr(Y), " AND ", g_expr(Z).
g_cond(notbetween(X,Y,Z)) --> g_expr(X), " NOT BETWEEN ", 
                              g_expr(Y), " AND ", g_expr(Z).
g_cond(not(CL))		  --> "NOT ", g_condlist(CL," OR ").
g_cond(in(X,Y))	          --> "(", g_expr(X), ") IN (", g_exprlist(Y), ")".
g_cond(notin(X,Y))        --> "(", g_expr(X), ") NOT IN (", g_exprlist(Y), ")".
g_cond(isnull(E))	  --> !, g_expr(E), " IS NULL".
g_cond(isnotnull(E))	  --> !, g_expr(E), " IS NOT NULL".


g_rel(eq)		--> "=".
g_rel(nq)		--> "<>".
g_rel(ls)		--> "<".
g_rel(lq)		--> "<=".
g_rel(gr)		--> ">".
g_rel(gq)		--> ">=".

g_from([])              --> "DUAL".
g_from([X])             --> !, g_range(X).
g_from([H|T])           --> g_range(H), ", ", g_from(T).


g_range(R-A-_)	        --> !, g_atom(R), " ", g_alias(A).
g_range(R-A)	        --> g_atom(R), " ", g_alias(A).
g_range(R)	        --> g_atom(R).

g_exprlist([E])         --> !, g_expr(E).
g_exprlist([E|L])       --> g_expr(E), !, ", ", g_exprlist(L).

g_expr(add(X,Y))        --> !, g_term(X), "+", g_expr(Y).
g_expr(sub(X,Y))        --> !, g_term(X), "-", g_expr(Y).
g_expr(neg(X))          --> !, "-", g_term(X).
g_expr(X)               --> g_term(X).

g_term(mul(X,Y))        --> !, g_faktor(X), "*", g_term(Y).
g_term(quo(X,Y))        --> !, g_faktor(X), "/", g_term(Y).
g_term(X)               --> g_faktor(X).

g_faktor(icol(noref-C))          --> !, g_col(C).
g_faktor(icol(T-C))              --> !, g_alias(T), ".", g_col(C).
g_faktor(cst(C,number))          --> !, {number(C), number_chars(C,Str)}, Str.
g_faktor(cst(C,n(_)))            --> !, {number(C), number_chars(C,Str)}, Str.
g_faktor(cst(C,list))            --> !, "'", listXstr(C), "'".
g_faktor(cst(C,interval))        --> !, {number(C), number_chars(C,Str)}, Str.
g_faktor(cst(C,event))           --> !, {number(C), number_chars(C,Str)}, Str.
g_faktor(cst(C,span))            --> !, {number(C), number_chars(C,Str)}, Str.
g_faktor(cst(C,_))               --> !, "'", C, "'".
g_faktor(interval(S,E))          --> !, g_expr(S), ",", g_expr(E).
g_faktor(event(E))               --> !, g_expr(E).
g_faktor(span(E))                --> !, g_expr(E).
g_faktor(null)                   --> !, " NULL ".
g_faktor(A-all)                  --> !, g_alias(A), ".", "*".
g_faktor(C-noref)                --> !, g_col(C).
g_faktor(A-C)                    --> !, g_alias(A), ".", g_col(C).
g_faktor(expr(X,noref))          --> !, g_expr(X).
g_faktor(expr(X))                --> !, g_expr(X).
g_faktor(expr(interval(S,E), X)) --> !, {generate_interval_col(X, S1, E1)}, 
                                     g_expr(S), " ", g_col(S1), ",",
                                     g_expr(E), " ", g_col(E1).
g_faktor(expr(X,C))              --> !, g_expr(X), " ", g_col(C).
g_faktor(all)                    --> !, "*".
g_faktor(dist(X))                --> !, "DISTINCT", g_expr(X).
g_faktor(count(E))               --> !, "COUNT(", g_expr(E), ")".
g_faktor(max(E))                 --> !, "MAX(", g_expr(E), ")".
g_faktor(min(E))                 --> !, "MIN(", g_expr(E), ")".
g_faktor(abs(E))                 --> !, "ABS(", g_expr(E), ")".
g_faktor(mod(X,Y))               --> !, "MOD(", g_expr(X), ",", g_expr(Y), ")".
g_faktor(trunc(X))               --> !, "TRUNC(", g_expr(X), ")".
g_faktor(sfkt(X,Y))              --> !, g_atom(X), "(", g_expr(Y), ")".
g_faktor(asc(X))                 --> !, g_expr(X).
g_faktor(desc(X))                --> !, g_expr(X), " DESC ".
g_faktor(input(N))          	 --> !, ":", {number(N), number_chars(N,Str)}, Str.
g_faktor(gst([]))	         --> !, "0".
g_faktor(gst([H|T]))	         --> !, "GREATEST(", g_exprlist([H|T]), ")".
g_faktor(lst([]))	         --> !, "93312000000".
g_faktor(lst([H|T]))	         --> !, "LEAST(", g_exprlist([H|T]), ")".
g_faktor(col(X))      	         --> !, "(", g_selstat(X), ")".
g_faktor(E)                      --> {functor(E,X,_), 
                                     memberchk(X, [add,sub,neg,mul,quo])},
                                     "(", g_expr(E), ")".

g_col(C+D)              --> !, g_col(C), g_col(D).
g_col(C)                --> {atom(C), !, atom_chars(C,L)}, L.
g_col(C)                --> !, C.

g_alias(A)              --> {atom(A), !, atom_chars(A,L)}, L.
g_alias(A)              --> !, A.

g_atom(X+Y)             --> !, g_atom(X), g_atom(Y).
g_atom([H|T])           --> {prolog(swi)}, !, bf([H|T]).  % SWI - BUG
g_atom([H|T])           --> !, [H|T].    
g_atom(C)               --> {atom(C), !, atom_chars(C,L)}, L.
g_atom(N)               --> {number(N), number_chars(N,Str)}, !, Str.

bf(X,I,O) :- append(X,O,I).

/************************************************************************/

%% sql_redo_fetch_row(+Cursor)
sql_redo_fetch_row(C) :- 
  sql_fetch_row(C,0), 
  (true; !, sql_redo_fetch_row(C)).

%% sql_create_all_columns(+Tupel&+Sort,+Cursor,+ArgCount)
sql_create_all_columns([],_,_).
sql_create_all_columns([var(_,n(f))|TS],C,AC) :- !,
  sql_create_column_float(C,AC),
  AC1 is AC + 1,
  sql_create_all_columns(TS,C,AC1).
sql_create_all_columns([var(_,n(i))|TS],C,AC) :- !,
  sql_create_column_integer(C,AC),
  AC1 is AC + 1,
  sql_create_all_columns(TS,C,AC1).
sql_create_all_columns([var(_,t(i))|TS],C,AC) :- !, 
  sql_create_column_float(C,AC),
  AC1 is AC + 1,
  sql_create_column_float(C,AC1),
  AC2 is AC1 + 1,
  sql_create_all_columns(TS,C,AC2).
sql_create_all_columns([var(_,t(p))|TS],C,AC) :- !,
  sql_create_column_float(C,AC),
  AC1 is AC + 1,
  sql_create_all_columns(TS,C,AC1).
sql_create_all_columns([var(_,_)|TS],C,AC) :-
  sql_create_column_string(C,AC),
  AC1 is AC + 1,
  sql_create_all_columns(TS,C,AC1).

%% sql_get_all_columns(-Tupel&+Sort,+Cursor,+ArgCount)
sql_get_all_columns([],_,_).
sql_get_all_columns([var(N,n(f))|TS],C,AC) :- !,
  sql_get_column_float(C,AC,N,_),
  AC1 is AC+1,
  sql_get_all_columns(TS,C,AC1).
sql_get_all_columns([var(N,n(i))|TS],C,AC) :- !,
  sql_get_column_integer(C,AC,N,_),
  AC1 is AC+1,
  sql_get_all_columns(TS,C,AC1).
sql_get_all_columns([var(UT,t(i))|TS],C,AC) :- !,
  sql_get_column_float(C,AC,Start,_),
  AC1 is AC+1,
  sql_get_column_float(C,AC1,End,_),
  int2time(cst(Start,n(f)),cst(End,n(f)),TimeStamp),
  u_fac(TimeStamp,UT,[]),
  AC2 is AC1+1,
  sql_get_all_columns(TS,C,AC2).
sql_get_all_columns([var(UT,t(p))|TS],C,AC) :- !,
  sql_get_column_float(C,AC,X,_),
  int2timept(cst(X,n(f)),TimePt),
  u_fac(TimePt,UT,[]),
  AC1 is AC+1,
  sql_get_all_columns(TS,C,AC1).
sql_get_all_columns([var(L,list)|TS],C,AC) :- !,
  sql_get_column_string(C,AC,Str,_),
  listXstr(L,Str,[]),
  AC1 is AC+1,
  sql_get_all_columns(TS,C,AC1).
sql_get_all_columns([var(Str,_)|TS],C,AC) :-
  sql_get_column_string(C,AC,Str,_),
  AC1 is AC+1,
  sql_get_all_columns(TS,C,AC1).

%% sql_put_all_variables(+Cursor,+Input)
sql_put_all_variables([],_,_).
sql_put_all_variables([var(I,n(i))|Input],C,IC) :- !,
  (clause(tracing(on),true) -> format(" >:~d = ~d~n",[IC,I]); true),
  sql_put_var_integer(C,IC,I),
  IC1 is IC+1,
  sql_put_all_variables(Input,C,IC1), !.
sql_put_all_variables([var(L,list)|Input],C,IC) :- !,
  listXstr(L,Str,[]),
  (clause(tracing(on),true) -> 
     findall(X, (member(Y,L),(Y=[_|_]->atom_chars(X,Y);X=Y)), L1),
     format(" >:~d = ~w~n",[IC,L1]);
     true),
  sql_put_var_string(C,IC,Str),
  IC1 is IC+1,
  sql_put_all_variables(Input,C,IC1).
sql_put_all_variables([var(Str,_)|Input],C,IC) :-
  (clause(tracing(on),true) -> format(" >:~d = ~s~n",[IC,Str]); true),
  sql_put_var_string(C,IC,Str),
  IC1 is IC+1,
  sql_put_all_variables(Input,C,IC1).

/*****************************************************************************/
/* Mode:	sql_exec(+Input,+Stat,-NumRowsProcessed,-ReturnCode)	     */
/* Purpose:	execute the sql statement 'Stat' (trans format)		     */
/* Example:	sql_exec([var(5,n(i))], insert(...),NumRows,RetCode)	     */
/* Sideeffects:	depends on sql statement, only on database		     */
/* Call:	exits always						     */
/* Redo:	fails always						     */
/*****************************************************************************/

open_exec_cursor(S,Cursor):-
  g_stat(S,SQLStr,[]),
  (clause(tracing(on),true) -> format("~s;~n",[SQLStr]); write('.'), ttyflush),
  sql_open_cursor(Cursor),
  sql_compile_statement(Cursor,SQLStr).

do_exec_cursor(I,Cursor) :-
  (retract(tracing(on)) ->
    sql_put_all_variables(I,Cursor,1),
    assert(tracing(on));
    sql_put_all_variables(I,Cursor,1)),
  sql_exec_cursor(Cursor,_,_).

%% sql_exec(+Input,+Stat,-NumRowsProc,-ReturnCode)
sql_exec(I,S,NumRows,ReturnCode) :- 
  g_stat(S,SQLStr,[]), 
  (clause(tracing(on),true) -> format("~s;~n",[SQLStr]); write('.'), ttyflush),
  sql_open_cursor(C),
  sql_compile_statement(C,SQLStr),
  sql_put_all_variables(I,C,1),
  sql_exec_cursor(C,NumRows,ReturnCode),
  sql_close_cursor(C), !.


/****************************************************************************/
/* Mode:	sql_sel(+Inputs,+SelStat,-Tupel&+Sort)        		    */
/* Purpose:	enumerate tupels of a sql select statement		    */
/* Example:	sql_sel([var(5,n(i))],sel(...), [TS]))			    */
/* Sideeffects:	none							    */
/* Call:	exits if there is a tupel				    */
/* Redo:	exits if there are more tupels				    */
/****************************************************************************/

open_sel_cursor(SS,TS,Cursor):-
  g_selstat(SS,SQLStr,[]), 
  atom_chars(SQLStat,SQLStr),
  (clause(tracing(on),true) -> format("~w;~n", [SQLStat]); write('.'), ttyflush),
  sql_open_cursor(Cursor),
  sql_compile_statement(Cursor,SQLStr),
  sql_create_all_columns(TS,Cursor,0).

do_sel_cursor(I,Cursor,TS) :-
  (retract(tracing(on)) ->
    sql_put_all_variables(I,Cursor,1),
    assert(tracing(on));
    sql_put_all_variables(I,Cursor,1)),
  sql_exec_cursor(Cursor,_,_),
  findall(TS, 
          (sql_redo_fetch_row(Cursor),sql_get_all_columns(TS,Cursor,0)), 
          TSL),
  member(TS,TSL).
  
%% sql_sel(+Input,+SelStat,-Tupel&+Sort)
sql_sel(I,SqlStat,TS) :-
  g_selstat(SqlStat,SQLStr,[]),
  (clause(tracing(on),true) -> format("~s;~n",[SQLStr]); write('.'), ttyflush),
  sql_open_cursor(C),
  sql_compile_statement(C,SQLStr),
  sql_create_all_columns(TS,C,0),
  sql_put_all_variables(I,C,1),
  sql_exec_cursor(C,_,_),
  findall(TS, (sql_redo_fetch_row(C),sql_get_all_columns(TS,C,0)), TSL),
  sql_close_cursor(C), !,
  member(TS,TSL).

my_sql_sel(I,SQLStr,TS) :-
  (clause(tracing(on),true) -> format("~s;~n",[SQLStr]); write('.'), ttyflush),
  sql_open_cursor(C),
  sql_compile_statement(C,SQLStr),
  sql_create_all_columns(TS,C,0),
  sql_put_all_variables(I,C,1),
  sql_exec_cursor(C,_,_),
  findall(TS, (sql_redo_fetch_row(C),sql_get_all_columns(TS,C,0)), TSL),
  sql_close_cursor(C), !,
  member(TS,TSL).
  
