/******************************************************************************/
/* Module:      sql3_calendar.pl                                              */
/* Project:     TimeDB 1.03                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Date:        December 12, 1995                                             */
/* Export:                                                                    */
/*              sql3_granularity/1 (calendar.pl)                              */
/*              sql3_get_inc/2 (calendar.pl)                                  */
/*              sql3_to_internal/3 (calendar.pl)                              */
/*              internal_to_sql3/2 (calendar.pl)                              */
/*              sql3_span_to_internal/4 (calendar.pl)                         */
/*              internal_to_sql3_span/2 (calendar.pl)                         */
/*              sql3_min_number/1 (calendar.pl)                               */
/*              sql3_max_number/1 (calendar.pl)                               */
/* Import:                                                                    */
/*              generate_error_flag/2 (errors.pl)                             */
/*              get_tt_date/2 (time.pl)                                       */
/******************************************************************************/

sql3_calendar_start_year(0).

sql3_granularity(second).

sql3_get_inc(year, Inc)   :- Inc is 12*31*24*60*60.
sql3_get_inc(month, Inc)  :- Inc is 31*24*60*60.
sql3_get_inc(day, Inc)    :- Inc is 24*60*60.
sql3_get_inc(hour, Inc)   :- Inc is 60*60.
sql3_get_inc(minute, Inc) :- Inc is 60.
sql3_get_inc(second, Inc) :- Inc is 1.

/****************************************************************************/
/* Mode:	sql3_to_internal(-Seconds)				    */
/* Purpose:	Parses a date-string and converts it to an integer 	    */
/*              representing amount of seconds gone by since 1970/1/1       */
/* Example:								    */
/* Sideeffects:	None							    */
/* Call:        Exits always 						    */
/* Redo:        Fails always						    */
/****************************************************************************/

sql3_now(Now) --> [now], !, {get_tt_date(now, Now)}.
sql3_now(Now) --> [present], {get_tt_date(now, Now)}.

sql3_beginning(BEGINNING) --> [beginning], {sql3_min_number(BEGINNING), !}. 
sql3_forever(FOREVER)     --> [forever], {sql3_max_number(FOREVER), !}. 

sql3_from_year(Seconds) --> 
	[num(Year)],
	!,
	{sql3_calendar_start_year(Y),
	 (Year >= Y, !; generate_error_flag(invalid_year, Y))}, 
	sql3_from_year2(Year, Seconds).
sql3_from_year2(Years, Seconds) --> 
	[-],
	!, 
	{sql3_calendar_start_year(Y), Months is (Years-Y) * 12},
	sql3_from_month(Months, Seconds).
sql3_from_year2(Years, Seconds) --> 
	[],  
	{sql3_calendar_start_year(Y), 
	 Seconds is (((Years-Y)*12 + 1) * 31 + 1) * 24 * 60 * 60}.


sql3_from_month(M1, Seconds) --> 
	[num(Month)],
	!,
	{Month >= 0, Month < 13, M is M1 + Month}, 
	sql3_from_month2(M, Seconds).
sql3_from_month2(Months, Seconds) --> 
	[-], 
	{Days is Months*31},
	sql3_from_day(Days, Seconds).
sql3_from_month2(Months, Seconds) --> 
	[], 
	{Seconds is ((Months * 31) + 1) * 24 * 60 * 60}.

sql3_from_day(D, Seconds) --> 
	[num(Day)], 
	!,
	{Day >= 0, Day =< 31, Days is D + Day}, 
	sql3_from_day2(Days, Seconds).
sql3_from_day2(Days, Seconds) --> 
	{Hours is Days * 24},
	sql3_from_hours(Hours, Seconds),
	!.
sql3_from_day2(Days, Seconds) --> 
	[], 
	{Seconds is Days * 24 * 60 * 60}.

sql3_from_hours(H, Seconds) --> 
	[num(Hour)], 
	!,
	{Hour >= 0, Hour < 24, Hours is H + Hour}, 
	sql3_from_hours2(Hours, Seconds).
sql3_from_hours2(Hours, Seconds) --> 
	[:], 
	!,
	{Minutes is Hours * 60},
        sql3_from_minutes(Minutes, Seconds).
sql3_from_hours2(Hours, Seconds) --> 
	[], 
	{Seconds is Hours * 60 * 60}.

sql3_from_minutes(M, Seconds) --> 
	[num(Minute)], 
	!,
	{Minute >= 0, Minute < 60, Mins is M + Minute}, 
	sql3_from_minutes2(Mins, Seconds).
sql3_from_minutes2(Minutes, Seconds) --> 
	[:], 
	!,
	{Sec is Minutes * 60},
        sql3_from_seconds(Sec, Seconds).
sql3_from_minutes2(Minutes, Seconds) --> 
	[], 
	{Seconds is Minutes * 60}.

sql3_from_seconds(S, Seconds) --> 
	[num(Secs)], 
	{Secs >= 0, Secs < 60, Seconds is S+Secs}.


sql3_to_internal(Seconds) --> 
	sql3_now(Seconds),
        !.
sql3_to_internal(Seconds) --> 
	sql3_forever(Seconds),
        !.
sql3_to_internal(Seconds) --> 
	sql3_beginning(Seconds),
        !.
sql3_to_internal(Seconds) --> 
	sql3_from_year(Seconds),
	!,
 	{sql3_max_number(MaxSecs), Seconds < MaxSecs}.

/****************************************************************************/
/* Mode:	internal_to_sql3(+Internal, -Date_Literal)		    */
/* Purpose:	Converts an integer representing amount of seconds 	    */
/*              gone by since start year into a date-string                 */
/* Example:								    */
/* Sideeffects:	None							    */
/* Call:        Exists always						    */
/* Redo:        Fails always			 	                    */
/****************************************************************************/

sql3_to_chars(I, D, [D, 0'0|Str]) :- I < 10, !, number_chars(I, Str).
sql3_to_chars(I, D, [D|Str])      :- number_chars(I, Str).

sql3_to_seconds(Internal, Date) :-
 	S is Internal mod 60, S == 0,
	!,
	Rest is Internal // 60,
	sql3_to_minutes(S, Rest, Date).
sql3_to_seconds(Internal, Date) :-
	!,
 	S is Internal mod 60,
	sql3_to_chars(S, 58, Sec),
	Rest is Internal // 60,
	sql3_to_minutes(S, Rest, Min),
	append(Min, Sec, Date).

sql3_to_minutes(0, Internal, Date) :-
 	M is Internal mod 60, M == 0,
	!,
	Rest is Internal // 60,
	sql3_to_hours(0, Rest, Date).
sql3_to_minutes(_, Internal, Date) :-
	!,
 	M is Internal mod 60,
	sql3_to_chars(M, 58, Min),
	Rest is Internal // 60,
	sql3_to_hours(1, Rest, Hrs),
	append(Hrs, Min, Date).

sql3_to_hours(0, Internal, Date) :-
 	H is Internal mod 24, H == 0,
	!,
	Rest is Internal // 24,
	sql3_to_days(0, Rest, Date).
sql3_to_hours(_, Internal, Date) :-
	!,
 	H is Internal mod 24,
	sql3_to_chars(H, 32, Hrs),
	Rest is Internal // 24,
	sql3_to_days(1, Rest, Days),
	append(Days, Hrs, Date).

/*
sql3_to_days(0, Internal, Date) :-
 	D is Internal mod 31, D == 1,
	!,
	Rest is Internal // 31,
	sql3_to_months(0, Rest, Date).
*/
sql3_to_days(_, Internal, Date) :-
 	D is Internal mod 31, D == 0,
	!,
	sql3_to_chars(31, 45, Days),
	Rest is (Internal-31) // 31,
	sql3_to_months(1, Rest, Months),
	append(Months, Days, Date).
sql3_to_days(_, Internal, Date) :-
	!,
 	D is Internal mod 31,
	sql3_to_chars(D, 45, Days),
	Rest is Internal // 31,
	sql3_to_months(1, Rest, Months),
	append(Months, Days, Date).

/*
sql3_to_months(0, Internal, Date) :-
 	M is Internal mod 12, M == 1,
	!,
	Rest is Internal // 12,
	sql3_to_years(Rest, Date).
*/
sql3_to_months(_, Internal, Date) :-
 	M is Internal mod 12, M == 0,
	!,
	sql3_to_chars(12, 45, Months),
	Rest is (Internal-12) // 12,
	sql3_to_years(Rest, Years),
	append(Years, Months, Date).
sql3_to_months(_, Internal, Date) :-
	!,
 	M is Internal mod 12,
	sql3_to_chars(M, 45, Months),
	Rest is Internal // 12,
	sql3_to_years(Rest, Years),
	append(Years, Months, Date).

sql3_to_years(Internal, Date) :-
	!,
	sql3_calendar_start_year(Y), 
 	Year is Internal + Y,
	number_chars(Year, Date).


internal_to_sql3(Internal, "forever") :-
        number_chars(I, Internal), 
        sql3_max_number(I1), 
	I >= I1,
        !.
internal_to_sql3(Internal, "eot") :-
        number_chars(I, Internal), I < 0, !.
internal_to_sql3(Internal, "beginning") :-
        number_chars(I, Internal), 
        sql3_min_number(I1), 
	I =< I1, I >= 0,
        !.
internal_to_sql3(Internal, Literal) :-
	!,
        number_chars(I, Internal), 
        sql3_to_seconds(I, Literal).


/****************************************************************************/
/* Mode:	sql3_span_to_internal(-Seconds)			    */
/* Purpose:	Parses a duration-string and converts it to an integer 	    */
/*              representing amount of seconds                              */
/* Example:								    */
/* Sideeffects:	None							    */
/* Call:        Exits always				                    */
/* Redo:        Fails always						    */
/****************************************************************************/

sql3_chronons(year, Value, Chronons)   :- Chronons is Value *12*31*24*60*60.
sql3_chronons(month, Value, Chronons)  :- Chronons is Value *31*24*60*60.
sql3_chronons(day, Value, Chronons)    :- Chronons is Value *24*60*60.
sql3_chronons(hour, Value, Chronons)   :- Chronons is Value *60*60.
sql3_chronons(minute, Value, Chronons) :- Chronons is Value *60.
sql3_chronons(second, Chronons, Chronons).
 
sql3_span_to_internal(Q, Chronons) --> [num(V)], {sql3_chronons(Q, V, Chronons)}. 


/*****************************************************************************/
/* Mode:	internal_to_sql3_span(+Internal, -Span_Literal)            */
/* Purpose:	Converts an integer representing duration in seconds         */
/* Example:								     */
/* Sideeffects:	None							     */
/* Call:        Exists always						     */
/* Redo:        Fails always						     */
/*****************************************************************************/

internal_to_sql3_span(Span, SpanStr) :- 
	!, 
	number_chars(Span, Str), 
	append(Str, " second", SpanStr).


/*****************************************************************************/
/* Constants used in translate.pl (-inf, +inf)                               */
/*****************************************************************************/

sql3_min_number(0).
sql3_max_number(1000000000000).
