#include "core.h"

#include "mod_matrix.h"

IMPLEMENT_DYNAMIC(CMatrix, CAssociate)
IMPLEMENT_DYNAMIC(CMatrixRect, CRectangle)

///////////////////////////////////////////////////////
// Helper functions
//

// Split a matrix row
void SplitXRow(CMatrixRect * pRect, const CTime & x);
void SplitYRow(CMatrixRect * pRect, const CTime & x);

// Associate a rectangle to all CMatrixRect's that
// are contained in it.
void AssociateRect(CRectangle * a, CMatrixRect * pRect);
  void AssociateRow(CRectangle * a, CMatrixRect * pRect);

// Clear the lables of a matrix
void ClearLabels(CMatrixRect * pRect);

// Patch edges of a matrix
void MarkBlankX(CMatrixRect * a);
  bool MarkBlankRowX(bool legal, CMatrixRect * a);
void MarkColorX(CMatrixRect * a);
  bool MarkColorRowX(bool legal, CMatrixRect * a);
void MarkBlankY(CMatrixRect * a);
  bool MarkBlankRowY(bool legal, CMatrixRect * a);
void MarkColorY(CMatrixRect * a);
  bool MarkColorRowY(bool legal, CMatrixRect * a);

// Create the resulting rectangles and associte them to some object
void CreateBlank(CAssociate * pRes, CMatrixRect * a);
  void CreateBlankRow(CAssociate * pRes, CMatrixRect * a);
void CreateColor(CAssociate * pRes, CMatrixRect * a);
  void CreateColorRow(CAssociate * pRes, CMatrixRect * a);
void MarkAsIncluded(CMatrixRect * a, int x, int y);
  void MarkAsIncludedRow(CMatrixRect * a, int y);


///////////////////////////////////////////////////////
// Implementation of helper functions
//

void SplitXRow(CMatrixRect * pRect, const CTime & x){
  if(pRect->m_pNextY) SplitXRow(pRect->m_pNextY, x);
  // Create split
  CMatRectRef r = new CMatrixRect(x,pRect->y1,pRect->x2,pRect->y2);
  pRect->x2=x;
  // Adjust pointers
  r->m_pNextX = pRect->m_pNextX;
  pRect->m_pNextX = r;
  if(pRect->m_pNextY)
    r->m_pNextY = pRect->m_pNextY->m_pNextX;
  // Adjust associations
  StrictForEachAssociate(CRectangle, q, pRect){
    Associate(r,q);
  }
}


void SplitYRow(CMatrixRect * pRect, const CTime & y){
  if(pRect->m_pNextX) SplitYRow(pRect->m_pNextX, y);
  // Create split
  CMatRectRef r = new CMatrixRect(pRect->x1,y,pRect->x2,pRect->y2);
  pRect->y2=y;
  // Adjust pointers
  r->m_pNextY = pRect->m_pNextY;
  pRect->m_pNextY = r;
  if(pRect->m_pNextX)
    r->m_pNextX = pRect->m_pNextX->m_pNextY;
  // Adjust associations
  StrictForEachAssociate(CRectangle, q, pRect){
    Associate(r,q);
  }
}

void AssociateRect(CRectangle * a, CMatrixRect * pRect){
  if(pRect->m_pNextX) AssociateRect(a, pRect->m_pNextX);
  AssociateRow(a, pRect);
}

void AssociateRow(CRectangle * a, CMatrixRect * pRect){
  if(pRect->m_pNextY) AssociateRow(a, pRect->m_pNextY);
  if(Overlap(*a, *pRect)) Associate(a,pRect);
}

void ClearLabels(CMatrixRect * pRect){
  CString noColor;
  
  CMatrixRect * row = pRect;
  while(row){
    CMatrixRect * r = row;
    while(r){
      r->m_sColor = noColor;
      r->m_bKeepX = false;
      r->m_bKeepY = false;
      r->m_bIncluded = false;
      r=r->m_pNextX;
    }
    row=row->m_pNextY;
  }
}

//////////////////////
/*
 * The following 8 functions are similar
 */

void MarkBlankX(CMatrixRect * a){
  MarkBlankRowX(false,a);
  if(a->m_pNextX) MarkBlankX(a->m_pNextX);
}

bool MarkBlankRowX(bool legal, CMatrixRect * a){
  CMatrixRect * b = a->m_pNextX;
  if(!b) return false;
  if(IsBlankStarter(a,b)) a->m_bKeepX = true;
  else if(legal)
    if(IsBlankBorder(a,b)) a->m_bKeepX = true;
  if(a->m_pNextY)
    if(MarkBlankRowX(a->m_bKeepX, a->m_pNextY))
      if(IsBlankBorder(a,b)) a->m_bKeepX = true;
  return a->m_bKeepX;
}

void MarkColorX(CMatrixRect * a){
  MarkColorRowX(false,a);
  if(a->m_pNextX) MarkColorX(a->m_pNextX);
}

bool MarkColorRowX(bool legal, CMatrixRect * a){
  CMatrixRect * b = a->m_pNextX;
  if(!b) return false;
  if(IsColorStarter(a,b)) a->m_bKeepX = true;
  else if(legal)
    if(IsColorBorder(a,b)) a->m_bKeepX = true;
  if(a->m_pNextY)
    if(MarkColorRowX(a->m_bKeepX, a->m_pNextY))
      if(IsColorBorder(a,b)) a->m_bKeepX = true;
  return a->m_bKeepX;
}

void MarkBlankY(CMatrixRect * a){
  MarkBlankRowY(false,a);
  if(a->m_pNextY) MarkBlankY(a->m_pNextY);
}

bool MarkBlankRowY(bool legal, CMatrixRect * a){
  CMatrixRect * b = a->m_pNextY;
  if(!b) return false;
  if(IsBlankStarter(a,b)) a->m_bKeepY = true;
  else if(legal)
    if(IsBlankBorder(a,b)) a->m_bKeepY = true;
  if(a->m_pNextX) 
    if(MarkBlankRowY(a->m_bKeepY, a->m_pNextX))
      if(IsBlankBorder(a,b)) a->m_bKeepY = true;
  return a->m_bKeepY;
}

void MarkColorY(CMatrixRect * a){
  MarkColorRowY(false,a);
  if(a->m_pNextY) MarkColorY(a->m_pNextY);
}
bool MarkColorRowY(bool legal, CMatrixRect * a){
  CMatrixRect * b = a->m_pNextY;
  if(!b) return false;
  if(IsColorStarter(a,b)) a->m_bKeepY = true;
  else if(legal)
    if(IsColorBorder(a,b)) a->m_bKeepY = true;
  if(a->m_pNextX) 
    if(MarkColorRowY(a->m_bKeepY, a->m_pNextX))
      if(IsColorBorder(a,b)) a->m_bKeepY = true;
  return a->m_bKeepY;
}

void CreateBlank(CAssociate * pRes, CMatrixRect * a){
  CMatrixRect * row = a;
  while(row){
    CMatrixRect * col = row;
    while(col){
      if(!col->m_bIncluded){
        if(!col->IsColored()){
          CMatrixRect * p = col;
          int xDim=0;
          while((!col->m_bKeepX) && (col->m_pNextX)){
            col=col->m_pNextX;
            xDim++;
          }
          CMatrixRect * q = col;
          int yDim=0;
          while((!q->m_bKeepY) && (q->m_pNextY)){
            q=q->m_pNextY;
            yDim++;
          }
          CRectRef r = new CRectangle(p->x1,p->y1, q->x2, q->y2);
          Associate(pRes, r);
          MarkAsIncluded(p,xDim,yDim);
        }
      }
      col = col->m_pNextX;
    }
    row = row->m_pNextY;
  }
}

void CreateColor(CAssociate * pRes, CMatrixRect * a){
  CMatrixRect * row = a;
  while(row){
    CMatrixRect * col = row;
    while(col){
      if(!col->m_bIncluded){
        if(col->IsColored()){
          CMatrixRect * p = col;
          int xDim=0;
          while((!col->m_bKeepX) && (col->m_pNextX)){
            col=col->m_pNextX;
            xDim++;
          }
          CMatrixRect * q = col;
          int yDim=0;
          while((!q->m_bKeepY) && (q->m_pNextY)){
            q=q->m_pNextY;
            yDim++;
          }
          CRectRef r = new CRectangle(p->x1,p->y1, q->x2, q->y2);
          MarkAsIncluded(p,xDim,yDim);
          Associate(pRes, r);
          // For colored CMatrixRect's it is nice to know
          // which CRectangles decided their color.
          // Therefore we associate with the result the
          // generating rectangles.
          //   It will be possible to associate rows to 
          // the generating rectangles and thereby compute
          // aggregate values from CMatrixRects.
          StrictForEachAssociate(CRectangle, gen, p){
          	Associate(gen,r);
          }
        }
      }
      col = col->m_pNextX;
    }
    row = row->m_pNextY;
  }
}

void MarkAsIncluded(CMatrixRect * a, int x, int y){
  CMatrixRect * row = a;
  for(int i=0;i<=x;i++){
    CMatrixRect * col = row;
    for(int j=0;j<=y;j++){
      col->m_bIncluded = true;
      col = col->m_pNextY;
    }
    row = row->m_pNextX;
  }
}

///////////////////////////////////////////////////////////
// Implementation of CMatrix class functions

void CMatrix::Create(CRectangle * r){
  m_rRect = new CRectangle(*r);
  m_rMat = new CMatrixRect(*r);
}

void CMatrix::AddRects(CRectSet * pRects){
  StrictForEachAssociate(CRectangle, r, pRects){
    if(Overlap(*r, *m_rRect)){
      Associate(this, r);
      // Make sure that we can make a perfect coloring
      // by splitting the matrix along the edges of r.
      SplitX(r->x1);
      SplitX(r->x2);
      SplitY(r->y1);
      SplitY(r->y2);
      // Associate r with the CMatrixRect's that are
      // contained within r.
      AssociateRect(r, m_rMat);
    }
  }
}

void CMatrix::SplitX(const CTime & x){
  CMatrixRect * pRect = m_rMat;
  // go to the right row
  while(pRect && x>pRect->x2) pRect = pRect->m_pNextX;
  // split it
  if( pRect && (pRect->x1<x) && (x<pRect->x2) )
    SplitXRow(pRect, x);
}

void CMatrix::SplitY(const CTime & y){
  CMatrixRect * pRect = m_rMat;
  // go to the right row
  while(pRect && y>pRect->y2) pRect = pRect->m_pNextY;
  // split it
  if( pRect && (pRect->y1<y) && (y<pRect->y2) )
    SplitYRow(pRect, y);
}

void CMatrix::AssociateBlank(CAssociate * pRes){
  // 1. Clear labels
  ClearLabels(m_rMat);
  // 2. Colorize
  Colorize();
  // 3. Mark blank edges
  MarkBlankX(m_rMat); // mark x borders and
  MarkBlankY(m_rMat); // mark y borders.
  // 4. Create result
  CreateBlank(pRes,m_rMat);
}

void CMatrix::AssociateColor(CAssociate * pRes){
  // 1. Clear labels
  ClearLabels(m_rMat);
  // 2. Colorize
  Colorize();
  // 3. Mark colored edges
  MarkColorX(m_rMat); // mark x borders and
  MarkColorY(m_rMat); // mark y borders.
  // 4. Create result
  CreateColor(pRes,m_rMat);
}

void CMatrix::Colorize(){
  long l=0;
  CString sep(',');
  ForEachAssociate(CRectangle, r, this){
    CString name(l);
    ForEachAssociate(CMatrixRect, m, r){
      if(m->m_sColor.GetLength()!=0) m->m_sColor+= sep;
      m->m_sColor+= name;
    }
    l++;
  }
}
