/************************************************************************/
/*									*/
/* 		SWI Prolog 2.7.5 --- Oracle 7.2.2 Interface             */
/*									*/
/* Version:	1.0		      Language:    C	        	*/
/* Author:	Michael Boehlen	      System:	   UNIX 	       	*/
/* Created:	18.6.96 	      Machine:	   SUN	        	*/
/* 									*/
/* Module:	swi_ora.c						*/
/* Purpose:	SWI-Prolog Interface to Oracle 7.2.2		        */
/* Export:	init_ora	                                        */
/*                init_ora installs:		                        */
/*		    ora_open_db						*/
/*		    ora_close_db				       	*/
/* 		    ora_open_cursor	       				*/
/*		    ora_compile_statement			       	*/
/*		    ora_exec_cursor				       	*/
/*		    ora_close_cursor					*/
/*		    ora_fetch_row				       	*/
/*	            ora_put_var			                        */
/*	            ora_describe_column			       	        */
/*	            ora_get_column			       	        */
/*	            ora_commit			       	                */
/*	            ora_rollback			       	        */
/*	            ora_savepoint			       	        */
/*									*/
/************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "SWI-Prolog.h"

#include "oratypes.h"
#include "ocidfn.h"
#ifdef __STDC__
#include <ociapr.h>
#else
#include <ocikpr.h>
#endif
#include <ocidem.h>

/* Oracle errors */
#define NO_DATA_FOUND 1403
#define NULL_VALUE_FETCHED 1405
#define SINGLE_ROW_EXPECTED 1427
#define SELECT_COL_END 1007

#define PARSE_NO_DEFER 0
#define PARSE_DEFER 1
#define PARSE_V7_LNG 2

#define MaxCursors 200


/* 
 * rask removed the static keywork in order to make lda
 * accessible for the external modules as well...
 * 
 * See module.cc in 
 *   init_module()
 *
 */

Lda_Def *lda; 	/* database workspace */

#define MAX_IN_VARS 32
#define MAX_SELECT_COLS 32
#define MAX_COL_NAME_LENGTH 30
#define MAX_ITEM_SIZE 10000

typedef struct In_Var {
  void* val; /* pointer to the actual value */
  int idNr;  /* number of the input variable */
  int size;  /* size of the input value */
  int type;  /* Oracle internal type information */
} In_Var_Rec;

typedef struct Out_Var {
  sb1 colName[MAX_COL_NAME_LENGTH];
  sword colType;
  sb2 indic;
  void* val;  /* pointer to area where to put the output value */
  short rc;   /* return code for each column of a select list */
} Out_Var_Rec;

typedef struct Sql_Fetch {
  Cda_Def *cda;	                    /* cursor workspace */
  In_Var_Rec iv[MAX_IN_VARS];       /* array of input variables */
  Out_Var_Rec ov[MAX_SELECT_COLS];  /* array of output variables */
} *Sql_Fetch_Rec_Ptr;

Sql_Fetch_Rec_Ptr cursor[MaxCursors];

/********************** cursor handling *********************************/

void get_cursor(f)
  int *f;
/*    OUT    */
{
  *f = 0;
  while (*f < MaxCursors && cursor[*f] != NULL) (*f)++;
  if (*f >= MaxCursors) {
    printf("max number of open cursors exceeded (%d) \n",MaxCursors);
    fflush(stdout);
    exit(EXIT_FAILURE);
  }
  cursor[*f] = (Sql_Fetch_Rec_Ptr)malloc(sizeof(struct Sql_Fetch));
}

 /* --------------------------------- */

void free_cursor(f)
  int f;
/*    IN    */
{
  free(cursor[f]);
  cursor[f] = NULL;
}

/*************************** Error Handling ********************************/

static int ora_err(rc)
  short rc;
{
  char msg[256];

  if (rc != 0) {
    oermsg(rc,msg);
    printf("\n |%d| %s\n",rc,msg);
    fflush(stdout);
    return 0;
  }
  else return 1;
}

 /* --------------------------------- */

void swi_err(err)
  int err;
{
  if (err == 0) {
    printf("\n Interface error |%d|\n",err);
    fflush(stdout);
    exit(EXIT_FAILURE);
  }
}

/*********************** db handling ************************************/
     
foreign_t ora_open_db(dbname)
    term_t dbname;
/*  	   IN 		*/
{
  char* str;

  lda = (Lda_Def*)malloc(sizeof(Lda_Def));
  swi_err(PL_get_list_chars(dbname,&str,BUF_DISCARDABLE));
  olon(lda, (text*)str, (sword)-1, (text*)0, (sword)-1, (sword)-1);
  if (ora_err(lda->rc) == 0) PL_fail;
  ocof(lda);
  return ora_err(lda->rc);
}

/* --------------------------------- */

foreign_t ora_close_db()
{
  ologof(lda);
  free(lda);
  return ora_err(lda->rc);
}

/******************* cursor handling ***********************************/

foreign_t ora_open_cursor(cur)
    term_t cur;
/*	   OUT		*/
{
  int f,i;

  get_cursor(&f);
  cursor[f]->cda = (Cda_Def*)malloc(sizeof(Cda_Def));
  oopen(cursor[f]->cda, lda,(text*)0, (sword)-1, (sword)-1, (text*)0, (sword)-1); 			
  if (ora_err(cursor[f]->cda->rc) == 0) {
    free_cursor(f);
    PL_fail;
  }
  if (PL_unify_integer(cur,(long)f) == 0) {
    free_cursor(f);
    PL_fail;
  }
  for (i=0; i<MAX_SELECT_COLS; i++) cursor[f]->iv[i].val = NULL;
  for (i=0; i<MAX_SELECT_COLS; i++) cursor[f]->ov[i].val = NULL;
  PL_succeed;
}

/* --------------------------------- */

int prepare_query(f)
  int f;
/*    IN          */
{
  sb4 dbsize, dsize;
  sb2 dbtype, scale;
  sword colSize;
  sword i;

  for (i = 0; i < MAX_SELECT_COLS; i++) {
    if (odescr(cursor[f]->cda, i + 1, &dbsize, &dbtype, (sb1*)0, (sb4*)0,
               &dsize, (sb2*)0, &scale, (sb2*)0)) {
      if (cursor[f]->cda->rc == SELECT_COL_END) break;
      else ora_err(cursor[f]->cda->rc);
    }
    switch (dbtype) {
    case NUMBER_TYPE:
      if (scale != 0) {
	colSize = (sword)sizeof(float);
	cursor[f]->ov[i].val = (float*)malloc(colSize);
	cursor[f]->ov[i].colType = FLOAT_TYPE;
      } else {
	colSize = (sword)sizeof(sword);
	cursor[f]->ov[i].val = (sword*)malloc(colSize);
	cursor[f]->ov[i].colType = INT_TYPE;
      }
      break;
    case ROWID_TYPE:
      colSize = 18;
      cursor[f]->ov[i].val = (ub1*)malloc(colSize);
      cursor[f]->ov[i].colType = STRING_TYPE;
      break;
    default:
      if (dsize == 0) colSize = MAX_ITEM_SIZE; else colSize = dsize + 1;
      cursor[f]->ov[i].val = (ub1*)malloc(colSize);
      cursor[f]->ov[i].colType = STRING_TYPE;
    }
    odefin(cursor[f]->cda, i + 1, (ub1*)cursor[f]->ov[i].val, colSize,
           cursor[f]->ov[i].colType,
           (sword)-1, &cursor[f]->ov[i].indic, (text*)0, (sword)-1, (sword)-1,
           (ub2*)0, &(cursor[f]->ov[i].rc));
    if (ora_err(cursor[f]->cda->rc) == 0) PL_fail;
  }
  PL_fail;
}

 /* --------------------------------- */

foreign_t ora_compile_statement(cur,stat,stmt_type)
    term_t cur, stat, stmt_type;
/*	   IN,  IN,   OUT		*/
{
  int f;
  char* str;

  swi_err(PL_get_integer(cur,&f));
  swi_err(PL_get_list_chars(stat,&str,BUF_DISCARDABLE));
  if (ora_err(oparse(cursor[f]->cda, (text*)str, (sb4)-1,
                     (sword)PARSE_NO_DEFER, (ub4)PARSE_V7_LNG)) == 0) PL_fail;
  switch (cursor[f]->cda->ft) {
  case FT_SELECT:
    return PL_unify_atom_chars(stmt_type,"query");
    break;
  case FT_INSERT:
  case FT_DELETE:
  case FT_UPDATE:
    return PL_unify_atom_chars(stmt_type,"dml");
    break;
  default:
    return PL_unify_atom_chars(stmt_type,"other");
  }
}

 /* --------------------------------- */

foreign_t ora_exec_cursor(cur,rows)
    term_t cur, rows;
/*         IN   OUT      */
{
  int f,i;
 
  swi_err(PL_get_integer(cur,&f));
  i = 0;
  while (cursor[f]->iv[i].val != NULL) {
    obndrn(cursor[f]->cda,cursor[f]->iv[i].idNr,cursor[f]->iv[i].val,
           cursor[f]->iv[i].size,cursor[f]->iv[i].type,-1,0,0,-1,-1);
    if (ora_err(cursor[f]->cda->rc) == 0) PL_fail;
    i++;
  }
  if (cursor[f]->cda->ft == FT_SELECT) prepare_query(f);
  if (ora_err(oexec(cursor[f]->cda)) == 0) PL_fail;
  return PL_unify_integer(rows,(long)cursor[f]->cda->rpc);
}

/* --------------------------------- */

foreign_t ora_close_cursor(cur)
    term_t cur;
/*	    IN		*/
{
  int f,i;

  swi_err(PL_get_integer(cur,&f));
  oclose(cursor[f]->cda);
  if (ora_err(cursor[f]->cda->rc) == 0) PL_fail;
  i = 0;
  while (cursor[f]->iv[i].val != NULL) {
    free(cursor[f]->iv[i].val);
    i++;
  }
  i = 0;
  while (cursor[f]->ov[i].val != NULL) {
    free(cursor[f]->ov[i].val);
    i++;
  }
  free(cursor[f]->cda);
  free_cursor(f);
  PL_succeed;
}

/******************* header handling *************************************/

foreign_t ora_describe_column(cur,pos,val)
    term_t cur, pos, val;
/*	   IN,  IN,  OUT		*/
{
  int f,i;
  sb4 dbsize;
  sb4 resl = 30;
  sb1 res[30];

  swi_err(PL_get_integer(cur,&f));
  swi_err(PL_get_integer(pos,&i));
  if (odescr(cursor[f]->cda, i + 1, &dbsize, (sb2*)0, res, &resl,
             (sb4*)0, (sb2*)0, (sb2*)0, (sb2*)0)) {
    if (cursor[f]->cda->rc == SELECT_COL_END) PL_fail;
    else ora_err(cursor[f]->cda->rc);
  }
  res[resl] = '\0';
  return PL_unify_list_chars(val,res);
}

/******************* column handling *************************************/

foreign_t ora_get_column(cur,pos,val)
    term_t cur, pos, val;
/*	   IN,  IN,  OUT		*/
{
  int f,i;

  swi_err(PL_get_integer(cur,&f));
  swi_err(PL_get_integer(pos,&i));
  if (cursor[f]->ov[i].indic == -1) return PL_unify_atom_chars(val,"null");
  if (cursor[f]->ov[i].val == NULL) PL_fail;
  switch (cursor[f]->ov[i].colType) {
  case INT_TYPE:
    return PL_unify_integer(val,*(long*)cursor[f]->ov[i].val);  
  case FLOAT_TYPE:
    return PL_unify_float(val,*(float*)cursor[f]->ov[i].val);
  default:
    return PL_unify_list_chars(val,(char*)cursor[f]->ov[i].val);
  }
}

/*********************** variable handling *****************************/

foreign_t ora_put_var(cur,pos,val)
    term_t cur, pos, val;
/*	   IN,  IN,  IN		*/
{
  int f,i,j;
  
  swi_err(PL_get_integer(cur,&f));
  swi_err(PL_get_integer(pos,&i));
  j = 0;
  while (j<MAX_IN_VARS && cursor[f]->iv[j].val!=NULL && cursor[f]->iv[j].idNr!=i) j++;
  if (j >= MAX_IN_VARS) {
    printf("\nOCI error: number of input variables exceeded\n\n");
    fflush(stdout);
    PL_fail;
  }
  cursor[f]->iv[j].idNr = i;
  free(cursor[f]->iv[j].val);
  switch (PL_term_type(val)) {
  case PL_INTEGER:
    cursor[f]->iv[j].val = (long*)malloc(sizeof(long));
    swi_err(PL_get_integer(val,cursor[f]->iv[j].val));
    cursor[f]->iv[j].size = sizeof(long);
    cursor[f]->iv[j].type = INT_TYPE;
    break;
  case PL_FLOAT:
    cursor[f]->iv[j].val = (double*)malloc(sizeof(double));
    swi_err(PL_get_float(val,cursor[f]->iv[j].val));
    cursor[f]->iv[j].size = sizeof(double);
    cursor[f]->iv[j].type = FLOAT_TYPE;
    break;
  default: /* a string respectively a list */
    swi_err(PL_get_list_chars(val,(char**)&cursor[f]->iv[j].val,BUF_MALLOC));
    cursor[f]->iv[j].size = strlen((char*)cursor[f]->iv[j].val);
    cursor[f]->iv[j].type = 1;
  }
  PL_succeed;
}

/********************** row handling **********************************/

foreign_t ora_fetch_row(cur)
    term_t cur;
/*         IN     */
{
  int f;
  
  swi_err(PL_get_integer(cur,&f));
  if (ofetch(cursor[f]->cda) == 0) PL_succeed;
  else {
    if (cursor[f]->cda->rc != NO_DATA_FOUND) ora_err(cursor[f]->cda->rc);
    PL_fail;
  }
}

/********************** transaction handling **************************/

foreign_t ora_enable_autocommit()
{
  ocon(lda);
  if (ora_err(lda->rc) == 0) PL_fail;
  PL_succeed;
}

/* --------------------------------- */

foreign_t ora_disable_autocommit()
{
  ocof(lda);
  if (ora_err(lda->rc) == 0) PL_fail;
  PL_succeed;
}

/* --------------------------------- */

foreign_t ora_commit()
{
  ocom(lda);
  if (ora_err(lda->rc) == 0) PL_fail;
  PL_succeed;
}

/* --------------------------------- */

foreign_t ora_rollback()
{
  orol(lda);
  if (ora_err(lda->rc) == 0) PL_fail;
  PL_succeed;
}

/************************  init_ora ****************************/

foreign_t init_ora()
{
  int i;
  printf("installing predicates to access ORACLE 8.0.4\n");
  fflush(stdout);
  for (i=0; i<MaxCursors; i++) cursor[i] = NULL;
  PL_register_foreign("ora_open_db",1,ora_open_db,0); 
  PL_register_foreign("ora_close_db",0,ora_close_db,0); 
  PL_register_foreign("ora_open_cursor",1,ora_open_cursor,0); 
  PL_register_foreign("ora_compile_statement",3,ora_compile_statement,0); 
  PL_register_foreign("ora_exec_cursor",2,ora_exec_cursor,0); 
  PL_register_foreign("ora_close_cursor",1,ora_close_cursor,0); 
  PL_register_foreign("ora_get_column",3,ora_get_column,0); 
  PL_register_foreign("ora_describe_column",3,ora_describe_column,0); 
  PL_register_foreign("ora_put_var",3,ora_put_var,0); 
  PL_register_foreign("ora_fetch_row",1,ora_fetch_row,0); 
  PL_register_foreign("ora_enable_autocommit",0,ora_enable_autocommit,0); 
  PL_register_foreign("ora_disable_autocommit",0,ora_disable_autocommit,0); 
  PL_register_foreign("ora_commit",0,ora_commit,0); 
  PL_register_foreign("ora_rollback",0,ora_rollback,0); 
  PL_succeed;
}

/* compilation
  gcc -I/user/boehlen/lib/pl-2.7.16/include -I/pack/oracle/rdbms/demo -c swi_ora.c
  gcc -shared -o swi_ora.so swi_ora.o -L/pack/oracle/lib/libclntsh.so
*/

/* ******************************************

./pl-ora
init_ora.
ora_open_db("boehlen/boehlen").

ora_open_cursor(C), write(C), nl, fail.
ora_compile_statement(0,"create table q(a integer)").
ora_exec_cursor(0,Num), format("~w ~w~n",[Num]), fail.
ora_close_cursor(0).

ora_open_cursor(C), write(C), nl, fail.
ora_compile_statement(0,"insert into q values (8)").
ora_exec_cursor(0,Num), format("~w ~w~n",[Num]), fail.
ora_close_cursor(0).

ora_open_cursor(C), write(C), nl, fail.
ora_compile_statement(0,"drop table q").
ora_exec_cursor(0,Num), format("~w ~w~n",[Num]), fail.
ora_close_cursor(0).

ora_open_cursor(C), write(C), nl, fail.
ora_compile_statement(0,"select * from cat").
ora_exec_cursor(0,Num), format("~w ~w~n",[Num]), fail.
ora_fetch_row(0).
ora_get_column(0,0,Value), format("~s ~w~n",[Value]), fail.
ora_get_column(0,1,Value), format("~s ~w~n",[Value]), fail.
ora_close_cursor(0).

ora_open_cursor(C), write(C), nl, fail.
ora_compile_statement(0,"select a0.a from p a0").
ora_exec_cursor(0,Num), format("~w ~w~n",[Num]), fail.
ora_fetch_row(0).
ora_get_column(0,0,Value), format("~w ~w~n",[Value]), fail.
ora_close_cursor(0).

****************************************** */
