/*!****************************************************************************/
/*! File:      interpret.pl                                                   */
/*! Project:   Tiger                                                          */
/*! Software:  SWI Prolog 2.7.16, Oracle 8.0.4                                */
/*! Export:                                                                   */
/*!            atsql/0 (tiger.pl)                                             */
/*! Import:                                                                   */
/*!            check_option/1 (tiger.pl)                                      */
/*!            set_of/3 (tiger.pl)                                            */
/*!            s_readstring/1 (scanner.pl)                                    */
/*!            s_scan/2 (scanner.pl)                                          */
/*!            p_parse/2 (parser.pl)                                          */
/*!            u_stat/3 (unparser.pl)                                         */
/*!            r_norm_query_expr/3 (rewrite.pl)                               */
/*!            r_norm_delete/6 (rewrite.pl)                                   */
/*!            r_norm_update/8 (rewrite.pl)                                   */
/*!            r_norm_create/5 (rewrite.pl)                                   */
/*!            r_norm_assertion/3 (rewrite.pl)                                */
/*!            m_update_cols/2 (meta.pl)                                      */
/*!            m_table_type/3 (meta.pl)                                       */
/*!            m_sql_exec/5 (meta.pl)                                         */
/*!            v_uc2tuc/1 (views.pl)                                          */
/*!            d_get_deps/2 (deps.pl)                                         */
/*!            d_check_deps/0 (deps.pl)                                       */
/*!            d_check_meta_oper/0 (deps.pl)                                  */
/*!            d_process_deps/0 (deps.pl)                                     */
/*!            e_insert/5 (eval.pl)                                           */
/*!            e_delete/6 (eval.pl)                                           */
/*!            e_update/7 (eval.pl)                                           */
/*!            t_trans_query/3 (trans.pl)                                     */
/*!            c_add_constraints/1 (constraint.pl)                            */
/*!            c_uc2tuc/1 (constraint.pl)                                     */
/*!            c_check_ics/0 (constraint.pl)                                  */
/*!            f_raise_err/2 (fejl.pl)                                        */
/*!            f_print_err/0 (fejl.pl)                                        */
/*!            z_inc_current/0 (zeit.pl)                                      */
/*!            z_get_current/1 (zeit.pl)                                      */
/*!****************************************************************************/

close_cursors :-
  retract(cursor(C)),
    ora_close_cursor(C),
    fail
    ;
    true.

/******************************************************************************/
/* Mode:	disp(+StatementType, +Result)                                 */
/* Purpose:	displays the result of a statement                            */
/* Sideeffects:	result written to stdout                                      */
/* Call:        Exists always						      */
/* Redo:        Fails always                                                  */
/******************************************************************************/

out(X,CSep) :- is_list(X), !, format("~s~s",[X,CSep]).
out(X,CSep) :- format("~w~s",[X,CSep]).

disp(query, header-Tpl) :- !,
  nl, nl,
  disp(query, Tpl), 
  format("~40c~n", [0'-]).
disp(query, Tpl) :-
  clause(terminal_ctrl(CSep,RSep),true),
  (member(X, Tpl),
     out(X,CSep),
     fail;
     format("~s",[RSep])).
disp(dml, 1) :- !,
  (  clause(verbose,true)
  -> format("1 tuple processed~n",[])
  ;  format("~n1 tuple processed~n",[])
  ).
disp(dml, NumTpls) :- 
  (  clause(verbose,true)
  -> format("~d tuples processed~n",[NumTpls])
  ;  format("~n~d tuples processed~n",[NumTpls])
  ).
disp(other, _) :-
  (  clause(verbose,true)
  -> format("statement processed~n",[])
  ;  format("~nstatement processed~n",[])
  ).

/******************************************************************************/
/* Mode:	atsql                                                         */
/* Purpose:	Interpreter-loop                                              */
/* Sideeffects:	Results or error messages are printed                         */
/* Call:        Exists always				                      */
/* Redo:        Fails always				      		      */
/******************************************************************************/

%% interpret(+ParseTree, +InputString)
/* statements without database access */
interpret(quit,_) :- !.
interpret(verbose(on), _)  :- !,
  assert(verbose),
  pl_set_verbose("Y"), 
  format("~nSwitched to verbose mode~n",[]).
interpret(verbose(off), _) :- !,
  retractall(verbose),
  pl_set_verbose("N"),
  format("~nVerbose mode switched off~n",[]).
interpret(change(sysdate,T), _) :-
  retractall(sysdate(_)),
  ( T == now -> true; assert(sysdate(T))),
  format("~nsysdate changed~n",[]).
interpret(open(Login),_) :- !,
  (  clause(db_is_open,true)
  -> f_raise_err(db_already_open,_)
  ;  ora_open_db(Login),
     m_sql_exec([], lit("ALTER SESSION SET NLS_DATE_FORMAT='SYYYY/MM/DD~HH24:Mi:SS'"),
                nil, other, _),
     z_get_current(Curr),
     assert(transaction_status([],Curr,[])),
     assert(db_is_open),
     format("~nDatabase opened~n",[])
  ).
interpret(_, _) :-
  (  clause(db_is_open,true)
  -> fail
  ;  !, f_raise_err(no_db_open, _)
  ).

/* miscellaneous statements */
interpret(batch(In,Out), _) :- !,
  atom_chars(I,In),
  atom_chars(O,Out),
  (  access_file(I,read)
  -> true
  ;  f_raise_err(file_err,I)
  ),
  (O == user -> see(I); see(I), tell(O)).
interpret(backend(Str), _) :- !,
  format("~n*** WARNING you bypass Tiger at your own risk!! ***",[]),
  format("~n*** This may corrupt the integrity of the DB!! ***~2n",[]),
  (m_sql_exec([], lit(Str), nil, StmtType, Res),
     disp(StmtType, Res),
     fail
     ;
     true).
interpret(close, _) :- !,
  close_cursors,
  ora_close_db,
  retractall(db_is_open),
  retractall(transaction_status(_,_,_)),
  format("~nDatabase closed~n",[]).
interpret(eof, _) :- seen, told.

/* transaction handling */
interpret(check,_) :- !,
  c_check_ics,
  format("~nNo integrity constraints violated~n",[]).	
interpret(commit,_) :- !,
  z_get_current(Curr),
  (  c_check_ics
  -> (  ora_commit
     -> retract(transaction_status(_XMRS,_LCT,_)),
        %patch_tt(XMRS,LCT,Curr),
        assert(transaction_status([],Curr,[])),
        format("~nTransaction commited~n",[])
     ;  retract(transaction_status(_,_,_)),
        assert(transaction_status([],Curr,[])),
        format("~Unsuccessful backend commit; transaction rolled back~n",[]),
        fail
     )
  ;  ora_rollback,
     retract(transaction_status(_,_,_)),
     assert(transaction_status([],Curr,[])),
     format("~nTransaction rolled back because of violated constraints~n",[]),
     fail
  ).
interpret(rollback,_) :- !,
  ora_rollback,
  retract(transaction_status(_,LCT,_)),
  assert(transaction_status([],LCT,[])),
  format("~nRollback completed~n",[]).

/* handling SQL statements */
interpret(_,String) :-
  \+ clause(tiger_flag, true), !,
    d_check_meta_oper, 
    d_check_deps,
    d_process_deps,
    format("*** Executing an UC statement ***",[]),
    (clause(verbose,true) -> nl; true),
    (m_sql_exec([], lit(String), nil, StmtType, Res),
       disp(StmtType, Res),
       fail
       ;
       true).

/* queries */
interpret(query(Flag, QExpr, Coal, OrderBy),_) :- !,
  r_norm_query_expr(QExpr, Flag, QExprN),
  t_trans_query(query(Flag,QExprN,Coal,OrderBy), TypeList, QueryStmt),
  (m_sql_exec([], QueryStmt, TypeList, query, Res),
     disp(query, Res),
     fail
     ;
     true).

/* data manipulation commands */
interpret(dml(Flag,insert(TabName,ColNames,QueryExpr)), _) :- !,
  r_norm_query_expr(QueryExpr, Flag, QueryExpr1),
  d_check_meta_oper,
  d_process_deps,
  e_insert(TabName, ColNames, Flag, QueryExpr1, NumTpls),
  format("~n~d tuple(s) inserted~n", [NumTpls]).
interpret(dml(Flag,delete(TabName,TabAlias,Where)),_) :- !,
  r_norm_delete(Flag, TabName, TabAlias, Where, Flag1, Where1),
  d_check_meta_oper,
  d_process_deps,
  m_table_type(TabName, [], TabType),
  e_delete(TabType, TabName, TabAlias, Flag1, Where1, NumTpls),
  format("~n~d tuple(s) deleted~n", [NumTpls]).
interpret(dml(Flag,update(TabName,TabAlias,Set,Where)), _) :- !,
  r_norm_update(Flag, TabName, TabAlias, Set, Where, Flag1, Set1, Where1),
  d_check_meta_oper,
  d_process_deps,
  m_table_type(TabName, [], TabType),
  e_update(TabType, TabName, TabAlias, Flag1, Set1, Where1, NumTpls),
  format("~n~d tuple(s) updated~n", [NumTpls]).

/* data definition commands (create) */
interpret(create_table(TabName,ColDefs,TabType), _) :- !,
  r_norm_create(ColDefs, TabName, TabType, ColDefs1, TDB_ICs),
  (  TabType == sn
  -> ColDefs2 = ColDefs1
  ;  TabType == vt
  -> ColDefs2 = [colDef("vt",per,nil,ic(nil,nil,nil))|ColDefs1]
  ;  TabType == tt
  -> ColDefs2 = [colDef("tt",per,nil,ic(nil,nil,nil))|ColDefs1]
  ;  /* else: TabType == bi */
     ColDefs2 = [colDef("tt",per,nil,ic(nil,nil,nil)),
                 colDef("vt",per,nil,ic(nil,nil,nil))|ColDefs1]
  ),
  m_sql_exec([], create_table(TabName,ColDefs2,sn), nil, other, _),
  c_add_constraints(TDB_ICs),
  ora_commit, /* Oracle transactions do not allow to mix DDL and DML */
  format("~nTable created~n",[]).
interpret(create_view(_ViewName,_ColNames,Flag,QueryExpr), _String) :- !,
  r_norm_query_expr(QueryExpr, Flag, QExpr1),
  m_table_type(derived_table(Flag,QExpr1), [], _ViewType),
%  e_create_view(ViewName, ColNames, Flag, QExpr1, ViewType, String),
  format("~nView created~n",[]).
interpret(create_assertion(AssN,Flag,Cond),_) :- !,
  r_norm_assertion(Flag, Cond, QueryExpr),
  set_of(R, d_get_deps(QueryExpr,R), Rels),
  ( member(Object-IsView, Rels),
      m_sql_exec([var(AssN,s(_)),var(Object,s(_)),var(IsView,s(1))],
         lit("INSERT INTO TDB$DEPS VALUES(:1,'ASS',:2, :3, 'Y')"), nil, dml, _),
      fail
      ;
      true
  ),
  u_stat(query(Flag,QueryExpr,nil,[]), QueryStr, []),
  m_sql_exec([var(AssN,s(_)),var(QueryStr,s(_))],
           lit("INSERT INTO TDB$ICS VALUES(:1, :2)"), nil, dml, _),
  retract(transaction_status(XMRS,LCT,ICs)),
  assert(transaction_status(XMRS,LCT,[AssN|ICs])),
  !,
  format("~nAssertion added~n",[]).

/* data definition commands (drop) */
interpret(drop_table(TabName), _) :- !,
  d_check_meta_oper,
  d_check_deps,
  d_process_deps,
  m_sql_exec([], drop_table(TabName), nil, other, _),
  format("~nTable dropped~n",[]).
interpret(drop_view(_ViewName), _) :- !,
        format("~nView dropped~n",[]).
interpret(drop_assertion(AssN), _) :- !,
  m_sql_exec([var(AssN,s(_))], lit("DELETE FROM TDB$DEPS WHERE OBJECT=:1"),
           nil, dml, _),
  m_sql_exec([var(AssN,s(_))], lit("DELETE FROM TDB$ICS WHERE NAME=:1"),
           nil, dml, _),
  format("~nAssertion dropped~n",[]).

/* schema modifications */
interpret(alter(TabName,add_tt), _) :- !,
  m_table_type(TabName, [], TabType),
  (  memberchk(TabType,[tt,bi])
  -> f_raise_err(altering_tt_table, _)
  ;  ( TabType==vt -> true ; c_uc2tuc(TabName) ),
     z_get_current(Curr),
     m_update_cols(TabName, "TT$"),
     m_sql_exec([], alter(TabName,add([colDef("TT",per,period(Curr,now),[])])),
              nil, other, _)
  ),
  v_uc2tuc(TabName),
  format("~nTable altered~n",[]).
interpret(alter(TabName,add_vt), _) :- !,
  m_table_type(TabName, [], TabType),
  (  memberchk(TabType,[vt,bi])
  -> f_raise_err(altering_vt_table, _)
  ; ( TabType==tt -> true ; c_uc2tuc(TabName) ),
     z_get_current(Curr),
     m_update_cols(TabName, "VT$"),
     m_sql_exec([], alter(TabName,add([colDef("VT",per,period(Curr,now),[])])),
              nil, other, _)
  ),
  v_uc2tuc(TabName),
  format("~nTable altered~n",[]).
interpret(alter(TabName,Action), _) :- !,
  m_sql_exec([], alter(TabName,Action), nil, other, _),
  format("~nTable altered~n",[]).

/* session parameters */
interpret(alter_session(nls_date_format,_), _) :- !,
  f_raise_err(nls_date_format, _).
interpret(alter_session(nls_period_sep,V), _) :- !,
  (  V=[C],
     C\==0'/, C\==0'~, C\==0':,
     \+ (C >= 0'a, C =< 0'z),
     \+ (C >= 0'A, C =< 0'Z),
     \+ (C >= 0'0, C =< 0'9)
  -> retractall(nls_period_sep(_,_)),
     atom_chars(A,V),
     assert(nls_period_sep(C,A)),
     format("~nPeriod separator altered~n",[])
  ;  f_raise_err(nls_period_sep,_)
  ).

echo(Str) :-
  (  check_option('-e') 
  -> (  append("open   ",_,Str)
     -> format("opening a database...~n",[])
     ;  append("OPEN   ",_,Str)
     -> format("opening a database...~n",[])
     ;  Str=backend(Str2)
     -> format("~s~n",[Str2])
     ;  format("~s~n",[Str])
     )
  ;  true
  ).

translate(Str, ParseTree) :-
  s_scan(Str, Tokens),
  p_parse(Tokens, ParseTree).	

atsql :-
  repeat,
    close_cursors,
    retractall(error(_,_)),
    retractall(candidate_action(_,_)),
    retractall(tiger_flag),
    retractall(cols(_,_)),
    s_readstring(Str),
    echo(Str),
    s_scan(Str, Tokens),
    p_parse(Tokens, ParseTree),
    ( interpret(ParseTree, Str) -> nl ;  f_print_err ),
    z_inc_current,
    %%    print_flags,
    ParseTree == quit.

