/*!****************************************************************************/
/*! File:      parser.pl                                                      */
/*! Project:   Tiger                                                          */
/*! Software:  SWI Prolog 2.7.16, Oracle 8.0.4                                */
/*! Export:                                                                   */
/*!            p_parse/2 (interpret.pl,views.pl,constraint.pl)                */
/*! Import:                                                                   */
/*!            m_table_type/3 (meta.pl)                                       */
/*!            f_raise_err/2 (fejl.pl)                                        */
/*!            z_p_per/2 (zeit.pl)                                            */
/*!            z_p_ts/2 (zeit.pl)                                             */
/*!****************************************************************************/

p_chk(X)   --> [X], !.
p_chk(X)   --> [Y], {f_raise_err(expected_instead, X-Y)}.

p_id(X, _) --> [id(X)], !.
p_id(_, M) --> [Y], {f_raise_err(M, Y)}.

/******************************** time flags **********************************/

p_time_mode(ns)  --> [nonsequenced], !, {assert(tiger_flag)}.
p_time_mode(seq) --> [sequenced], !, {assert(tiger_flag)}.

p_dimension(vt) --> [valid], !, {assert(tiger_flag)}.
p_dimension(tt) --> [transaction], !, {assert(tiger_flag)}.

p_domain(D)   --> p_expr(D,ari), !.
p_domain(nil) --> [].

p_range(P)   --> [set], !, p_chk(valid), {assert(tiger_flag)}, p_expr(P,ari).
p_range(nil) --> [].

instantiate(vt, Mode, Dom, Mode, Dom, _, _) :- !.
instantiate(tt, Mode, Dom, _, _, Mode, Dom) :- !.
instantiate(vt, _, _, _, _, _, _) :- f_raise_err(vt_overspecified, _), !.
instantiate(tt, _, _, _, _, _, _) :- f_raise_err(tt_overspecified, _), !.

p_second_flag(vt(VTM,VTD), tt(TTM,TTD)) -->
  [and], !,
  p_time_mode(Mod),
  p_dimension(Dim), !,
  p_domain(Dom),
  {instantiate(Dim, Mod, Dom, VTM, VTD, TTM, TTD)}.
p_second_flag(vt(tuc,nil), _) --> [], !.
p_second_flag(_, tt(tuc,nil)) --> [], !.

p_first_flag(flag(vt(VTM,VTD),tt(TTM,TTD),VTR)) -->
  p_time_mode(Mod),
  p_dimension(Dim), !,
  p_domain(Dom),
  {instantiate(Dim, Mod, Dom, VTM, VTD, TTM, TTD)},
  p_second_flag(vt(VTM,VTD), tt(TTM,TTD)),
  p_range(VTR).
p_first_flag(flag(vt(tuc,nil),tt(tuc,nil),VTR)) --> p_range(VTR).

p_flag(Flag, yes) --> ['('], p_first_flag(Flag).
p_flag(Flag, no)  --> p_first_flag(Flag).

/******************************* top level commands ***************************/

p_atsql(quit)                   --> [quit], !.
p_atsql(verbose(X))             --> [verbose], !, p_on_off(X). 
p_atsql(open(S))                --> [open], !, p_chk(str(S)).
p_atsql(_)                      --> {  clause(db_is_open,true)
                                    -> fail
                                    ;  !,
                                       f_raise_err(no_db_open, _)
                                    }.
p_atsql(batch(I,O))             --> [batch], !, p_chk(str(I)), p_chk(str(O)).
p_atsql(close)                  --> [close], !.
p_atsql(eof)                    --> [id(eof)], !.
p_atsql(commit)                 --> [commit], !.
p_atsql(check)                  --> [check], !.
p_atsql(rollback)               --> [rollback], !.
p_atsql(change(sysdate,P))      -->
  [change],
  !,
  {assert(tiger_flag)},
  p_chk(sysdate),
  p_chk(to),
  [TP],
  {z_p_ts(TP,P)}.
p_atsql(alter_session(P,V))   -->
   [alter],
   [session],
   !,
   p_chk(set),
   p_session_par(P),
  {assert(tiger_flag)},
   p_chk(eq),
   p_chk(str(V)).
p_atsql(alter(T,A))             -->
  [alter],
  !,
  p_chk(table), 
  p_id(T, table_name_expected),
  p_alter_cmd(A).
p_atsql(dml(Flag, DMLStmt))     --> p_dml(Flag, DMLStmt), !.
p_atsql(DDLStmt)                --> p_ddl(DDLStmt), !.
p_atsql(query(Flags, QueryExpr, Coal, OrderBy)) -->
  p_query(Flags, QueryExpr, Coal), !,
  p_orderby(OrderBy).

p_on_off(on)               --> [on], !.
p_on_off(off)              --> [off], !.
p_on_off(_), [Y]           --> [Y], {f_raise_err(on_off_expected, Y)}.

/***************************** query expressions ******************************/

update_coal(F+S, S, F+S).
update_coal(_+F, S, F+S).

p_coal(F+S, Coal) -->
  ['('],
  !,
  p_dimension(D),
  p_chk(')'),
  {update_coal(F+S, D, Coal1)},
  p_coal(Coal1, Coal).
p_coal(Coal, Coal) --> [].

p_query(Flag, QueryExpr, Coal) --> 
  p_flag(Flag, LPar), 
  p_query_expr(QueryExpr),
  ({LPar==yes} -> p_chk(')'); {true}),
  p_coal(nil+nil, Coal).

p_query_expr(Query) --> 
  p_query_term(Term),
  p_query_expr2(Term, Query).

p_query_expr2(T1, Query) --> 
  [union],
  !,
  p_query_term(T2),
  p_query_expr2(union(T1,T2), Query), !.
p_query_expr2(T1, Query) --> 
  [except],
  !,
  p_query_term(T2),
  p_query_expr2(except(T1,T2), Query), !.
p_query_expr2(Query, Query) --> [].


p_query_term(Term) --> 
  p_query_fac(Factor),
  p_query_term2(Factor, Term).

p_query_term2(F1, Term) --> 
  [isct],
  !,
  p_query_fac(F2),
  p_query_term2(isct(F1,F2), Term), !.
p_query_term2(Factor, Factor) --> [].

p_query_fac(Exp) --> 
  ['('],
  p_query_expr(Exp), 
  p_chk(')').
p_query_fac(sel(Mode, Select, From, Where, Grouping, Having)) -->
  p_select(Mode, Select),
  p_from(From),
  p_where(Where),
  p_groupby(Grouping),
  p_having(Having).

/***************************** select clause **********************************/

p_select(Mode, Select) --> 
  [select],
  !,
  p_mode(Mode),
  p_sel_col(Select).

p_mode(all)      --> [all], !.
p_mode(distinct) --> [distinct], !.
p_mode(all)      --> [].

p_sel_col([SelectItem|SelectList]) -->
  select_item(SelectItem), 
  p_sel_col2(SelectList).

p_sel_col2([SelectItem|SelectList]) -->
  [','], !, 
  select_item(SelectItem), 
  p_sel_col2(SelectList).
p_sel_col2([]) --> [].

select_item(sel_col(all,_))        --> ['*'], !.
select_item(sel_col(all(Table),_)) --> [id(Table)], ['.'], ['*'], !.
select_item(sel_col(Exp,Alias))    --> p_expr(Exp,ari), p_alias(_, Alias), !.
select_item(_) --> [X], {f_raise_err(select_item_expected, X)}. 

%% p_alias(+DefaultAlias, -Alias)
p_alias(_, Alias)         --> [as], !, p_id(Alias, alias_expected).
p_alias(_, Alias)         --> [id(Alias)], !.
p_alias(Default, Default) --> [].

/******************************** from clause *********************************/

%% p_tab_ref(+TableRefType, -TableDef)
p_tab_ref(extended, tab_ref(derived_tab(Flag,QueryExpr),TabAlias,Coal)) -->
  ['('],
  !,
  p_flag(Flag, no), 
  p_query_expr(QueryExpr),
  p_chk(')'),
  p_coal(nil+nil, Coal),
  p_alias(nil, TabAlias).
p_tab_ref(extended, tab_ref(TabN,TabAlias,Coal)) -->
  p_id(TabN, table_name_expected),
  { m_table_type(TabN,[sn],_) -> assert(tiger_flag); true },
  p_coal(nil+nil, Coal),
  p_alias(_, TabAlias).
p_tab_ref(restricted, tab_ref(TabN,TabAlias,nil+nil)) -->
  p_id(TabN, table_name_expected),
  {m_table_type(TabN,[sn],_) -> assert(tiger_flag); true},
  p_alias(_, TabAlias).

%% p_from(-FromList)
p_from([Table|TableList]) -->
  p_chk(from),
  !,
  p_tab_ref(extended, Table),
  p_from2(TableList).

p_from2([Table|TableList]) -->
  [','],
  !,
  p_tab_ref(extended, Table),
  p_from2(TableList).
p_from2([]) --> [].

/**************************** where clause ************************************/

p_where(PredExp) --> [where], !, p_expr(PredExp,boo).
p_where(true)    --> [].

/**************************** group by clause *********************************/

p_groupby([Col|ColList]) -->
  [group], 
  !,
  p_chk(by),
  p_col_ref(Col),
  p_groupby2(ColList).
p_groupby([]) --> [].

p_groupby2([Col|ColList]) --> [','],  p_col_ref(Col), p_groupby2(ColList).
p_groupby2([])            --> [].

/******************************* having clause ********************************/

p_having(PredExp) --> [having], !, p_expr(PredExp,boo).
p_having(true)    --> [].

/******************************** order by clause *****************************/

p_orderby([]) --> [order], !, {f_raise_err(not_implemented, "ORDER BY")}.
p_orderby(_)  --> [X], !, {f_raise_err(orderby_expected, X)}.
p_orderby([]) --> [].

/********************** boolean/arithmetic expressions ************************/

p_add_op(add) --> ['+'], !.
p_add_op(sub) --> ['-'], !.

p_mul_op(mul) --> ['*'], !.
p_mul_op(div) --> ['/'], !.

p_comparison_op(Rel) --> [Rel],
  {memberchk(Rel,[eq,nq,lq,ls,gq,gr,pre,ove,con,mee])}.

p_agg_op(A) -->  [A], {memberchk(A,[avg,max,min,sum,count])}.

p_expr(Res,T)    --> p_term(T1,T), p_expr2(T1,Res,T).
p_expr2(I,O,boo) --> [or], !, p_term(T1,boo), p_expr2(or(I,T1),O,boo).
p_expr2(I,O,T)   --> p_add_op(Op), !, p_term(T1,T), p_expr2(bin(Op,I,T1),O,T).
p_expr2(I,I,_)   --> [].

p_term(Res,T)    --> p_fac(F,T), p_term2(F,Res,T).
p_term2(I,O,boo) --> [and], p_fac(F,boo), !, p_term2(and(I,F),O,boo).
p_term2(I,O,T)   --> p_mul_op(Op), !, p_fac(F,T), p_term2(bin(Op,I,F),O,T).
p_term2(I,I,_)   --> [].

p_fac1(null,ari)          --> [null].
p_fac1(sysdate,ari)       --> [sysdate].
p_fac1(un(Op,F),ari)      --> p_add_op(Op), !, p_fac1(F,ari).
p_fac1(input(N,_),ari)    --> [colon], !, p_chk(num(N)).
p_fac1(cst(Val,n(T)),ari) --> [num(Val)], {float(Val) -> T=f; true}.
p_fac1(cst(S,s),ari)      --> [str(S)].
p_fac1(P1,ari)            -->
  [period], {assert(tiger_flag)}, [P], {z_p_per(P,P1)}, !.
p_fac1(P,ari)             -->
  [timestamp], {assert(tiger_flag)}, [TS], !, {z_p_ts(TS,P)}. 
p_fac1(cst(S-Q,span),ari) --> [interval], !, p_chk(str(S)), p_granularity(Q).
p_fac1(C,ari)             --> p_col_ref(C), !.
p_fac1(period(E1,E2),ari) -->
  [period],
  !,
  p_chk('('),
  p_expr(E1,ari),
  p_chk(','),
  p_expr(E2,ari),
  p_chk(')').
p_fac1(vtime(T),ari) -->
  [vtime],
  !,
  p_chk('('), 
  p_id(T, table_name_expected),
  p_chk(')').
p_fac1(ttime(T),ari) -->
  [ttime],
  !,
  p_chk('('),
  p_id(T, table_name_expected),
  p_chk(')').
p_fac1(first(E1,E2),ari) -->
  [first],
  !,
  p_chk('('),
  p_expr(E1,ari),
  p_chk(','),
  p_expr(E2,ari),
  p_chk(')').
p_fac1(last(E1,E2,ari)) -->
  [last],
  !,
  p_chk('('),
  p_expr(E1,ari),
  p_chk(','),
  p_expr(E2,ari),
  p_chk(')').
p_fac1(begin(E),ari) -->
  [begin], !, p_chk('('), p_expr(E,ari), p_chk(')').
p_fac1(end(E),ari) -->
  [end], !, p_chk('('), p_expr(E,ari), p_chk(')').
p_fac1(gst(L),ari) -->
  [greatest], !, p_expr_list(L,yes,ari).
p_fac1(lst(L),ari) -->
  [least], !, p_expr_list(L,yes,ari).
p_fac1(to_num(E),ari) -->
  [to_number], !, p_chk('('), p_expr(E,ari), p_chk(')').
p_fac1(trunc(E),ari) -->
  [trunc], !, p_chk('('), p_expr(E,ari), p_chk(')').
p_fac1(abs(E),ari) -->
  [abs], !, p_chk('('), p_expr(E,ari), p_chk(')').
p_fac1(nvl(E1,E2),ari) -->
  [nvl], !, p_chk('('), p_expr(E1,ari), p_chk(','), p_expr(E2,ari), p_chk(')').
p_fac1(agg(count,all,all),ari) -->
  [count], ['('], ['*'], p_chk(')').
p_fac1(agg(A,M,E),ari) -->
  p_agg_op(A), p_chk('('), p_mode(M), p_expr(E,ari), p_chk(')').
p_fac1(not(F),T) -->
  [not], !, p_fac1(F,T).
p_fac1(exists(Q),boo) -->
  [exists], !, p_query_expr(Q).
p_fac1(EL,T) -->
  p_expr_list(EL,yes,T).
p_fac1(duration(P,G),T) -->
  [duration], !, p_chk('('), p_expr(P,T), 
  p_chk(','), p_granularity(G), p_chk(')'),
  {assert(tiger_flag)}, 
  {f_raise_err(duration_not_supported, _)}, !.

p_fac2(E1,rel(R,E1,E2)) -->
  p_comparison_op(R), !, p_expr(E2,ari).
p_fac2(E,isnull(E)) -->
  [is], !, p_chk(null).
p_fac2(E,between(E,A,B)) -->
  [between], !, p_expr(A,ari), p_chk(and), p_expr(B,ari).
p_fac2(E1,like(E1,E2)) -->
  [like], !, p_expr(E2,ari).
p_fac2(E,in(F,L)) --> 
  [in], p_expr_list(L,_,ari), !, {is_list(E) -> F=E; F=[E]}.
p_fac2(E,in(F,Q)) -->
  [in], p_query_expr(Q), !, {is_list(E) -> F=E; F=[E]}.
p_fac2(E1,not(E2)) -->
  [not], !, p_fac2(E1,E2).

p_fac(F,ari) --> !, p_fac1(F,ari).
p_fac(F,boo) --> p_fac1(F,boo), !.
p_fac(F,boo) --> !, p_fac1(E,ari), p_fac2(E,F).

%% p_expr_list(-ExprList, +Parenthesized, +Type)
p_expr_list([E|L],yes,T)  --> ['('], !, p_expr(E,T), p_expr_list2(L,T), p_chk(')').
p_expr_list([E|L],no,T)   --> p_expr(E,T), p_expr_list2(L,T).

p_expr_list2([E|EL],T)    --> [','], !, p_expr(E,T), p_expr_list2(EL,T).
p_expr_list2([],_)        --> [].

/************************** DML statements ************************************/

p_dml(Flag, X) --> p_flag(Flag, no), p_dml2(X).

p_source_values(values(EL)) --> [values], !, p_expr_list(EL,yes,ari).
p_source_values(QueryExpr)  --> p_query_expr(QueryExpr).

p_dml2(insert(TabN,ColNs,ValueList)) --> 
  [insert],
  !,
  p_chk(into),
  p_tab_ref(restricted, tab_ref(TabN,_,_)),
  {assert(candidate_action(insert,TabN))},
  p_col_names(ColNs),
  p_source_values(ValueList).
p_dml2(delete(TabN,TabAlias,Where)) --> 
  [delete],
  !,
  p_chk(from),
  p_tab_ref(restricted, tab_ref(TabN,TabAlias,_)),
  {assert(candidate_action(delete,TabN))},
  p_where(Where).
p_dml2(update(TabN,TabAlias,Set,Where)) --> 
  [update],
  !,
  p_tab_ref(restricted, tab_ref(TabN,TabAlias,_)), 
  {assert(candidate_action(update,TabN))},
  p_chk(set),
  p_column_assignment(Set), 
  p_where(Where).

p_column_assignment(set(C1,L1)) --> 
  p_expr(C,ari), p_chk(eq), p_expr(L,ari), !, {to_list(C,C1,L,L1)}.
p_column_assignment(nil) --> [].

to_list(C,C1,L,L1) :-
  (is_list(C) -> C1=C; C1=[C]),
  (is_list(L) -> L1=L; L1=[L]).

/****************************** DDL statements ********************************/

p_ddl(Create) --> [create], !, create(Create).
p_ddl(Drop)   --> [drop], !, drop(Drop).

create(create_table(TabN,TabDef,Type)) -->
  [table],
  !,
  p_id(TabN, table_name_expected), 
  p_table_def(TabDef),
  p_table_type(Type).
create(create_view(ViewN,ColNs,Flag,QueryExpr,Coal)) -->
  [view],
  !,
  p_id(ViewN, table_name_expected), 
  p_col_names(ColNs),
  p_chk(as),
  p_query(Flag, QueryExpr, Coal).
create(create_assertion(AssN,Flag,Cond)) -->
  [assertion],
  !,
  {assert(tiger_flag)},
  p_chk(id(AssN)),
  p_chk(check),
  p_flag(Flag, no),
  p_chk('('),
  p_expr(Cond,boo),
  p_chk(')').
create(_) --> [T], {f_raise_err(keyword_table_view_expected, T)}.

drop(drop_table(TabN)) -->
  [table],
  !,
  p_tab_ref(restricted, tab_ref(TabN,_,_)),
  {assert(candidate_action(drop,TabN))}.
drop(drop_view(ViewN)) -->
  [view],
  !,
  p_tab_ref(restricted, tab_ref(ViewN,_,_)),
  {assert(candidate_action(drop,ViewN))}.
drop(drop_assertion(AssN)) --> 
  [assertion],
  !,
  {assert(tiger_flag)},
  p_id(AssN, assertion_name_expected).
drop(drop_sequence(SeqN)) -->
  [sequence],
  !,
  p_id(SeqN, sequence_name_expected),
  {assert(candidate_action(drop,SeqN))}.
drop(_) --> [T], {f_raise_err(keyword_table_view_expected, T)}.

/***************************** DDL auxiliaries ********************************/

p_table_def(ColDefs)    -->  ['('], p_col_defs(ColDefs,[]), !, p_chk(')').

p_col_defs([C|Cs], Sep) --> Sep, !, p_col_def(C), p_col_defs(Cs, [',']).
p_col_defs([], _)       --> [].

p_col_def(colDef(N,Type,Def,Props)) -->
  [id(N)],
  p_data_type(Type), 
  p_default(Def),
  p_props([], Props).

p_default(Def) --> [default], !, p_expr(Def,ari).
p_default(nil) --> [].

p_constraint_name(cst(N,s(_)))  --> [constraint], !, p_chk(id(N)).
p_constraint_name(nil)          --> [].

p_prop(notnull)     --> [not], !, p_chk(null).
p_prop(primary_key) --> [primary], !, p_chk(key).
p_prop(references(TabN,ColN)) -->
  [references],
  !,
  p_chk(id(TabN)),
  {m_table_type(TabN,[sn],_) -> assert(tiger_flag); true},
  p_chk('('),
  p_chk(id(ColN)),
  p_chk(')').

p_props(In, Out) --> 
  p_constraint_name(ICName),
  p_flag(Flag, no),
  p_prop(Prop),
  !,
  p_props([ic(ICName,Flag,Prop)|In], Out).
p_props(In, In) --> [].

p_table_type2(bi) --> [as], [valid], [and], !, p_chk(transaction).
p_table_type2(bi) --> [as], [transaction], [and], !, p_chk(valid).
p_table_type2(vt) --> [as], [valid], !.
p_table_type2(tt) --> [as], [transaction], !.
p_table_type2(sn) --> [].

p_table_type(T) --> p_table_type2(T), !, {T==sn -> true; assert(tiger_flag)}.

p_alter_cmd(add_vt)        --> [add], [valid], !, {assert(tiger_flag)}.
p_alter_cmd(add_tt)        --> [add], [transaction], !, {assert(tiger_flag)}.
p_alter_cmd(add(ColDef))   --> [add], !, p_col_def(ColDef).

/******************************************************************************/

p_col_ref(col_ref(Tab,Col,_)) --> [id(X)], !, p_col_ref2(X, Tab, Col).

p_col_ref2(Tab, Tab, Col)     --> ['.'], !, [id(Col)].
p_col_ref2(Col, _, Col)       --> [].

/******************************************************************************/

p_col_names([CN|CNs]) --> ['('], !, [id(CN)], p_col_names2(CNs), [')'].
p_col_names([])       --> [].

p_col_names2([CN|CNs]) --> [','], [id(CN)],  p_col_names2(CNs).
p_col_names2([]) --> [].

/******************************************************************************/

p_granularity(year)   --> [year], !.
p_granularity(month)  --> [month], !.
p_granularity(day)    --> [day], !.
p_granularity(hour)   --> [hour], !.
p_granularity(minute) --> [minute], !.
p_granularity(second) --> [second], !.
p_granularity(none)   --> [].

/******************************************************************************/

p_session_par(nls_period_sep)  --> [id("NLS_PERIOD_SEP")], !.
p_session_par(nls_date_format) --> [id("NLS_DATE_FORMAT")], !.

/******************************************************************************/

p_data_type(n(f))        --> [float], !.
p_data_type(n(i))        --> [integer], !.
p_data_type(n(_))        --> [number], !.
p_data_type(per)         --> [period], !, {assert(tiger_flag)}.
p_data_type(ts)          --> [timestamp], !, {assert(tiger_flag)}.
p_data_type(ts)          --> [date], !.
p_data_type(interval)    --> [interval], !.
p_data_type(s(L))        --> [char], !, p_chk('('), p_chk(num(L)), p_chk(')').
p_data_type(s(L))        --> [varchar], !, p_chk('('), p_chk(num(L)),  p_chk(')').
p_data_type(T)           --> [T], {f_raise_err(datatype_expected, T)}.

/******************************************************************************/
/* Mode:           p_parse(+TokenList, -ParseTree)                            */
/* Purpose:        parses an ATSQL command from a token list                  */
/* Sideeffects:    None                                                       */
/* Call:           Exits always                                               */
/* Redo:           Fails always                                               */
/******************************************************************************/

%% remove_empty(+Lst, -NewLst)
remove_empty([], []). 
remove_empty([[]|Lst], NewLst) :- !, 
  remove_empty(Lst, NewLst).
remove_empty([A|Lst], [A|NewLst]) :-
  remove_empty(Lst, NewLst).

%% p_parse(+TokenList, -ParseTree)
p_parse(backend(S), backend(S)) :- !.
p_parse([id("EOF")], eof) :- !.
p_parse(Tokens, ParseTree) :- 
   remove_empty(Tokens, NewTokens), 
   p_atsql(ParseTree, NewTokens, []), !.
p_parse(_, nil).

