/*!****************************************************************************/
/*! File:      rewrite.pl                                                     */
/*! Project:   Tiger                                                          */
/*! Software:  SWI Prolog 2.7.16, Oracle 8.0.4                                */
/*! Export:                                                                   */
/*!            r_type/2 (unparser.pl,meta.pl,zeit.pl)                         */
/*!            r_leftmost_select/2 (meta.pl,trans.pl)                         */
/*!            r_realias_cols/3 (trans.pl)                                    */
/*!            r_norm_query_expr/3 (interpret.pl,views.pl,constraint.pl)      */
/*!            r_norm_delete/6 (interpret.pl)                                 */
/*!            r_norm_update/8 (interpret.pl)                                 */
/*!            r_norm_create/5 (interpret.pl)                                 */
/*!            r_norm_assertion/3 (interpret.pl)                              */
/*! Import:                                                                   */
/*!            m_get_col/4 (meta.pl)                                          */
/*!            m_table_type/3 (meta.pl)                                       */
/*!            f_raise_err/2 (fejl.pl)                                        */
/*!****************************************************************************/

%% r_subquery_free(+(Query)Expr)
%% succeeds if a (query)expression contains no subqueries
r_subquery_free(sel(_,_,_,_,_,_)) :- !, fail.
r_subquery_free(except(_,_))      :- !, fail.
r_subquery_free(union(_,_))       :- !, fail.
r_subquery_free(isct(A,_))        :- !, r_subquery_free(A).
r_subquery_free(exists(_))        :- !, fail.
r_subquery_free(in(_,A))          :- !, r_subquery_free(A).
r_subquery_free(and(A,B))         :- !, r_subquery_free(A), r_subquery_free(B).
r_subquery_free(or(A,B))          :- !, r_subquery_free(A), r_subquery_free(B).
r_subquery_free(not(A))           :- !, r_subquery_free(A).
r_subquery_free(rel(_,_,A))       :- !, r_subquery_free(A).
r_subquery_free(_).

/******************************************************************************/

%% r_table_ref(+ParseTree, -TableRef)
r_table_ref(tab_ref(derived_table(_,QE),_,_),T) :- !, r_table_ref(QE,T).
r_table_ref(tab_ref(TabName,_,_), TabName) :- !.
r_table_ref(cst(_,_), _) :- !, fail.
r_table_ref(col_ref(_,_,_), _) :- !, fail.
r_table_ref(L,Z) :- is_list(L), !, member(Y,L), r_table_ref(Y,Z).
r_table_ref(X,T) :- X=..[_|L], member(Y,L), r_table_ref(Y,T).

/******************************************************************************/

%% r_type(+Expr, -Type)
r_type(vtime(_),       per)  :- !.
r_type(ttime(_),       per)  :- !.
r_type(period(_,_),    per)  :- !.
r_type(isct(_),        per)  :- !.
r_type(now,            ts)   :- !.
r_type(begin(_),       ts)   :- !.
r_type(end(_),         ts)   :- !.
r_type(first(_,_),     ts)   :- !.
r_type(last(_,_),      ts)   :- !.
r_type(col_ref(_,_,T), T)    :- !.
r_type(cst(_,T),       T)    :- !.
r_type(un(_,E),        T)    :- !, r_type(E, T).
r_type(bin(_,E,_),     T)    :- !, r_type(E, T).
r_type(agg(count,_,_), n(i)) :- !.
r_type(agg(_,_,C),     T)    :- !, r_type(C, T).
r_type(E,              T)    :- var(T), f_raise_err(unknown_type, E).

/******************************************************************************/

%% instantiate_alias(+Name, !Alias)
instantiate_alias(Name, Name) :- !.
instantiate_alias(_, _).

%% r_leftmost_select(+QueryExpression, -SelectList)
r_leftmost_select(union(QE,_), SelList)     :- r_leftmost_select(QE, SelList).
r_leftmost_select(except(QE,_), SelList)    :- r_leftmost_select(QE, SelList).
r_leftmost_select(isct(QE,_), SelList) :- r_leftmost_select(QE, SelList).
r_leftmost_select(sel(_,SelList,_,_,_,_), SelList).

%% r_realias_cols(+OriginalSelectList, +TableAlias, -NewSelectList)
r_realias_cols([],_,[]).
r_realias_cols([sel_col(ColExpr,ColAlias)|SL], TabAlias,
             [sel_col(col_ref(TabAlias,ColAlias,ColType),ColAlias)|SL1]) :-
  r_type(ColExpr, ColType),
  r_realias_cols(SL, TabAlias, SL1).

/******************************************************************************/
/* Mode:            r_norm_query_expr(+QueryExpr, +Flag, -QueryExpr)          */
/* Purpose:         normalizes a query expression, i.e.,                      */
/*                    eliminates *                                            */
/*                    binds column aliases in select lists                    */
/*                    binds table aliases in column references                */
/*                    binds column type in column references                  */
/*                    rewrites table references to derived table expressions  */
/*                      if either a time domain or coalescing is specified    */
/* Sideeffects:     none                                                      */
/* Call:            exits always                                              */
/* Redo:            fails always                                              */
/******************************************************************************/

%% r_norm_query_expr(+QueryExpression, +Flag, -QueryExpression)
r_norm_query_expr(QueryExpr, Flag, QueryExpr1) :- 
  norm_query_expr(QueryExpr, Flag, [], QueryExpr1).

%% norm_query_expr(+QueryExpression, +Flag, +Context, -QueryExpression)
norm_query_expr(union(A,B), Flag, Context, union(A1,B1)) :- !,
  norm_query_expr(A, Flag, Context, A1),
  norm_query_expr(B, Flag, Context, B1).
norm_query_expr(except(A,B), Flag, Context, except(A1,B1)) :- !,
  norm_query_expr(A, Flag, Context, A1),
  norm_query_expr(B, Flag, Context, B1).
norm_query_expr(isct(_,_), flag(vt(seq,_),_,_), _, _) :- !,
  f_raise_err(not_implemented, "sequenced intersection").
norm_query_expr(isct(_,_), flag(_,tt(seq,_),_), _, _) :- !,
  f_raise_err(not_implemented, "sequenced intersection").
norm_query_expr(isct(A,B), Flag, Context, isct(A1,B1)) :- !,
  norm_query_expr(A, Flag, Context, A1),
  norm_query_expr(B, Flag, Context, B1).
norm_query_expr(sel(M,S,F,W,G,H), Flag, Context, sel(M,S1,F1,W1,G,H)) :- !,
  norm_from(F, Flag, F1),
  norm_select(S, Flag, [F1|Context], S1),
  norm_expr(W, Flag, [F1|Context], W1).
norm_query_expr(Values, _, _, Values).

/************************ normalization of from lists *************************/

check_tab_type(TabN, flag(vt(VTM,_),tt(TTM,_),_)) :-
  m_table_type(TabN, [], TabType),
  (  memberchk(VTM,[seq,ns]), memberchk(TTM,[seq,ns])
  -> (  TabType==bi
     -> true
     ;  f_raise_err(bi_tab_expected, TabN)
     )
  ;  memberchk(VTM,[seq,ns])
  -> (  memberchk(TabType,[vt,bi])
     -> true
     ;  f_raise_err(vt_tab_expected, TabN)
     )
  ;  memberchk(TTM,[seq,ns])
  -> (  memberchk(TabType,[tt,bi])
     -> true
     ;  f_raise_err(tt_tab_expected, TabN)
     )
  ;  true
  ).

%% norm_from(+FromList, +Flag, -FromList)
norm_from([], _, []).
norm_from([tab_ref(derived_tab(Flags,QueryExpr),TabAlias,Coal)|F], Flag,
	[tab_ref(derived_tab(Flags,QueryExpr1),TabAlias,Coal)|F1]) :- !,
	norm_query_expr(QueryExpr, Flags, [], QueryExpr1),
	norm_from(F, Flag, F1).
norm_from([tab_ref(TabName,TabAlias,nil+nil)|F],
	flag(vt(VTM,nil),tt(TTM,nil),VTR),
	[tab_ref(TabName,TabAlias,nil+nil)|F1]) :- !,
  check_tab_type(TabName, flag(vt(VTM,nil),tt(TTM,nil),VTR)),
  instantiate_alias(TabName, TabAlias),
  norm_from(F, flag(vt(VTM,nil),tt(TTM,nil),VTR), F1).
norm_from([tab_ref(TabName,TabAlias,Coal)|F],  Flag,
          [tab_ref(derived_tab(flag(vt(VTM,nil),tt(TTM,nil),nil),QExpr),
                   TabAlias,Coal)|F1]) :-
  check_tab_type(TabName, Flag),
  instantiate_alias(TabName, TabAlias),
  m_table_type(TabName, [], TabType),
  (memberchk(TabType,[vt,bi]) -> VTM=seq; VTM=tuc),
  (memberchk(TabType,[tt,bi]) -> TTM=seq; TTM=tuc),
  norm_query_expr(
    sel(all,[sel_col(all,_)],[tab_ref(TabName,TabName,nil+nil)],true,[],true),
    flag(vt(VTM,nil),tt(TTM,nil),nil),
    [], QExpr),
  norm_from(F, Flag, F1).

/********************** normalization of select lists *************************/

%% bind_sel_col_name(+SelectColumnExpr, ?SelectColumnName)
bind_sel_col_name(_, ColName)                    :- nonvar(ColName), !.
bind_sel_col_name(col_ref(_,ColName,_), ColName) :- !.
bind_sel_col_name(_, nil).

%% get_all_tab_cols2(+TabDefinition, +TabAlias, -SelList)
get_all_tab_cols2(derived_tab(_,QueryExpr), TabAlias, SelList1) :- !,
  r_leftmost_select(QueryExpr, SelList),
  r_realias_cols(SelList, TabAlias, SelList1).
get_all_tab_cols2(TabName, TabAlias, SelList) :-
  findall(sel_col(col_ref(TabAlias,ColName,DataType),ColName), 
          m_get_col(TabName, ["VT$","TT$"], ColName, DataType),
          SelList).

%% get_all_tab_cols(+Context, +TableRef, -SelectList)
get_all_tab_cols([F|_], TabAlias, SelList) :-
  member(tab_ref(TabDef,TabAlias,_), F),
  get_all_tab_cols2(TabDef,TabAlias, SelList).

%% get_all_cols(+Flag, +Context, -SelectList)
get_all_cols(Flag, [F|Context], S) :-
  findall(sel_col(all(Alias),_), member(tab_ref(_,Alias,_),F), S1),
  norm_select(S1, Flag, [F|Context], S).

%% norm_select(+SelectList, +Flag, +Context, -SelectList)
norm_select([], _, _, []).
norm_select([sel_col(all,_)], Flag, Context, S) :- !,
  get_all_cols(Flag, Context, S).
norm_select([sel_col(all(TableRef),_)|T], Flag, Context, SelectList) :- !,
  get_all_tab_cols(Context, TableRef, Cols),
  norm_select(T, Flag, Context, T1),
  append(Cols, T1, SelectList).
norm_select([sel_col(Expr,Alias)|T], Flag, Context,
            [sel_col(Expr1,Alias)|T1]) :- !,
  norm_expr(Expr, Flag, Context, Expr1),
  bind_sel_col_name(Expr1, Alias),
  norm_select(T, Flag, Context, T1).
norm_select([E|T], Flag, Context, [E1|T1]) :-
  norm_expr(E, Flag, Context, E1),
  norm_select(T, Flag, Context, T1).

/******************* normalization of expressions *****************************/

%% norm_expr(+Expression , +Flag, +Context, -Expression)
norm_expr(or(E1,E2), Flag, Context, or(E3,E4)) :- !,
  norm_expr(E1, Flag, Context, E3),
  norm_expr(E2, Flag, Context, E4).
norm_expr(and(E1,E2), Flag, Context, and(E3,E4)) :- !,
  norm_expr(E1, Flag, Context, E3),
  norm_expr(E2, Flag, Context, E4).
norm_expr(not(E1), Flag, Context, not(E2)) :- !,
  norm_expr(E1, Flag, Context, E2).
norm_expr(rel(Rel,E1,E2), Flag, Context, rel(Rel,E3,E4)) :- !,
  norm_expr(E1, Flag, Context, E3),
  norm_expr(E2, Flag, Context, E4).
norm_expr(true, _, _, true) :- !.
norm_expr(exists(_), flag(vt(seq,_),_,_), _, _) :- !,
  f_raise_err(not_implemented, "sequenced subqueries").
norm_expr(exists(_), flag(_,tt(seq,_),_), _, _) :- !,
  f_raise_err(not_implemented, "sequenced subqueries").
norm_expr(exists(QExpr), Flag, Context, exists(QExpr1)) :- !,
  norm_query_expr(QExpr, Flag, Context, QExpr1).
norm_expr(in(_,_), flag(vt(seq,_),_,_), _, _) :- !,
  f_raise_err(not_implemented, "sequenced subqueries").
norm_expr(in(_,_), flag(_,tt(seq,_),_), _, _) :- !,
  f_raise_err(not_implemented, "sequenced subqueries").
norm_expr(in(ExprList,InSet), Flag, Context, in(ExprList1,InSet1)) :- !,
  norm_exprlist(ExprList, Flag, Context, ExprList1),
  (  is_list(InSet)
  -> norm_exprlist(InSet, Flag, Context, InSet1)
  ;  norm_query_expr(InSet, Flag, Context, InSet1)
  ).

norm_expr(col_ref(A,C,T), _, Context, col_ref(A,C,T)) :- !,
  check_col_name(Context, col_ref(A,C,T)).
norm_expr(un(Op,Expr1), Flag, Context, un(Op,Expr2)) :- !,
  norm_expr(Expr1, Flag, Context, Expr2).
norm_expr(bin(Op,E1,E2), Flag, Context, bin(Op,E3,E4)) :- !,
  norm_expr(E1, Flag, Context, E3),
  norm_expr(E2, Flag, Context, E4).
norm_expr(cst(V,T), _, _, cst(V,T)) :- !.
norm_expr(now, _, _, now) :- !.
norm_expr(sysdate, _, _, sysdate) :- !.
norm_expr(null, _, _, null) :- !.
norm_expr(vtime(TabName), Flag, _, vtime(TabName)) :- !,
  (  Flag = flag(vt(tuc,_),_,_)
  -> f_raise_err(ts_in_tuc, "vt")
  ;  true
  ).
norm_expr(ttime(TabName), Flag, _, ttime(TabName)) :- !,
  (  Flag = flag(_,vt(tuc,_),_)
  -> f_raise_err(ts_in_tuc, "tt")
  ;  true
  ).
norm_expr(period(E1,E2), Flag, Context, period(E3,E4)) :- !,
  norm_expr(E1, Flag, Context, E3),
  norm_expr(E2, Flag, Context, E4).
norm_expr(first(E1,E2), Flag, Context, first(E3,E4)) :- !,
  norm_expr(E1, Flag, Context, E3),
  norm_expr(E2, Flag, Context, E4).
norm_expr(last(E1,E2), Flag, Context, last(E3,E4)) :- !,
  norm_expr(E1, Flag, Context, E3),
  norm_expr(E2, Flag, Context, E4).
norm_expr(begin(E), Flag, Context, begin(E1)) :- !,
  norm_expr(E, Flag, Context, E1).
norm_expr(end(E), Flag, Context, end(E1)) :- !,
  norm_expr(E, Flag, Context, E1).
norm_expr(gst(EL), Flag, Context, gst(EL1)) :- !,
  norm_exprlist(EL, Flag, Context, EL1).
norm_expr(lst(EL), Flag, Context, lst(EL1)) :- !,
  norm_exprlist(EL, Flag, Context, EL1).
norm_expr(to_num(E), Flag, Context, to_num(E1)) :- !,
  norm_expr(E, Flag, Context, E1).
norm_expr(trunc(E), Flag, Context, trunc(E1)) :- !,
  norm_expr(E, Flag, Context, E1).
norm_expr(abs(E), Flag, Context, abs(E1)) :- !,
  norm_expr(E, Flag, Context, E1).
norm_expr(nvl(E1,E2), Flag, Context, nvl(E3,E4)) :- !,
  norm_expr(E1, Flag, Context, E3),
  norm_expr(E2, Flag, Context, E4).
norm_expr(agg(_,_,_), flag(vt(seq,_),_,_), _, _) :- !,
  f_raise_err(not_implemented, "sequenced aggregates").
norm_expr(agg(_,_,_), flag(_,tt(seq,_),_), _, _) :- !,
  f_raise_err(not_implemented, "sequenced aggregates").
norm_expr(agg(count,M,all), _, _, agg(count,M,all)) :- !.
norm_expr(agg(Agg,M,E), Flag, Context, agg(Agg,M,E1)) :- !,
  norm_expr(E, Flag, Context, E1).
norm_expr(isnull(E), Flag, Context, isnull(E1)) :- !,
  norm_expr(E, Flag, Context, E1).
norm_expr(input(N,T1), _, _, input(N,T1)) :- !.

norm_expr(E, _, _, _) :- f_raise_err(unknown_expr, E).
 
%% norm_exprlist(+ExprList, +Flag, +Context, -ExprList)
norm_exprlist([], _, _, []) :- !.
norm_exprlist([E|T], Flag, Context, [E1|T1]) :- !,
  norm_expr(E, Flag, Context, E1),
  norm_exprlist(T, Flag, Context, T1).
 
/********************** normalization of column names  ************************/

%% check_col_name2(+FromList, +TableAliasMode, !ColumnReference)
check_col_name2(From, bound, col_ref(A,"ROWID",_)) :- !,
  member(tab_ref(_,A,_), From).
check_col_name2([tab_ref(_,A,_)], unbound, col_ref(A,"ROWID",_)) :- !.
check_col_name2(From, bound, col_ref(A,C,T)) :-
  member(tab_ref(TabDef,A,_), From),
  m_get_col(TabDef, [], C, T).
check_col_name2([tab_ref(TabDef,A,_)|From], unbound, col_ref(A1,C,T)) :-
  (  m_get_col(TabDef, [], C, T)
  -> (  A1 = A
     -> \+ check_col_name2(From, unbound, col_ref(A1,C,T))
     ;  f_raise_err(ambiguous_column, col_ref(nil,C,_))
     )
  ;  check_col_name2(From, unbound, col_ref(A1,C,T))
  ).

%% check_col_name(+Context, !ColumnReference)
check_col_name([], ColRef) :-
  f_raise_err(unknown_column, ColRef).
check_col_name([From|_], col_ref(A,C,T)) :-
  (var(A) -> Mode=unbound; Mode=bound),
  check_col_name2(From, Mode, col_ref(A,C,T)),
  !.
check_col_name([_|Context], ColRef) :-
  check_col_name(Context, ColRef), !.

/******************************************************************************/
/* Mode:            r_norm_delete(+Flag, +TabName, +TabAlias, +Where, -Flag)  */
/* Purpose:         normalizes a delete statement, i.e.,                      */
/*                    binds the table alias                                   */
/*                    if possible reduces a seq flag to a nseq one            */
/* Sideeffects:     none                                                      */
/* Call:            exits always                                              */
/* Redo:            fails always                                              */
/******************************************************************************/

%% r_norm_delete(+Flag, +TabName, +TabAlias, +Where, -Flag)
r_norm_delete(flag(vt(VTM,nil),tt(TTM,nil),nil), TabN, TabA, Where, 
              flag(vt(VTM1,nil),tt(TTM1,nil),nil), Where1) :- !,
  (TabA=TabN -> true; true),
  norm_expr(Where, flag(vt(VTM,nil),tt(TTM,nil),nil),
            [[tab_ref(TabN,TabA,nil)]], Where1),
  (  r_subquery_free(Where)    
  -> (VTM==seq -> VTM1=ns; VTM1=VTM),
     (TTM==seq -> TTM1=ns; TTM1=TTM)
  ;  VTM1=VTM,
     TTM1=TTM
  ).
r_norm_delete(flag(vt(_,nil),tt(_,nil),_), _, _, _, _, _) :- !,
  f_raise_err(vtr_in_delete, _).
r_norm_delete(flag(_,_,nil), _, _, _, _, _) :- !,
  f_raise_err(not_implemented, "time domain specification in delete").

/******************************************************************************/
/* Mode:            r_norm_update(+Flag, +TabName, +TabAlias, +Where, -Flag)  */
/* Purpose:         normalizes an update statement, i.e.,                     */
/*                    binds the table alias                                   */
/*                    if possible reduces a seq flag to a nseq one            */
/* Sideeffects:     none                                                      */
/* Call:            exits always                                              */
/* Redo:            fails always                                              */
/******************************************************************************/

%% r_norm_update(+Flag, +TabName, +TabAlias, +Set, +Where, -Flag, -Set, -Where)
r_norm_update(flag(vt(VTM,nil),tt(TTM,nil),nil), TabN, TabA, Set, Where, 
              flag(vt(VTM1,nil),tt(TTM1,nil),nil), Set, Where1) :- !,
  (TabA=TabN -> true; true),
  norm_expr(Where, flag(vt(VTM,nil),tt(TTM,nil),nil),
            [[tab_ref(TabN,TabA,nil)]], Where1),
  (  r_subquery_free(Where) , r_subquery_free(Set)   
  -> (VTM==seq -> VTM1=ns; VTM1=VTM),
     (TTM==seq -> TTM1=ns; TTM1=TTM)
  ;  VTM1=VTM,
     TTM1=TTM
  ).
r_norm_update(flag(_,_,nil), _, _, _, _, _, _, _) :- !,
  f_raise_err(not_implemented, "time domain specification in delete").
r_norm_update(flag(vt(_,nil),tt(_,nil),_), _, _, _, _, _, _, _) :- !,
  f_raise_err(not_implemented, "valid time range specification in update").

/******************************************************************************/
/* Mode:            r_norm_create(+TabName,+TabType,+ColDefs,-ColDefs,-ICs)   */
/* Purpose:         normalizes a create statement, i.e., extracts integrity   */
/*                  constraints that have to be checked by Tiger              */
/* Sideeffects:     none                                                      */
/* Call:            exits always                                              */
/* Redo:            fails always                                              */
/******************************************************************************/

%% r_norm_ic(+Prop, +TabName, +TabType, +ColName, -IC, -TDB_IC)
r_norm_ic(ic(ICN,flag(vt(VTM,nil),tt(TTM,nil),nil),notnull), TabN, _, ColN,
          nil,
          ic(ICN,flag(vt(VTM,nil),tt(TTM,nil),nil),notnull,TabN,ColN)) :- !.
r_norm_ic(ic(ICN,flag(vt(VTM,nil),tt(TTM,nil),nil),primary_key), TabN, _, ColN,
          nil,
          ic(ICN,flag(vt(VTM,nil),tt(TTM,nil),nil),primary_key,TabN,ColN)) :- !.
r_norm_ic(ic(ICN,flag(vt(VTM,nil),tt(TTM,nil),nil),references(TabN1,ColN1)),
          TabN, _, ColN, nil,
  ic(ICN,flag(vt(VTM,nil),tt(TTM,nil),nil),references(TabN1,ColN1),TabN,ColN)) :- !.
r_norm_ic(ic(_,flag(vt(_,VTD),tt(_,TTD),VTR),IC), _, _, _, _, _) :-
  (  VTD\==nil
  -> f_raise_err(not_implemented, 
       "valid time domain specification in integrity constraints")
  ;  TTD\==nil
  -> f_raise_err(not_implemented, 
       "transaction time domain specification in integrity constraints")
  ;  VTR\==nil
  -> f_raise_err(vt_range_notallowed, _)
  ;  f_raise_err(unknown_ic, IC)
  ).

bind_name(nil, bin(cat,cst("TDB$",s(_)),seq("TDB$S0",next))-
               bin(cat,cst("TDB$",s(_)),seq("TDB$S0",curr))) :- !.
bind_name(ICN, ICN-ICN).

%% r_norm_props(+Props, +TabName, +TabType, +ColName, -Props, -TDB_ICs)
r_norm_props([], _, _, _, [], []).
r_norm_props([ic(ICN,F,P)|Props], TabN, TabType, ColN, Props1, TDB_ICs) :-
  bind_name(ICN, ICN1),
  r_norm_ic(ic(ICN1,F,P), TabN, TabType, ColN, Prop1, TDB_IC),
  (Prop1==nil -> Props1=Props2; Props1=[Prop1|Props2]),
  (TDB_IC==nil -> TDB_ICs=TDB_ICs1; TDB_ICs=[TDB_IC|TDB_ICs1]),
  r_norm_props(Props, TabN, TabType, ColN, Props2, TDB_ICs1).

%% r_norm_create(+ColDefs, +TabName, +TabType, -ColDefs, -TDB_ICs)
r_norm_create([], _, _, [], []).
r_norm_create([colDef(ColN,DataType,Default,Props)|CD], TabN, TabType,
              [colDef(ColN,DataType,Default,Props1)|CD1], TDB_ICs2) :-
  r_norm_props(Props, TabN, TabType, ColN, Props1, TDB_ICs),
  r_norm_create(CD, TabN, TabType, CD1, TDB_ICs1),
  append(TDB_ICs, TDB_ICs1, TDB_ICs2).

/******************************************************************************/
/* Mode:            r_norm_assertion(+Flag, +Where, -QuerExpr)                */
/* Purpose:         normalizes an assertion, i.e., determines a ATSQL query   */
/*                  expression that can be used to check it                   */
/* Sideeffects:     none                                                      */
/* Call:            exits always                                              */
/* Redo:            fails always                                              */
/******************************************************************************/

r_norm_assertion(Flag, not(exists(sel(M,_,F,W,G,H))), QExpr) :- !,
  r_norm_query_expr(sel(M,[cst(1,n(i))],F,W,G,H), Flag, QExpr).
r_norm_assertion(Flag, Where, sel(all,[cst(1,n(i))],[],not(Where1),[],true)) :-
  norm_expr(Where, Flag, [], Where1).

