/******************************************************************************/
/* Module:      eval.pl                                                       */
/* Project:     Oracle Emulator                                               */
/* Author:      Michael Boehlen                                               */
/* Language:    SWI Prolog 2.1.9                                              */
/* Machine:     SPARC/Solaris                                                 */
/* Export:                                                                    */
/*              compile/3 (oracle.pl)                                         */
/*              exec_query/4 (oracle.pl)                                      */
/*              exec_stmt/3 (oracle.pl)                                       */
/* Import:                                                                    */
/*              abolish/1 (oracle.pl)                                         */
/*              set_of/3 (oracle.pl)                                          */
/*              type/2 (norm_check.pl)                                        */
/*              msg/1 (sqlmsg.pl)                                             */
/*              error/1 (sqlmsg.pl)                                           */
/******************************************************************************/

log_retract(A) :-
  retract(A),
  asserta(undo_log(assert(A))).

log_assert(A) :-
  assert(A),
  asserta(undo_log(retract(A))).

/******************************************************************************/
/* Mode:                compile(+Stmt,+Cursor,-CompiledStmt) 		      */
/* Purpose:             compiles a previousely normalized statement	      */
/* Example:             compile(+Stmt,0,db0,CompiledStmt)          	      */
/* Sideeffects:         none                                                  */
/* Call:                exits always                                          */
/* Redo:                fails always                                          */
/******************************************************************************/

%% compile(+Statement,+Cursor,-CompiledStmt)
compile(query(_,Q,O),Cur,query(DA,K,H,SelCl)) :-
  trans(Q,O,goal,[],Cur,DA,K,H,SelCl).
compile(create_table(T,Cs,[]),_,
        create_table(T,Cs,Cs-all-H-sel(fail,ac([],[],true,[])))) :- !,
  length(Cs,L),
  functor(H,T,L).
compile(create_table(T,Cs,query(S,Q,[])),Cur,create_table(T,Cs,S-DA-H-SelCl)) :-
  trans2(Q,T,[],Cur,DA,H,SelCl).
compile(drop_table(T),_,drop_table(T)).
compile(delete(T-_,Q),Cur,delete(DA-H-B)) :-
  trans2(Q,T,[],Cur,DA,H,sel(B,_)).
compile(insert(T,[],query(_,Q,[])),Cur,insert(DA-H-SelCl)) :- !,
  trans2(Q,T,[],Cur,DA,H,SelCl).
compile(insert(T,Cs,query(_,Q,[])),Cur,
        insert(DA-H1-sel(B,ac(A,Tx,CalcCl,ResTbl)))) :-
  trans2(Q,T,[],Cur,DA,H,sel(B,ac(A,Tx,CalcCl,ResTbl))),
  H1=..[T|ResTbl],
  unify_heads(T,H,1,Cs,H1).
compile(create_view(V,Cs,query(S,Q,[])),Cur,create_view(V,Cs,DA-H-SelCl-S)) :- 
  trans2(Q,V,[],Cur,DA,H,SelCl).
compile(drop_view(V),_,drop_view(V)).
compile(create_sequence(N,S,I),_,create_sequence(N,S,I)).
compile(drop_sequence(S),_,drop_sequence(S)).
compile(update(T-A,CL,set(query(_,Q1,[])),which(Q2)),Cur,
        update(T-A,CL,set(M1,H1,Cl),UTpl0,UTpl1,which(M2,H2,SelCl2))) :-
  atom_chars(T,TStr),
  db:num_cols(TStr,Arity),
  functor(UTpl0,T,Arity),
  functor(UTpl1,'$$upd_tpl',Arity),
  UTpl0=..[_|Args], UTpl1=..[_|Args],
  trans2(Q1,dummy,[A-UTpl1],Cur,M1,H1,Cl),
  trans2(Q2,T,[],Cur,M2,H2,SelCl2).
compile(savepoint(RPP),_,savepoint(RPP)).
compile(rollback(RPP),_,rollback(RPP)).
compile(commit,_,commit).

%% unify_heads(+TableName,*Head1,+ArgCnt1,+NameList,*Head2)
unify_heads(_,_,_,[],_).
unify_heads(T,H,CNr,[N|NR],H1) :-
  atom_chars(T,TStr),
  atom_chars(N,NStr),
  db:user_tab_columns(TStr,NStr,_,_,Pos),
  arg(CNr,H,A),
  arg(Pos,H1,A),
  CNr1 is CNr+1,
  unify_heads(T,H,CNr1,NR,H1).

%% gen_template(+FromList,+Context,-Context,-OpenPrologClause)
gen_template([],[],[],oterm(OT,OT)).
gen_template([],[A-Tpl|C],[A-Tpl|C1],oterm((Tpl,Tpl1),OT)) :-
  gen_template([],C,C1,oterm(Tpl1,OT)).
gen_template([R-A|Rs],C,[A-Tpl|C1],oterm((Tpl,Tpl1),OT)) :-
  atom_chars(R,RStr),
  db:num_cols(RStr,Arity),
  functor(Tpl,R,Arity),
  gen_template(Rs,C,C1,oterm(Tpl1,OT)).

%% trans(+Query,+HeadPredicate,+Context,+Cursor,-Mode,-Head,-SelectClause)
trans(sel(DA,S,F,W,[]-_-[]),O,P,C,Cur,DA,K,Head,sel(T1,ac([],[],true,[]))) :- !,
  gen_template(F,C,C1,oterm(T,O1)),
  length(S,L),
  functor(Head,P,L),
  trans_sel(S,1,Head,C1,Cur,oterm(O1,O2)),
  trans_where(W,C1,Cur,Cl,oterm(O2,(Cl,O3))),
  trans_orderby(O,Head,C1,K,oterm(O3,true)),
  rearrange(T,T1).
trans(Q,O,P,C,Cur,DA,K,Head,SelCl) :-
  trans2(Q,P,C,Cur,DA,Head,SelCl),
  trans_orderby(O,Head,_,K,_).

trans2(union(Q1,Q2),P,C,Cur,DA,Head,union(SelCl1,SelCl2)) :-
  trans2(Q1,P,C,Cur,DA,Head,SelCl1),
  trans2(Q2,P,C,Cur,_,Head,SelCl2).
trans2(minus(Q1,Q2),P,C,Cur,DA,Head,minus(SelCl1,SelCl2)) :-
  trans2(Q1,P,C,Cur,DA,Head,SelCl1),
  trans2(Q2,P,C,Cur,_,Head,SelCl2).
trans2(inter(Q1,Q2),P,C,Cur,DA,Head,inter(SelCl1,SelCl2)) :-
  trans2(Q1,P,C,Cur,DA,Head,SelCl1),
  trans2(Q2,P,C,Cur,_,Head,SelCl2).
trans2(sel(DA,S,F,W,A-Sx-Tx),P,C,Cur,DA,Head,sel(T1,ac(A,Tx,CalcCl,ResTbl))) :-
  gen_template(F,C,C1,oterm(T,O1)),
  (A=[] -> S1=S; S1=Sx),
  length(S1,L),
  functor(Head,P,L),
  trans_sel(S1,1,Head,C1,Cur,oterm(O1,O2)),
  trans_where(W,C1,Cur,Cl,oterm(O2,Cl)),
  trans_agg_expr_list(A,S,C1,Cur,oterm(CalcCl,true),ResTbl),
  rearrange(T,T1).

%% trans_agg_expr_list(+AggExprList,+Context,+Cur,-PrologClause,-ResTable)
trans_agg_expr_list([],_,_,_,oterm(true,true),[]) :- !.
trans_agg_expr_list(_,[],_,_,oterm(O,O),[]).
trans_agg_expr_list(A,[colS(agg(E),_)|R],C,Cur,oterm(T,O),[X|ResT]) :- !,
  trans_expr(agg(E),C,Cur,Cl,oterm(T,(X=Cl,O1))),
  trans_agg_expr_list(A,R,C,Cur,oterm(O1,O),ResT).
trans_agg_expr_list(A,[colS(E,_)|R],C,Cur,oterm(T,O),[X|ResT]) :-
  trans_expr(E,C,Cur,Cl,oterm(T,(X is Cl,O1))),
  trans_agg_expr_list(A,R,C,Cur,oterm(O1,O),ResT).

%% trans_where(+WhereClause,+Context,+Cursor,-PrologClause,-OpenPrologTerm)
trans_where(and(A,B),C,Cur,(ClA,ClB),oterm(T,O)) :- !,
  trans_where(A,C,Cur,ClA,oterm(T,O1)),
  trans_where(B,C,Cur,ClB,oterm(O1,O)).
trans_where(or(A,B),C,Cur,call(db:(ClA->true;ClB)),oterm(T,O)) :- !,
  trans_where(A,C,Cur,ClA,oterm(T,O1)),
  trans_where(B,C,Cur,ClB,oterm(O1,O)).
trans_where(not(A),C,Cur,\+ClA,oterm(T,O)) :- !,
  trans_where(A,C,Cur,ClA,oterm(T,O)).
trans_where(ex(query(_,Q,[])),C,Cur,X,oterm(O,O)) :- !,
  trans2(Q,dummy,C,Cur,_,H,Cl),
  (Cl=sel(SelCl,_) -> X=call(db:(SelCl,!)); X=call(oracle:ex_hdlr(H,Cl))).
trans_where(in(EL,query(_,Q,[])),C,Cur,
            call(oracle:in_hdlr(Cl,VL,H,SelCl)),oterm(O,O)) :- !,
  trans_exprList(EL,C,Cur,VL,oterm(Cl,true)),
  trans2(Q,dummy,C,Cur,_,H,SelCl).
trans_where(rel(eq,A,B),C,Cur,Cl,oterm(T,O)) :- !,
  trans_expr(A,C,Cur,ClA,oterm(T,O1)),
  trans_expr(B,C,Cur,ClB,oterm(O1,O)),
  type(A,Type),
  (Type==num -> Cl=(ClA=:=ClB); Cl=(ClA==ClB)).
trans_where(rel(nq,A,B),C,Cur,Cl,oterm(T,O)) :- !,
  trans_expr(A,C,Cur,ClA,oterm(T,O1)),
  trans_expr(B,C,Cur,ClB,oterm(O1,O)),
  type(A,Type),
  (Type==num  -> Cl=(ClA=\=ClB); Cl=(ClA\==ClB)).
trans_where(rel(lq,A,B),C,Cur,Cl,oterm(T,O)) :- !,
  trans_expr(A,C,Cur,ClA,oterm(T,O1)),
  trans_expr(B,C,Cur,ClB,oterm(O1,O)),
  type(A,Type),
  (Type==num  -> Cl=(ClA=<ClB); Cl=(ClA@=<ClB)).
trans_where(rel(ls,A,B),C,Cur,Cl,oterm(T,O)) :- !,
  trans_expr(A,C,Cur,ClA,oterm(T,O1)),
  trans_expr(B,C,Cur,ClB,oterm(O1,O)),
  type(A,Type),
  (Type==num  -> Cl=(ClA<ClB); Cl=(ClA@<ClB)).
trans_where(rel(gq,A,B),C,Cur,Cl,oterm(T,O)) :- !,
  trans_expr(A,C,Cur,ClA,oterm(T,O1)),
  trans_expr(B,C,Cur,ClB,oterm(O1,O)),
  type(A,Type),
  (Type==num  -> Cl=(ClA>=ClB); Cl=(ClA@>=ClB)).
trans_where(rel(gr,A,B),C,Cur,Cl,oterm(T,O)) :- !,
  trans_expr(A,C,Cur,ClA,oterm(T,O1)),
  trans_expr(B,C,Cur,ClB,oterm(O1,O)),
  type(A,Type),
  (Type==num  -> Cl=(ClA>ClB); Cl=(ClA@>ClB)).
trans_where(true,_,_,true,oterm(O,O)).

%% neg(+NumOrStr,-LexicallyNegatedNumOrStr)
neg([],[]) :- !.
neg([X|Y],[X1|Y1]) :- !, X1 is -X, neg(Y,Y1).
neg(X,Y) :- Y is -X.

%% trans_orderby(+OrderbyList,+Head,+Context,-KeyList,-OpenTerm)
trans_orderby([],_,_,[],oterm(O,O)).
trans_orderby([num(N)-asc|O],H,C,[E1|K],oterm(O1,O2)) :- !,
  arg(N,H,E1),
  trans_orderby(O,H,C,K,oterm(O1,O2)).
trans_orderby([num(N)-desc|O],H,C,[E2|K],oterm(O1,O2)) :- !,
  arg(N,H,E1),
  trans_orderby(O,H,C,K,oterm(O1,(oracle:neg(E1,E2),O2))).
trans_orderby([E-asc|O],H,C,[E1|K],oterm(O1,O2)) :-
  trans_expr(E,C,_,E1,_),
  trans_orderby(O,H,C,K,oterm(O1,O2)).
trans_orderby([E-desc|O],H,C,[E2|K],oterm(O1,O2)) :-
  trans_expr(E,C,_,E1,_),
  trans_orderby(O,H,C,K,oterm(O1,(oracle:neg(E1,E2),O2))).

%% trans_exprList(+ExprList,+Context,+Cur,-PrologValList,-OpenPrologTerm)
trans_exprList([],_,_,[],oterm(O,O)).
trans_exprList([E|EL],C,Cur,[V|VL],oterm(T,O)) :-
  trans_expr(E,C,Cur,Cl,oterm(T,O1)),
  type(E,Type),
  (Type==num ->
    trans_exprList(EL,C,Cur,VL,oterm(O1,(V is Cl,O)));
    trans_exprList(EL,C,Cur,VL,oterm(O1,(V=Cl,O)))).

%% trans_sel(+SelectPart,+ArgNr,+Head,+Context,+Cursor,*PrologTerm)
trans_sel([],_,_,_,_,oterm(X,X)).
trans_sel([colS(SE,_)|R],Cnt,Head,C,Cur,oterm(PT,O)) :-
  trans_expr(SE,C,Cur,Cl,oterm(PT,(Clx,O1))),
  (var(Cl)        -> Clx=(Var=Cl);
   Cl=input(Nr,_) -> Clx=(oracle:in(Cur,Nr,Var));
   number(Cl)     -> Clx=(Var=Cl);
   is_list(Cl)    -> Clx=(Var=Cl);
                     Clx=(Var is Cl)),
  arg(Cnt,Head,Var),
  Cnt1 is Cnt+1,
  trans_sel(R,Cnt1,Head,C,Cur,oterm(O1,O)).

%% trans_expr(+SelElem,+Context,+Cursor,-PrologClause,*OpenPrologTerm)
trans_expr(colR(_-A,CNr,_),C,_,Cl,oterm(O,O)) :-
  member(A-RT,C), !,
  arg(CNr,RT,Cl).
trans_expr(bin(add,A,B),C,Cur,(ClA+ClB),oterm(T,O)) :- !,
  trans_expr(A,C,Cur,ClA,oterm(T,O1)),
  trans_expr(B,C,Cur,ClB,oterm(O1,O)).
trans_expr(bin(sub,A,B),C,Cur,(ClA-ClB),oterm(T,O)) :- !,
  trans_expr(A,C,Cur,ClA,oterm(T,O1)),
  trans_expr(B,C,Cur,ClB,oterm(O1,O)).
trans_expr(bin(mul,A,B),C,Cur,(ClA*ClB),oterm(T,O)) :- !,
  trans_expr(A,C,Cur,ClA,oterm(T,O1)),
  trans_expr(B,C,Cur,ClB,oterm(O1,O)).
trans_expr(bin(div,A,B),C,Cur,(ClA/ClB),oterm(T,O)) :- !,
  trans_expr(A,C,Cur,ClA,oterm(T,O1)),
  trans_expr(B,C,Cur,ClB,oterm(O1,O)).
trans_expr(un(add,A),C,Cur,Cl,oterm(T,O)) :- !,
  trans_expr(A,C,Cur,Cl,oterm(T,O)).
trans_expr(un(sub,A),C,Cur,-Cl,oterm(T,O)) :- !,
  trans_expr(A,C,Cur,Cl,oterm(T,O)).
trans_expr(gst(L),C,Cur,Max,oterm(T,O)) :- !,
  trans_exprList(L,C,Cur,ClList,oterm(T,(oracle:max_list(ClList,Max),O))).
trans_expr(lst(L),C,Cur,Min,oterm(T,O)) :- !,
  trans_exprList(L,C,Cur,ClList,oterm(T,(oracle:min_list(ClList,Min),O))).
trans_expr(str(S),_,_,S,oterm(O,O)) :- !.
trans_expr(num(N),_,_,N1,oterm((N1 is N,O),O)) :- !.
trans_expr(input(N,_),_,Cur,Val,oterm((oracle:in(Cur,N,Val),O),O)) :- !.
trans_expr(nextval(SStr),_,_,Val,oterm((oracle:nextval(SStr,Val),O),O)) :- !.
trans_expr(currval(SStr),_,_,Val,oterm((oracle:currval(SStr,Val),O),O)) :- !.
trans_expr(agg(X),_,_,X,oterm(O,O)) :- !.
trans_expr(to_num(E),C,Cur,N,oterm(T,O)) :- !,
  trans_expr(E,C,Cur,Cl,oterm(T,(number_chars(N,Cl),O))).
trans_expr(trunc(E),C,Cur,Cl,oterm(T,O)) :- !,
  trans_expr(E,C,Cur,Cl1,oterm(T,(Cl is floor(Cl1),O))).
trans_expr(abs(E),C,Cur,Cl,oterm(T,O)) :- !,
  trans_expr(E,C,Cur,Cl1,oterm(T,(Cl is abs(Cl1),O))).

%% rearrange(+PrologClause,-RearrangedPrologClause)
rearrange(Cl,U) :- rearrange2(Cl,oterm(U,C),oterm(C,true)).
rearrange2((A,B),oterm(U,OU),oterm(C,OC)) :- !,
  rearrange2(A,oterm(U,OU1),oterm(C,OC1)),
  rearrange2(B,oterm(OU1,OU),oterm(OC1,OC)).
rearrange2((A=B),oterm((A=B,OU),OU),oterm(OC,OC)) :-
  (var(A); atomic(A); is_list(A)),
  (var(B); atomic(B); is_list(A)),
  !.
rearrange2((A==B),oterm((A=B,OU),OU),oterm(OC,OC)) :-
  (var(A); is_list(A)),
  (var(B); is_list(B)),
  !.
rearrange2((A=:=B),oterm((A=B,OU),OU),oterm(OC,OC)) :-
  (var(A); number(A)),
  (var(B); number(B)),
  !.
rearrange2(A,oterm(OU,OU),oterm((A,OC),OC)).

/******************************************************************************/
/* Mode:                exec_query(+Query,+StmtStr,ResTbl,-NumTuples)         */
/* Purpose:             executes a previousely compiled query                 */
/* Example:             exec_query(Q,_,ResTbl,NumTuples)                      */
/* Sideeffects:         none                                                  */
/* Call:                exits always                                          */
/* Redo:                fails always                                          */
/******************************************************************************/

exec_query(query(DA,K,H,SelCl),_,ResTbl1,Num) :-
  H=..[_|Args],
  eval_res(DA,K-Args,SelCl,ResTbl),
  length(ResTbl,Num),
  keysort(ResTbl,ResTbl1).

/******************************************************************************/
/* Mode:                exec_stmt(+CompiledStmt,-NumTuples)		      */
/* Purpose:             executes a previousely compiled command		      */
/* Example:             exec_stmt(CompiledStmt,NumTuples)                     */
/* Sideeffects:         database is changed according to the statement        */
/* Call:                exits always                                          */
/* Redo:                fails always                                          */
/******************************************************************************/

exec_stmt(create_table(T,Cs,S-DA-H-SelCl),_,0) :-
  atom_chars(T,TStr),
  log_assert(db:user_catalog(TStr,"table")),
  log_assert(db:user_tables(TStr)),
  insert_col_names(Cs,S,TStr,1),
  length(S,L),
%  db:dynamic(T/L),
  log_assert(db:num_cols(TStr,L)),
  assert_tuples(DA,H,SelCl,_),
  msg(creaTab).
exec_stmt(drop_table(T),_,0) :-
  atom_chars(T,TStr),
  log_retract(db:user_catalog(TStr,"table")),
  log_retract(db:user_tables(TStr)),
  (log_retract(db:user_tab_columns(TStr,_,_,_,_)), fail; true),
  log_retract(db:num_cols(TStr,Num)),
  functor(A,T,Num),
  (log_retract(db:A), fail; true),
  msg(dropTab).
exec_stmt(delete(_-H-B),_,Num) :-
  findall(1, (call(db:B), log_retract(db:H)), L),
  length(L,Num),
  msg(delRec(Num)).
exec_stmt(insert(DA-H-SelCl),_,Num) :-
  assert_tuples(DA,H,SelCl,Num),
  msg(creaRec(Num)).
exec_stmt(create_view(V,Cs,DA-H-B-S),StmtStr,0) :-
  atom_chars(V,VStr),
  log_assert(db:user_catalog(VStr,"view")),
  log_assert(db:(H:-oracle:derive_tuples(DA,B,H))),
  log_assert(db:user_views(VStr,StmtStr)),
  insert_col_names(Cs,S,VStr,1),
  length(S,L),
  log_assert(db:num_cols(VStr,L)),
  msg(creaVw).
exec_stmt(drop_view(V),_,0) :-
  atom_chars(V,VStr),
  log_retract(db:user_catalog(VStr,"view")),
  log_retract(db:user_views(VStr,_)),
  (log_retract(db:user_tab_columns(VStr,_,_,_,_)), fail; true),
  log_retract(db:num_cols(VStr,Num)),
  functor(A,V,Num),
  (log_retract(db:A), fail; true),
  msg(dropVw).
exec_stmt(create_sequence(N,S,I),_,0) :-
  atom_chars(N,SStr),
  log_assert(db:user_catalog(SStr,"sequence")),
  log_assert(db:user_sequences(SStr)),
  S1 is S-I,
  log_assert(db:sequence(SStr,S1,I,undef)),
  msg(creaSeq).
exec_stmt(drop_sequence(S),_,0) :-
  atom_chars(S,SStr),
  log_retract(db:user_catalog(SStr,"sequence")),
  log_retract(db:user_sequences(SStr)),
  log_retract(db:sequence(SStr,_,_,_)),
  msg(dropSeq).
exec_stmt(update(_,CL,set(M1,H1,SelCl1),UTpl0,UTpl1,which(_,H2,sel(B,_))),_,L) :-
  findall(H2, call(db:B), OldTpls),
  H1=..[_|Args],
  findall(NewTpl,
          (member(UTpl0,OldTpls),
           assert(db:UTpl1),
           calc(UTpl0,CL,M1,SelCl1,Args,NewTpl),
           retract(db:UTpl1)),
          NewTpls),
  (err(_) ->
     true;
     length(NewTpls,L),
     (length(OldTpls,L) ->
       (member(Old,OldTpls), log_retract(db:Old), fail; true),
       (member(New,NewTpls), log_assert(db:New), fail; true);
       true),
     msg(updRec(L))).
exec_stmt(rollback(RBP),_,0) :- !,
  (undo_log(savepoint(RBP)) ->
    retract(undo_log(X)),
      (X=savepoint(RBP) ->
         asserta(undo_log(savepoint(RBP)));
       X=savepoint(_) ->
         fail;
       /* else */
         once(X),
         fail),
     !,
     msg(rb_done);
     error(rbp_unknown)).
exec_stmt(savepoint(RBP),_,0) :- !,
  retractall(undo_log(savepoint(RBP))),
  asserta(undo_log(savepoint(RBP))),
  msg(svp_made).
exec_stmt(commit,_,0) :- !,
  abolish(undo_log/1),
  asserta(undo_log(savepoint('$$work$$'))),
  msg(commit_done).

calc(OldTuple,CL,M1,SelCl1,Args,NewTuple) :-
  eval_res(M1,[]-Args,SelCl1,ResTbl),
  length(ResTbl,L),
  (L==0 -> error(upd_undef);
   L>1  -> error(upd_ambig);
   ResTbl=[[]-Tpl], modify(OldTuple,Tpl,CL,NewTuple)).

find(Pos,[colR(_,Pos,_)|_],[Val|_],Val) :- !.
find(Pos,[_|CL],[_|Args],Val) :- find(Pos,CL,Args,Val).

modify(OldTuple,Args,CL,NewTuple) :-
  length(Args,L),
  (length(CL,L) ->
     functor(OldTuple,F,A),
     functor(NewTuple,F,A),
     modify2(A,Args,CL,OldTuple,NewTuple);
     error(upd_undef)).
modify2(0,_,_,_,_).
modify2(Pos,Args,CL,OldTuple,NewTuple) :-
  (find(Pos,CL,Args,Val) ->
     arg(Pos,NewTuple,Val);
     arg(Pos,OldTuple,X),
     arg(Pos,NewTuple,X)),
  Pos1 is Pos-1,
  modify2(Pos1,Args,CL,OldTuple,NewTuple).

%% eval_res(+Mode,+Key&Args,+Clause,-ResTable)
eval_res(_,KArgs,union(Cl1,Cl2),Res) :- !,
  eval_res(all,KArgs,Cl1,Res1),
  eval_res(all,KArgs,Cl2,Res2),
  set_of(X, (member(X,Res1);member(X,Res2)), Res).
eval_res(DA,KArgs,minus(Cl1,Cl2),Res) :- !,
  eval_res(DA,KArgs,Cl1,Res1),
  eval_res(DA,KArgs,Cl2,Res2),
  set_of(X, (member(X,Res1),\+memberchk(X,Res2)), Res).
eval_res(DA,KArgs,inter(Cl1,Cl2),Res) :- !,
  eval_res(DA,KArgs,Cl1,Res1),
  eval_res(DA,KArgs,Cl2,Res2),
  set_of(X, (member(X,Res1),memberchk(X,Res2)), Res).
eval_res(DA,KArgs,sel(B,ac([],[],true,[])),ResTbl) :- !,
  eval_res2(DA,KArgs,B,ResTbl).
eval_res(DA,KArgs,sel(B,ac(A,T,CalcCl,ResTbl)),ResTbl1) :-
  eval_res2(DA,KArgs,B,Tbl),
  (Tbl==[] ->
     ResTbl1=[];
     ResTbl1=[[]-ResTbl],
     calc_aggs(A,Tbl,T),
     call(CalcCl)).

eval_res2(all,KArgs,B,Res) :- findall(KArgs,db:B,Res).
eval_res2(distinct,KArgs,B,Res) :- set_of(KArgs,db:B,Res).

%% insert_col_names(+ColumnNames,+SelectList,+TableNameStr,+Cnt).
insert_col_names([],[],_,_).
insert_col_names([],[colS(SE,N)|R],TStr,Cnt) :-
  type(SE,Type),
  atom_chars(Type,TypeStr),
  atom_chars(N,NStr),
  log_assert(db:user_tab_columns(TStr,NStr,TypeStr,"null",Cnt)),
  Cnt1 is Cnt+1,
  insert_col_names([],R,TStr,Cnt1).
insert_col_names([N|NR],[colS(SE,_)|CR],TStr,Cnt) :-
  type(SE,Type),
  atom_chars(Type,TypeStr),
  atom_chars(N,NStr),
  log_assert(db:user_tab_columns(TStr,NStr,TypeStr,"null",Cnt)),
  Cnt1 is Cnt+1,
  insert_col_names(NR,CR,TStr,Cnt1).
insert_col_names([colD(N,Type,Null)|R],[colD(N,Type,Null)|R],TStr,Cnt) :-
  atom_chars(Type,TypeStr),
  atom_chars(N,NStr),
  atom_chars(Null,NullStr),
  log_assert(db:user_tab_columns(TStr,NStr,TypeStr,NullStr,Cnt)),
  Cnt1 is Cnt+1,
  insert_col_names(R,R,TStr,Cnt1).

%% assert_tuples(+Mode,+Head,+SelectClause,-NumTuples).
assert_tuples(DA,H,SelCl,Num) :-
  H=..[_|Args],
  eval_res(DA,Args,SelCl,ResTbl),
  (member(Args,ResTbl), log_assert(db:H), fail; true),
  length(ResTbl,Num).

%% derive_tuples(+Mode,+SelectClause,*Head).
derive_tuples(DA,SelCl,H) :-
  H=..[_|Args],
  eval_res(DA,Args,SelCl,ResTbl),
  member(Args,ResTbl).

%% calc_aggs(+AggFuncs,+Table,-Tuple)
calc_aggs(_,[],[]) :- !.
calc_aggs(Aggs,[_-H|T],Res) :-
  length([H|T],L),
  pre_proc_agg(Aggs,Res,L,Aggs1,Res1,CalcCl),
  calc_aggs2(Aggs1,T,H,Res1),
  call(CalcCl).
calc_aggs2(_,[],In,In).
calc_aggs2(AggList,[_-Tpl|Tbl],In,Out) :-
  calc_aggs3(AggList,Tpl,In,Tmp),
  calc_aggs2(AggList,Tbl,Tmp,Out).
calc_aggs3([],[],[],[]).
calc_aggs3([Agg|R],[Val|ValR],[Old|OldR],[New|NewR]) :-
  proc_agg(Agg,Val,Old,New),
  calc_aggs3(R,ValR,OldR,NewR).

%% pre_proc_agg(+Aggs,+Tuple,+Length,-Aggs,-Tuple,-Clause)
pre_proc_agg([],[],_,[],[],true).
pre_proc_agg([count|AR],[L|TR],L,[count|AR1],[_|TR1],Cl) :- !,
  pre_proc_agg(AR,TR,L,AR1,TR1,Cl).
pre_proc_agg([avg|AR],[X|TR],L,[sum|AR1],[Z|TR1],(X is Z/L,Cl)) :- !,
  pre_proc_agg(AR,TR,L,AR1,TR1,Cl).
pre_proc_agg([Agg|AR],[X|TR],L,[Agg|AR1],[X|TR1],Cl) :-
  pre_proc_agg(AR,TR,L,AR1,TR1,Cl).

proc_agg(min,X,Old,X) :- is_list(X), X@=<Old, !.
proc_agg(min,X,Old,Old) :- is_list(X), X@>Old, !.
proc_agg(max,X,Old,X) :- is_list(X), X@>=Old, !.
proc_agg(max,X,Old,Old) :- is_list(X), X@<Old, !.
proc_agg(sum,X,Old,New) :- New is X+Old.
proc_agg(min,X,Old,X) :- X=<Old.
proc_agg(min,X,Old,Old) :- X>Old.
proc_agg(max,X,Old,X) :- X>=Old.
proc_agg(max,X,Old,Old) :- X<Old.
proc_agg(count,_,_,_).

ex_hdlr(H,SelCl) :-
  H=..[_|Args],
  \+eval_res(all,Args,SelCl,[]).

in_hdlr(Cl,VL,H,SelCl) :- 
  H=..[_|Args],
  eval_res(all,Args,SelCl,ResTbl),
  call(Cl),
  memberchk(VL,ResTbl).

nextval(SStr,Val1) :-
  log_retract(db:sequence(SStr,Val,Incr,_)), !,
  Val1 is Val+Incr,
  log_assert(db:sequence(SStr,Val1,Incr,def)).
currval(SStr,Val) :- db:sequence(SStr,Val,_,def).

/******************************************************************************/
/* .									      */
/******************************************************************************/
