/******************************************************************************/
/* Module:      display.pl                                                    */
/* Project:     TimeDB 1.04                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SICStus Prolog                                                */
/* Machine:     SPARC/Solaris, PowerMac/MacOS                                 */
/* Export:                                                                    */
/*              get_parameters/4 (constraints.pl)                             */
/*              display_results/2 (timeDB.pl)                                 */
/* Import:                                                                    */
/*              get_expression/3 (basics.pl)                                  */
/*              internal_to_date/2 (calendar.pl)                              */
/*              internal_to_span/2 (calendar.pl)                              */
/******************************************************************************/

/****************************************************************************/
/* Mode:	get_columns(+LengthList, -Format, -Slashs)                  */
/* Purpose:	Generates headline for attributes.                          */
/* Sideeffects:	None        				                    */
/* Call:        Exists always				                    */
/* Redo:        Fails always					            */
/****************************************************************************/

generate_slash(1, [])        :- !.
generate_slash(N, [45|Rest]) :- N1 is N - 1, generate_slash(N1, Rest).

get_columns([], "~n", []) :- !.
get_columns([Length|LengthList], Format, [SlashStr|SlashList]) :-
	!,
	name(Length, LenStr),
	generate_slash(Length, SlashStr),
	get_columns(LengthList, Format1, SlashList),
	append([126,116,126,115,126|LenStr], "+", Format2),  % "~t~s~"
	append(Format2, Format1, Format).


/****************************************************************************/
/* Mode:	show(+LenghList, +ColumnNames, +Tupels)	                    */
/* Purpose:	Writes result table to screen                               */ 
/* Example:								    */
/* Sideeffects:	Messages on screen					    */
/* Call:        Exists always						    */
/* Redo:        Fails always						    */
/****************************************************************************/

show_tupels(_, []) :- !.
show_tupels(Format, [Tupel|Rest]) :-
	!,
	format(Format, Tupel),
	show_tupels(Format, Rest).
 

show(LengthList, ColumnNames, Tupels) :-
	get_columns(LengthList, Format, Slashs),
	nl, nl,
	format(Format, ColumnNames),
	format(Format, Slashs),
	show_tupels(Format, Tupels),
	!.

/****************************************************************************/
/* Mode:	get_tupel(+Result, +TypeList, -Output)                      */
/* Purpose:	Generates output.           		                    */
/* Sideeffects:	None        				                    */
/* Call:        Exists always						    */
/* Redo:        Fails always						    */
/****************************************************************************/

get_tupel([], [], []) :- !.
get_tupel([var(Str, _)|Tupel], [char|Types], [Str|Output]) :-
	!, get_tupel(Tupel, Types, Output).
get_tupel([var(StartInt, _), var(EndInt, _)|Tupel], [interval|Rest], [Str|Output]) :-
	internal_to_date(StartInt, StartDat),
	internal_to_date(EndInt, EndDat),
	append([91|StartDat], [45|EndDat], Str1),
	append(Str1, ")", Str),
	get_tupel(Tupel, Rest, Output).
get_tupel([var(Event, _)|Tupel], [event|RestTypes], [Str|Output]) :-
	internal_to_date(Event, EventDat),
	append([124|EventDat], "|", Str), 
	get_tupel(Tupel, RestTypes, Output). 
get_tupel([var(Span, _)|Tupel], [span|RestTypes], [Str|Output]) :-
	number_chars(Val, Span),
	internal_to_span(Val, Span1),
	append([37|Span1], "%", Str), 
	get_tupel(Tupel, RestTypes, Output). 
get_tupel([var(Number, _)|Tupel], [number|RestTypes], [Number|Output]) :- 
	get_tupel(Tupel, RestTypes, Output).
get_tupel([var(Long, _)|Tupel], [long|RestTypes], [Long|Output]) :- 
	get_tupel(Tupel, RestTypes, Output).


/****************************************************************************/
/* Mode:	transform_tupels(+TypeList, +ResultList, -TupelList)        */
/* Purpose:	Transforms result of query to displayable tupels.           */ 
/* Example:								    */
/* Sideeffects:	None            					    */
/* Call:        Exists always			               	            */
/* Redo:        Fails always				                    */
/****************************************************************************/

transform_tupels(_, [], []) :- !.
transform_tupels(TypeList, [Res|OutputList], [Tupel|TupelList]) :-
	get_tupel(Res, TypeList, Tupel),
	transform_tupels(TypeList, OutputList, TupelList),
	!.


/****************************************************************************/
/* Mode:	display_tupels(+ColumnNames, +Tupel_List)            	    */
/* Purpose:	Writes result table to screen		                    */ 
/* Example:								    */
/* Sideeffects:	Messages on screen					    */
/* Call:        Exists always			               	            */
/* Redo:        Fails always				                    */
/****************************************************************************/

compare_maximas([], [], []) :- !.
compare_maximas([L1|Rest1], [L2|Rest2], [L1|Rest]) :-
	L1 >= L2, 
	!,
	compare_maximas(Rest1, Rest2, Rest).
compare_maximas([_|Rest1], [L2|Rest2], [L2|Rest]) :-
	compare_maximas(Rest1, Rest2, Rest).


get_max([], []) :- !.
get_max([Str|Rest], [Len|RestLen]) :-
	length(Str, Len1),
	Len is Len1 + 1,
	get_max(Rest, RestLen).


get_max_length([List], MaxList)      :- !, get_max(List, MaxList).
get_max_length([List|Rest], MaxList) :- 
	!, 
	get_max(List, MaxList1), 
	get_max_length(Rest, MaxList2),
	compare_maximas(MaxList1, MaxList2, MaxList).

display_tupels(_, []) :-
	!, nl, nl, write('No tupels selected'), nl, nl.
display_tupels(ColumnNames, TupelList) :-
	get_max_length([ColumnNames], MaxLengthList1),
	get_max_length(TupelList, MaxLengthList2),
	compare_maximas(MaxLengthList1, MaxLengthList2, MaxLengthList),
	show(MaxLengthList, ColumnNames, TupelList).


/****************************************************************************/
/* Mode:           display_results(+ResultType, +ResultTable)               */
/* Purpose:        Display tuples in ResultTable.                           */
/* Example:                                                                 */
/* Sideeffects:    Text on screen.                                          */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

%% get_parameters(+ResultType, -ColumnNames, -OutputVars, -TypeList)
get_parameters([], [], [], []) :- !.
get_parameters([valid-interval-_|ResType], ["VALID"|ColNames],
	       [var(_, n), var(_, n)|OutputVars], [interval|Types]) :-
	!,
	get_parameters(ResType, ColNames, OutputVars, Types).
get_parameters([system-interval-_|ResType], ["TRANSACTION"|ColNames],
               [var(_, n), var(_, n)|OutputVars], [interval|Types]) :-
	!,
	get_parameters(ResType, ColNames, OutputVars, Types).
get_parameters([Exp-_-interval-_|ResType], [ExpStr|ColumnNames],
               [var(_, n), var(_, n)|OutputVars], [interval|Types]) :-
	!,
	get_expression(Exp, ExpStr, []),
	get_parameters(ResType, ColumnNames, OutputVars, Types).
get_parameters([Exp-_-char-Length|ResType], [ExpStr|ColumnNames],
               [var(_, s(Length))|OutputVars], [char|Types]) :-
	!,
	get_expression(Exp, ExpStr, []),
	get_parameters(ResType, ColumnNames, OutputVars, Types).
get_parameters([Exp-_-Type-_|ResType], [ExpStr|ColumnNames],
               [var(_, n)|OutputVars], [Type|Types]) :-
	!,
	get_expression(Exp, ExpStr, []),
	get_parameters(ResType, ColumnNames, OutputVars, Types).


display_results([], []) :- !.
display_results(ResultType, ResultQuery) :-
	get_parameters(ResultType, ColumnNames, OutputVars, TypeList),
	findall(OutputVars, sql_sel([], ResultQuery, OutputVars), OutputList),
	transform_tupels(TypeList, OutputList, TupelList),
	display_tupels(ColumnNames, TupelList),
	!.
