/******************************************************************************/
/* Module:      parser.pl                                                     */
/* Project:     TimeDB 1.07                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SICStus Prolog                                                */
/* Machine:     SPARC/Solaris, PowerMac/MacOS                                 */
/* Export:                                                                    */
/*              parse/1 (timeDB.pl)                                           */
/* Import:                                                                    */
/*              generate_alias/1 (basics.pl)                                  */
/*              generate_interval_col/3 (basics.pl)                           */
/*              scan/1 (scanner.pl)                                           */
/*              generate_error_flag/2 (errors.pl)                             */
/*              granularity/1 (calendar.pl)                                   */
/*              interval_to_internal/4 (calendar.pl)                          */
/*              event_to_internal/3 (calendar.pl)                             */
/*              span_to_int/3 (calendar.pl)                                   */
/******************************************************************************/

:- ensure_loaded([scanner]).


chk(X) --> [X], !.
chk(X) --> [Y], {generate_error_flag(expected_instead, X-Y)}, [_].
%chk(X) --> [Y], {generate_error_flag(expected_instead, X-Y)}, [_|_].

change_token(str(Str), Atom) :- !, atom_chars(Atom, Str).
change_token(num(N), N)      :- !.
change_token(id(ID), ID)     :- !.
change_token(T, T).

determine_coalescing(nored, C, C) :- !.
determine_coalescing(_, _, red).

add_coalescing(sfw(SFW, State-C1-TT), C2, sfw(SFW, State-C-TT)) :- 
	!, determine_coalescing(C1, C2, C).
add_coalescing(union(Union)-Time-C1, C2, union(Union)-Time-C)   :- 
	!, determine_coalescing(C1, C2, C).
add_coalescing(minus(Minus)-Time-C1, C2, minus(Minus)-Time-C)   :- 
	!, determine_coalescing(C1, C2, C).
add_coalescing(intersect(Intersect)-Time-C1, C2, intersect(Intersect)-Time-C) :- 
	!, determine_coalescing(C1, C2, C).


/*****************************************************************************/
/********************************* ATSQL2 DCG RULES **************************/
/*****************************************************************************/

sql(quit)                     --> [quit], !, chk(';').
sql(status_control(Ctrl))     --> status_control(Ctrl), !, chk(';').
sql(data_control(Ctrl))       --> data_control(Ctrl), !, chk(';').
sql(data_manipulation(Items)) --> data_manipulation(Items), !, chk(';').
sql(data_definition(Items))   --> data_definition(Items), !, chk(';').
sql(query(Query))             --> query_language(Query), !, chk(';').
sql(_, _, _)                  :- !, generate_error_flag(syntax_error, _).


valid_flag --> [valid], !.
valid_flag --> [validtime].

transaction_flag --> [transaction], !.
transaction_flag --> [transactiontime].

valid_exp(Exp) --> scalar_exp(Exp), !.
valid_exp([])  --> [].

time_flag(no_state-snapshot-Exp)   -->          % 8. result: snapshot/valid table 
        [nonsequenced], valid_flag, valid_exp(Exp), [and], 
        [nonsequenced], transaction_flag, !.
time_flag(no_state-snapshot-Exp)   -->          % 8. result: snapshot/valid table 
        [nonsequenced], transaction_flag, [and], 
        [nonsequenced], valid_flag, valid_exp(Exp), !.
time_flag(no_state-system-Exp)     -->          % 7. result: system/bitemporal table
        [nonsequenced], valid_flag, valid_exp(Exp), [and], transaction_flag, !. 
time_flag(no_state-system-Exp)     -->          % 7. result: system/bitemporal table
        transaction_flag, [and], [nonsequenced], valid_flag, valid_exp(Exp), !. 
time_flag(no_state-valid-Exp)       -->          % 9. result: valid table
        valid_flag, valid_exp(Exp), [and], [nonsequenced], transaction_flag, !.
%time_flag(no_state-valid-[])       -->          % 9. result: valid table
%        valid_flag, [and], [nonsequenced], transaction_flag, !.
time_flag(no_state-valid-Exp)       -->          % 9. result: valid table
        [nonsequenced], transaction_flag, [and], valid_flag, valid_exp(Exp), !.
%time_flag(no_state-valid-[])       -->          % 9. result: valid table
%        [nonsequenced], transaction_flag, [and], valid_flag, !.
time_flag(no_state-validsystem-[]) -->          % 4. result: bitemporal table
        transaction_flag, [and], valid_flag, !.
time_flag(no_state-systemvalid-[]) -->          % 4. result: bitemporal table
        valid_flag, [and], transaction_flag, !.
time_flag(tt-snapshot-Exp)         -->          % 5. result: snapshot/valid table
        [nonsequenced], valid_flag, !, valid_exp(Exp).  
time_flag(vt-snapshot-[])          -->          % 6. result: snapshot table
        [nonsequenced], transaction_flag, !.
time_flag(vt-system-[])            -->          % 3. result: system table
        transaction_flag, !.
time_flag(tt-valid-Exp)             -->          % 2. result: valid table
        valid_flag, valid_exp(Exp), !.
%time_flag(tt-valid-[])             -->          % 2. result: valid table
%        valid_flag, !.
time_flag(tuc-snapshot-[])         -->          % 1. result: snapshot table
        [].


query_language(Query) --> 
	time_flag(Flags), 
	table_set_expression(Flags, nored, Query).

subquery_select_statement(Flags, Query) --> table_set_expression(Flags, nored, Query).


table_set_expression(Flags, Coal, Query) --> 
        table_set_term(Flags, Term),
        table_set_expression2(Flags, Term, Query1),
	{add_coalescing(Query1, Coal, Query)}.

table_set_expression2(State-Time-Valid, T1, Query) --> 
        [union], !,
        table_set_term(State-Time-Valid, T2),
        table_set_expression2(State-Time-Valid, union([T1, T2])-Time-nored, Query), !.
table_set_expression2(State-Time-Valid, T1, Query) --> 
        [except], !,
        table_set_term(State-Time-Valid, T2),
        table_set_expression2(State-Time-Valid, minus([T1, T2])-Time-nored, Query), !.
table_set_expression2(_, Query, Query)             --> [].


table_set_term(Flags, Term) --> 
        table_set_faktor(Flags, Factor),
        table_set_term2(Flags, Factor, Term).

table_set_term2(State-Time-Valid, F1, Term) --> 
        [intersect], !,
        table_set_faktor(State-Time-Valid, F2),
        table_set_term2(State-Time-Valid, intersect([F1, F2])-Time-nored, Term), !.
table_set_term2(_, Factor, Factor)          --> [].


table_set_faktor(Flags, SFW) --> select_expression(Flags, SFW).
table_set_faktor(Flags, Exp) --> 
        ['('], time_flag(tuc-snapshot-[]), 
	       table_set_expression(Flags, Coal, Exp), 
	chk(')'), period(Coal).
table_set_faktor(_, Exp) --> 
        ['('], time_flag(Flags),
	       table_set_expression(Flags, Coal, Exp), 
	chk(')'), period(Coal).

period(red)   --> ['('], chk(period), chk(')'), !.
period(nored) --> [].


ordering_clause([ordering(Items)]) --> [order], chk(by), !, order_spec(Items).

order_spec([Item|ItemList]) --> order_item(Item), order_spec2(ItemList).
order_spec2(ItemList) --> [','], !, order_spec(ItemList).
order_spec2([])       --> [].

order_item(OrderItem) --> ordering_column(Item), order_item2(Item, OrderItem).

order_item2(Item, asc(Item))  --> [asc].
order_item2(Item, desc(Item)) --> [desc].
order_item2(Item, asc(Item))  --> [].

ordering_column(Col)     --> column_reference(Col).
ordering_column(Integer) --> integer(Integer).

/****************************************************************************/

%% due to time reasons :-), 19-03-96 as
extend_from(Exp, from(From), from([F|From])) :-
	F = derived_table(sfw([snapshot([expr(Exp, valid)]),
                          from([dual-'a#$_1']), 
%                          from(From), 
                          where([])], tt - nored - valid), 'a_#$99999', []),
	!.

select_expression(State-valid-Exp, sfw([Select, From|Rest], NewFlags)) -->
	{Exp \== []},
        select_clause(State-valid-[], nored, Select, NewFlags),
        from_clause(State-valid-Exp, From1),
	{extend_from(Exp, From1, From)},
        select_expression2(State-valid-Exp, Rest).
select_expression(Flags, sfw([Select, From|Rest], NewFlags)) -->
        select_clause(Flags, nored, Select, NewFlags),
        from_clause(Flags, From),
        select_expression2(Flags, Rest).

select_expression2(Flags, [Where|Rest])     -->
        where_clause(Flags, Where), 
        select_expression3(Flags, Rest).
select_expression2(Flags, [where([])|Rest]) --> 
        select_expression3(Flags, Rest).

%select_expression3(_-Time-Exp, [Group|Having]) --> 
%        grouping_clause(Group), select_expression4(Time, Exp, Having).
select_expression3(_, [])                      --> [].

select_expression4(Time, Exp, [H]) --> having_clause(Time, Exp, H).
select_expression4(_, _, [])       --> [].

/***************************************************************************/

select_clause(tuc-snapshot-[], _, Select, tuc-nored-snapshot)     --> 
        [select], !, select_clause2(snapshot, [], Select, _).
select_clause(State-Time-ValidExp, red, Select, State-red-Time)   --> 
        [select], !, select_clause2(Time, ValidExp, Select, _).
select_clause(State-Time-ValidExp, nored, Select, State-nored-TT) --> 
        [select], !, select_clause2(Time, ValidExp, Select, TT).

select_clause2(Time, ValidExp, all(Select), TT)      --> 
        [all], !, select_clause3(Time, ValidExp, Select, TT).
select_clause2(Time, ValidExp, distinct(Select), TT) --> 
        [distinct], !, select_clause3(Time, ValidExp, Select, TT).
select_clause2(Time, ValidExp, Select, TT)           --> 
        select_clause3(Time, ValidExp, Select, TT).

select_clause3(snapshot, [], snapshot(Select), snapshot)                      --> 
        !, select_spec(Select).
select_clause3(snapshot, Valid, snapshot([expr(Valid, valid)|Select]), valid) --> 
        !, select_spec(Select).
select_clause3(valid, [], valid(Select), valid)                               --> 
        !, select_spec(Select).
select_clause3(system, [], system(Select), system)                            --> 
        !, select_spec(Select).
select_clause3(system, Valid, system([expr(Valid, valid)|Select]), system)    --> 
        !, select_spec(Select).
select_clause3(validsystem, [], validsystem(Select), bitemporal)              --> 
        !, select_spec(Select).
select_clause3(systemvalid, [], systemvalid(Select), bitemporal)              --> 
        !, select_spec(Select).

select_spec([SelectItem|SelectList]) -->
        select_item(SelectItem), 
        select_spec2(SelectList).

select_spec2([SelectItem|SelectList]) -->
        [','], !, 
        select_item(SelectItem), 
        select_spec2(SelectList).
select_spec2([])                      --> [].

select_item(all)     --> [*], !.
select_item(Tab-all) --> range_variable(Tab), ['.'], [*], !.
select_item(Item)    --> scalar_exp(Exp), select_item2(Exp, Item), !.
select_item([])      --> [Token], !, {generate_error_flag(select_item, Token)}. 

select_item2(Ref, expr(Ref, Alias)) --> [as], !, ident(Alias).
select_item2(Ref, expr(Ref, Alias)) --> ident(Alias), !.
select_item2(Ref, Ref)              --> [].

/***************************************************************************/

period --> ['('], chk(period), chk(')').

%table_coal(tuc, sfw(SFW, _), sfw(SFW, tuc-nored-snapshot))     --> [], !.
% tuc is not propagated into derived table; 22.3.96, as
table_coal(_, sfw(SFW, State-_-TT), sfw(SFW, State-red-TT))    --> period, !.
table_coal(_, sfw(SFW, Flags), sfw(SFW, Flags))                --> [], !.

table_coal(tuc, union(Union)-_-_, union(Union)-snapshot-nored) --> [], !.
table_coal(_, union(Union)-Time-_, union(Union)-Time-red)      --> period, !.

table_coal(tuc, minus(Union)-_-_, minus(Union)-snapshot-nored) --> [], !.
table_coal(_, minus(Union)-Time-_, minus(Union)-Time-red)      --> period, !.

table_coal(tuc, intersect(Union)-_-_, intersect(Union)-snapshot-red) --> [], !.
table_coal(_, intersect(Union)-Time-_, intersect(Union)-Time-red)    --> period, !.

table_coal(tuc, T-A, T-A) --> [], !.
table_coal(State, T-A, 
           derived_table(sfw([valid([all]), from([T-A]), where([])], State-red-valid),
                         A, [])) --> 
        period, !.
table_coal(_, Table_Ref, Table_Ref) --> [].


from_clause(State-_-_, from([Table|TableList])) -->
        chk(from), !, table_ref(State, Table), from_clause2(State, TableList).

from_clause2(State, [Table|TableList]) -->
        [','], !, table_ref(State, Table), from_clause2(State, TableList).
from_clause2(_, [])                  --> [].


table_ref(State, derived_table(Query, Alias, Cols)) -->
        ['('], !, query_language(Query1), chk(')'), 
        table_coal(State, Query1, Query), 
        table_ref2(Alias),
        derived_column_list(Cols).
table_ref(State, Ref)                               -->
        table_name(TabName), 
        table_coal(State, TabName-Alias, Ref), 
        table_ref3(Alias),
        !.
table_ref(_, _)                                   -->
        [Token], !,
        {generate_error_flag(table_expected, Token)}.
 
table_ref2(Alias) --> [as], !, range_variable(Alias).
table_ref2(Alias) --> range_variable(Alias), !.
table_ref2(_)     --> [Token], {generate_error_flag(derived_table_alias, Token)}.

table_ref3(Alias) --> [as], range_variable(Alias), !.
table_ref3(Alias) --> range_variable(Alias), !.
table_ref3(Alias) --> [], {generate_alias(Alias)}.

derived_column_list(Columns) --> ['('], column_list(Columns), chk(')'). 
derived_column_list([])      --> [].

/***************************************************************************/

where_clause(Flags, where(PredExp)) --> 
        [where], !, predicate_exp(Flags, PredExp).

/***************************************************************************/

grouping_clause(group([Column|ColList])) -->
        [group], [by], !, 
        column_reference(Column),
        grouping_clause2(ColList).

grouping_clause2([Column|ColList]) -->
        [','], 
        column_reference(Column),
        grouping_clause2(ColList).
grouping_clause2([])                --> [].

having_clause(Time, Exp, having(PredExp)) --> 
        [having], !, predicate_exp(tt-Time-Exp, PredExp).

/****************************************************************************/

predicate_exp(Flags, PredExp) -->  
        predicate_term(Flags, Term), !, 
        predicate_exp2(Flags, Term, PredExp).

predicate_exp2(Flags, Term, or(Term, PredExp)) --> 
        [or], !, 
        predicate_term(Flags, Term1), 
        predicate_exp2(Flags, Term1, PredExp).
predicate_exp2(_, Term, Term)                  --> [].

predicate_term(Flags, PredicateTerm) --> 
        predicate_factor(Flags, Factor), 
        !,
        predicate_term2(Flags, Factor, PredicateTerm).

predicate_term2(Flags, Factor, and(Factor, PredicateTerm)) -->
        [and], !, 
        predicate_factor(Flags, Factor1),
        predicate_term2(Flags, Factor1, PredicateTerm).
predicate_term2(_, Factor, Factor)                         --> [].

predicate_factor(Flags, not(Condition)) --> 
        [not], !, condition(Flags, Condition).
predicate_factor(Flags, Condition)      --> 
        !, condition(Flags, Condition).

condition(Flags, Cond) --> compare_condition(Flags, Cond), !.
condition(Flags, Cond) --> in_condition(Flags, Cond), !.
condition(Flags, Cond) --> exists_condition(Flags, Cond), !.
condition(Flags, Cond) --> ['('], !, predicate_exp(Flags, Cond), [')'].
condition(_, Cond)     --> between_condition(Cond), !.


compare_condition(Flags, rel(Op, Exp1, Exp2))  -->
        scalar_exp(Exp1), compare_condition2(Flags, Op, Exp2), !.

compare_condition2(Flags, Op, Exp) -->
        compare_op(Op), !, compare_condition4(Flags, Exp).
compare_condition2(_, eq, Exp) -->
        [is], !, compare_condition3(Exp).

compare_condition3(notnull) --> 
	[not], [null], !, {generate_error_flag(not_implemented_yet, 'null-values')}.
compare_condition3(null)    --> 
	[null], !, {generate_error_flag(not_implemented_yet, 'null-values')}.

compare_condition4(_, Exp)                       --> 
        scalar_exp(Exp).
compare_condition4(Flags, query(Subquery)) -->
        ['('], !, subquery_select_statement(Flags, Subquery), chk(')').


in_condition(Flags, Cond)  --> 
        scalar_exp(Exp), !, in_condition2(Flags, Exp, Cond).

in_condition2(Flags, Exp, not(in(Exp, Set))) --> 
        [not], !, in_condition3(Flags, Set).
in_condition2(Flags, Exp, in(Exp, Set))      --> 
        in_condition3(Flags, Set).

in_condition3(Flags, Set) --> 
        [in], chk('('), set_of_scalars(Flags, Set), chk(')'), !.

set_of_scalars(Flags, query(Subquery))       --> 
        subquery_select_statement(Flags, Subquery).
set_of_scalars(_, _, cst([Element|ElementList])) -->
        constant(Element, _), !, set_of_scalars2(ElementList).

set_of_scalars2([Element|ElementList]) -->
        [','], !, constant(Element, _), set_of_scalars2(ElementList).
set_of_scalars2([])                    --> [].


exists_condition(Flags, exists(query(Subquery))) -->
        [exists], !, 
        chk('('), subquery_select_statement(Flags, Subquery), chk(')').


between_condition(Cond)  --> 
        column_reference(Col),
        between_condition2(Col, Cond),
        !.
between_condition(_)     --> 
        [Token],
        {generate_error_flag(column_expected, Token)}.

between_condition2(Col, not(between(Col, Exp1, Exp2))) -->
        [not], !,
        between_condition3(Exp1, Exp2).
between_condition2(Col, between(Col, Exp1, Exp2))           --> 
        between_condition3(Exp1, Exp2).
between_condition3(Exp1, Exp2) -->
        [between], !, scalar_exp(Exp1), [and], scalar_exp(Exp2).


like_condition(Cond)  --> 
        column_reference(Col), !, like_condition2(Col, Cond).

like_condition2(Col, not(like(Col, Exp))) -->
        [not], [like], !, scalar_exp(Exp).
like_condition2(Col, like(Col, Exp))    -->
        [like], !, scalar_exp(Exp).


compare_op(eq)       --> [=], !.
compare_op(nq)       --> [<>], !.
compare_op(ls)       --> [<], !.
compare_op(gr)       --> [>], !.
compare_op(lq)       --> [<=], !.
compare_op(gq)       --> [>=], !.
compare_op(precedes) --> [precedes], !.
compare_op(overlaps) --> [overlaps], !.
compare_op(meets)    --> [meets], !.
compare_op(contains) --> [contains], !.

/****************************************************************************/

scalar_exp(TermExp) -->
        [+], !, scalar_term(Term), scalar_exp2(Term, TermExp).
scalar_exp(neg(TermExp)) -->
        [-], !, scalar_term(Term), scalar_exp2(Term, TermExp).
scalar_exp(TermExp) -->
        scalar_term(Term), scalar_exp2(Term, TermExp).

scalar_exp2(Term1, add(Term1, TermExp)) --> 
        [+], !, scalar_term(Term2), scalar_exp2(Term2, TermExp).
scalar_exp2(Term1, sub(Term1, TermExp)) --> 
        [-], !, scalar_term(Term2), scalar_exp2(Term2, TermExp).
scalar_exp2(Term, Term) --> [].


scalar_term(Term) -->
        scalar_value(Faktor), !, scalar_term2(Faktor, Term).
scalar_term2(Faktor1, quo(Faktor1, Term)) -->
        [/], !, scalar_value(Faktor2), scalar_term2(Faktor2, Term).
scalar_term2(Faktor1, mul(Faktor1, Term)) -->
        [*], !, scalar_value(Faktor2), scalar_term2(Faktor2, Term).
scalar_term2(Faktor, Faktor) --> [].

scalar_value(Value)     --> column_reference(Value), !.
scalar_value(Value)     --> interval_primary(Value), !.
scalar_value(Value)     --> event_primary(Value), !.
scalar_value(Value)     --> span_primary(Value), !.
scalar_value(Value)     --> constant(Value, _), !.
scalar_value(gst(List)) --> [greatest], chk('('), !, scalar_exp_list(List), chk(')').
scalar_value(lst(List)) --> [least], chk('('), !, scalar_exp_list(List), chk(')').
scalar_value(Function)  --> set_function_reference(Function).
scalar_value(abs(Exp))  --> [abs], chk('('), scalar_exp(Exp), chk(')'), !.
scalar_value(Exp)       --> ['('], scalar_exp(Exp), [')'], !.

scalar_exp_list([Exp|SList]) -->
        scalar_exp(Exp), scalar_exp_list2(SList).
scalar_exp_list2([Exp|SList]) --> 
        [','], !, scalar_exp(Exp), scalar_exp_list2(SList).
scalar_exp_list2([])          --> [].

set_function_reference(fkt(count, all)) -->
        [count], ['('], [*], chk(')'), !, 
	{generate_error_flag(aggregates_niy, count)}.
set_function_reference(fkt(Op, Exp))    --> 
        set_function_name(Op), 
        ['('], set_function_reference2(Exp), chk(')').
set_function_reference2(Exp)            -->
        scalar_exp(Exp).
set_function_reference2(distinct(Ref))  --> 
        [distinct], column_reference(Ref).

set_function_name(count) --> [count], !, {generate_error_flag(aggregates_niy, count)}.
set_function_name(sum)   --> [sum], !, {generate_error_flag(aggregates_niy, sum)}.
set_function_name(avg)   --> [avg], !, {generate_error_flag(aggregates_niy, avg)}.
set_function_name(max)   --> [max], !, {generate_error_flag(aggregates_niy, max)}.
set_function_name(min)   --> [min], !, {generate_error_flag(aggregates_niy, min)}.

/***************************************************************************/
                              
interval_primary(Prim)               --> 
        interval_literal(Prim), !.
interval_primary(valid(Table))       --> 
        valid_flag, !, ['('], table_name(Table), chk(')').
interval_primary(transaction(Table)) -->
        transaction_flag, !, ['('], table_name(Table), chk(')').
interval_primary(Prim)               -->
        interval_value_function(Prim), !.

interval_literal(interval(cst(Start, interval), cst(End, interval))) -->
        [period], !, interval_to_internal(Start, End).


interval_value_function(interval(Exp1, Exp2)) -->
        [period], 
        chk('('), !, scalar_exp(Exp1), chk(','), scalar_exp(Exp2), chk(')').
interval_value_function(intersect(Exp1, Exp2)) --> 
        [intersect],
        chk('('), scalar_exp(Exp1), chk(','), scalar_exp(Exp2), chk(')').


event_primary(Prim) --> event_literal(Prim).
event_primary(Prim) --> event_value_function(Prim).

event_literal(cst(Time, event)) --> [timestamp], !, event_literal2(Time). 
event_literal(cst(Time, event)) --> [date], !, event_literal2(Time). 

event_literal2(Time) --> event_to_internal(Time), !.
event_literal2(_)    --> [T], 
	{change_token(T, A), generate_error_flag(expected_instead, event-A)}.

event_value_function(begin(Exp))        -->
        [begin], chk('('), !, scalar_exp(Exp), chk(')').
event_value_function(end(Exp)) -->
        [end], chk('('), !, scalar_exp(Exp), chk(')').
event_value_function(first(Exp1, Exp2)) -->
        [first], 
        chk('('), !, scalar_exp(Exp1), chk(','), scalar_exp(Exp2), chk(')').
event_value_function(last(Exp1, Exp2))  -->
        [last], 
        chk('('), !, scalar_exp(Exp1), chk(','), scalar_exp(Exp2), chk(')').
event_value_function(present)           --> 
        [present], !.


span_primary(Span)    --> span_literal(Span).
span_primary(Primary) --> span_value_function(Primary).

span_literal(cst(Span, span))      --> 
        [interval], [+], !, [str(S)], qualifier(Q), {span_to_int(Q, S, Span)}.
span_literal(neg(cst(Span, span))) --> 
        [interval], [-], !, [str(S)], qualifier(Q), {span_to_int(Q, S, Span)}.
span_literal(cst(Span, span))      --> 
        [interval], !, [str(S)], qualifier(Q), {span_to_int(Q, S, Span)}.

span_value_function(span(Exp)) -->
        [interval], chk('('), !, scalar_exp(Exp), chk(')').
span_value_function(absolute(Exp))         -->
        [absolute], chk('('), !, scalar_exp(Exp), chk(')').

qualifier(year)   --> [year], !.
qualifier(month)  --> [month], !.
qualifier(day)    --> [day], !.
qualifier(hour)   --> [hour], !.
qualifier(minute) --> [minute], !.
qualifier(second) --> [second], !.
qualifier(Gran)   --> [], {granularity(Gran)}.


/*************************************************************************************/

manip_time_flag(tt-snapshot-Exp) --> [nonsequenced], valid_flag, !, valid_exp(Exp).
manip_time_flag(tt-valid-Exp)    --> valid_flag, !, valid_exp(Exp). %19-03-96, as
manip_time_flag(tt-valid-[])     --> valid_flag, !.
manip_time_flag(tuc-snapshot-[]) --> [].


data_manipulation(Insert) --> manip_time_flag(Flags), insert_statement(Flags, Insert).
data_manipulation(Delete) --> manip_time_flag(Flags), delete_statement(Flags, Delete).
data_manipulation(Update) --> manip_time_flag(Flags), update_statement(Flags, Update).


insert_statement(Flags, insert(Flags, Table, ColumnList, ValueList)) --> 
        [insert], chk(into), !, table_name(Table),
        insert_statement1(Flags, ColumnList, ValueList).

%insert_statement1(Flags, ColumnList, ValueList) --> 
%       ['('], !, column_list(ColumnList), chk(')'), source_values(Flags, ValueList).
insert_statement1(Flags, [], ValueList)         --> 
        source_values(Flags, ValueList).

source_values(Flags, query(Query))               --> 
        subquery_select_statement(Flags, Query).
source_values(_, val([Val|Values])-[_-Type-_|Types]) --> 
        chk(values), 
        chk('('), !, insert_item(Val-Type), source_values2(Values-Types), chk(')').

source_values2([Val|Values]-[_-Type-_|Types]) --> 
        [','], insert_item(Val-Type), source_values2(Values-Types).
source_values2([]-[])                             --> [].

insert_item(Value-Type) --> constant(Value, Type).


delete_statement(Flags, delete(Flags, Table, Where)) --> 
        [delete], chk(from), table_ref(Table),
        delete_statement1(Flags, Where).

delete_statement1(Flags, Where) --> where_clause(Flags, Where).
delete_statement1(_, where([])) --> [].


update_statement(Flags, update(Flags, Table, Set, Where)) --> 
        [update], table_ref(Table), 
        chk(set), column_assignment(Set), 
        update_statement1(Flags, Where).

update_statement1(Flags, Where) --> where_clause(Flags, Where).
update_statement1(_, where([])) --> [].


column_assignment(set(ColList, query(Query)))     --> 
        ['('], !, column_list(ColList), chk(')'), chk('='), 
        subquery_select_statement(tuc-snapshot-[], Query), !.
column_assignment(val([Col-Val-Type|Rest]))       -->
        column_reference(Col), chk('='), constant(Val, Type),
        column_assignment1(Rest), !.
column_assignment(_) --> {generate_error_flag(invalid_update_columns, _)}.


column_assignment1([Col-Val-Type|Rest]) --> 
        [','], !, column_reference(Col), chk('='), constant(Val, Type),
        column_assignment1(Rest).
column_assignment1([])                   --> [].


table_ref(Table-Alias) --> table_name(Table), range_variable(Alias), !.
table_ref(Table-Alias) --> table_name(Table), {generate_alias(Alias)}.


/****************************************************************************/

data_definition(Create) --> create(Create).
data_definition(Drop)   --> drop(Drop).
data_definition(Alter)  --> alter(Alter).

create(Create) --> [create], base_table_def(Create), !.
create(Create) --> [create], view_def(Create), !.
create(Create) --> [create], create_assertion(Create), !.
create(_)      --> [create], [T], {generate_error_flag(keyword_table_view_expected, T)}.

drop(Drop) --> [drop], table_drop(Drop), !.
drop(Drop) --> [drop], view_drop(Drop), !.
drop(Drop) --> [drop], assertion_drop(Drop), !.
drop(_)    --> [drop], [T], {generate_error_flag(keyword_table_view_expected, T)}.


base_table_def(create_table(Table, Columns, Select, Temp)) -->
        [table], table_name(Table), 
        base_table_def1(Columns, Select),
        temp_spec(Temp).

base_table_def1(Columns, query(Query)) --> 
        ['('], column_list(Columns), [')'], !,
        chk(as), query_language(Query).
base_table_def1(Columns, [])           --> 
        ['('], base_table_element_spec(Columns), chk(')'), !.
base_table_def1([], query(Query))      -->
        chk(as), query_language(Query).


base_table_element_spec([S-interval-Consts, E-interval-Consts|ColumnList]) --> 
	column_def(Name-Type-Consts),
	{Type == interval, generate_interval_col(Name, S, E)}, !,
	base_table_element_spec2(ColumnList).
base_table_element_spec([Column|ColumnList])                                 --> 
	column_def(Column), 
	base_table_element_spec2(ColumnList).

base_table_element_spec2([S-interval-Consts, E-interval-Consts|ColumnList]) -->
        [','], 
	column_def(Name-Type-Consts),
        {Type == interval, generate_interval_col(Name, S, E)}, !,
        base_table_element_spec2(ColumnList).
base_table_element_spec2([Column|ColumnList])                                 -->
        [','], 
        column_def(Column), 
        base_table_element_spec2(ColumnList).
base_table_element_spec2([])                                                  --> [].


column_def(Name-Type-Constraints) --> 
	column_name(Name), 
	column_def1(Type, Constraints), !.
column_def(_)                     --> 
	[T], {generate_error_flag(identifier_expected, T)}.

column_def1(Type, Constraints) --> 
	data_type(Type), 
	column_constraint_def([], Constraints), !.
column_def1(_, _)              --> 
	[T], {generate_error_flag(datatype_expected, T)}.


column_constraint_def(ConstraintsIn, ConstraintsOut) --> 
	column_constraint(ConstraintsIn, ConstraintList), 
	column_constraint_def(ConstraintList, ConstraintsOut).
column_constraint_def(Constraints, Constraints)      --> [].


column_constraint(Constraints, [notnull|Constraints])                       --> 
	[not], chk(null), !.
column_constraint(Constraints, [key|Constraints])                           --> 
	[primary], chk(key), !.
column_constraint(Constraints, [unique|Constraints])                        --> 
	[unique], !, {generate_error_flag(not_implemented_yet, unique)}.
column_constraint(Constraints, [ref(T-C-valid)|Constraints])                --> 
	valid_flag, [references], !, ident(T), chk('('), ident(C), chk(')').
column_constraint(Constraints, [ref(T-C-snapshot)|Constraints])             --> 
	[references], ident(T), chk('('), ident(C), chk(')'), !.
column_constraint(Constraints, [check(not(PredExp), valid)|Constraints])    --> 
	valid_flag, [check], chk('('), predicate_exp(tt-valid-[], PredExp), chk(')'), !.
column_constraint(Constraints, [check(not(PredExp), snapshot)|Constraints]) --> 
	[check], chk('('), predicate_exp(tuc-snapshot-[], PredExp), chk(')'), !.


temp_spec(bitemporal) --> [as], valid_flag, time_granularity(_), [and], !, transaction_flag.
temp_spec(bitemporal) --> [as], transaction_flag, [and], !, valid_flag.
temp_spec(valid)      --> [as], valid_flag, !, time_granularity(_).
temp_spec(system)     --> [as], transaction_flag, !.
temp_spec([])         --> [].

table_drop(droptable(Name)) --> [table], table_name(Name).


view_def(create_view(View, Columns, Select)) --> 
        [view], !, table_name(View), 
        view_def1(Columns, Select).

view_def1(Columns, query(Query)) --> 
        ['('], column_list(Columns), chk(')'), 
        chk(as), query_language(Query).
view_def1([], query(Query))      -->
        chk(as), query_language(Query).

view_drop(dropview(Name)) --> [view], table_name(Name).


create_assertion(assertion(Name, Type, Query)) --> 
	[assertion], !, ident(Name), create_assertion1(Type, Query).	

create_assertion1(Type, Query) -->
	valid_flag, valid_exp(Exp), !,
	chk(check),
	chk('('), assertion_query(valid, Exp, Type, Query), chk(')').
create_assertion1(Type, Query) -->
	chk(check),
	chk('('), assertion_query(snapshot, [], Type, Query), chk(')').


assertion_query(snapshot, [], exists, Query)         --> 
	[exists], !, 
	chk('('), subquery_select_statement(tuc-snapshot-[], Query), chk(')').
assertion_query(valid, TimeExp, exists, Query)       --> 
	[exists], !, 
	chk('('), subquery_select_statement(tt-valid-TimeExp, Query), chk(')').
assertion_query(snapshot, [], notexists, Query)      --> 
	[not], chk(exists), !, 
	chk('('), subquery_select_statement(tuc-snapshot-[], Query), chk(')').
assertion_query(valid, TimeExp, notexists, Query)    --> 
	[not], chk(exists), !, 
	chk('('), subquery_select_statement(tt-valid-TimeExp, Query), chk(')').

assertion_drop(dropassertion(Name)) --> [assertion], table_name(Name).


alter(alter(T, A)) --> [alter], [table], table_name(T), alter_action(A).

alter_action(Const)                 --> [add], table_constraint(Const), !.
alter_action(add(valid))            --> [add], valid_flag, !, time_granularity(_).
alter_action(add(system))           --> [add], transaction_flag, !.
alter_action(add(Name-Type-Consts)) --> [add], column_def(Name-Type-Consts), !.
alter_action(drop(valid))           --> [drop], valid_flag.
alter_action(drop(system))          --> [drop], transaction_flag.
alter_action(drop(Column))          --> [drop], column_name(Column).

time_granularity(year)   --> 
	[year], !, {generate_error_flag(invalid_table_granularity, year)}.
time_granularity(month)  --> 
	[month], !, {generate_error_flag(invalid_table_granularity, month)}.
time_granularity(day)    --> 
	[day], !, {generate_error_flag(invalid_table_granularity, day)}.
time_granularity(hour)   --> 
	[hour], !, {generate_error_flag(invalid_table_granularity, hour)}.
time_granularity(minute) --> 
	[minute], !, {generate_error_flag(invalid_table_granularity, minute)}.
time_granularity(second) --> [second], !.
time_granularity(second) --> [].


table_constraint(check(not(Exp), valid))    --> 
	valid_flag, [check], chk('('), predicate_exp(tt-valid-[], Exp), chk(')'), !.
table_constraint(check(not(Exp), snapshot)) --> 
	[check], chk('('), predicate_exp(tuc-snapshot-[], Exp), chk(')'), !.

/****************************************************************************/

data_control(commit)     --> [commit], !.
data_control(check)      --> [check], !.
data_control(rollback)   --> [rollback], !.
data_control(open(S))    --> [open], !, chk(str(S)).
data_control(close)      --> [close], !.
data_control(eof)        --> [id(eof)], !.
data_control(batch(I,O)) --> [batch], !, chk(str(I)), chk(str(O)).

/****************************************************************************/

status_control(status)                 --> [status].
status_control(trace)                  --> [trace], !.
status_control(notrace)                --> [notrace], !.
status_control(remove_demo_tt)         --> [remove], [clock].
status_control(calendar(Cal))          --> 
        [set], [calendric], !, chk(system), calendar(Cal).
status_control(demo_tt(Chronons, Inc)) --> 
        [set], [clock], !, chk(to), event_literal(cst(Chronons, event)),
        granularity(Inc).

calendar(CalName) --> ident(CalName), !. 
calendar(fiscal)  --> [default], !. 
calendar(_)       --> [T], {generate_error_flag(calendar_expected, T)}.


granularity(year)   --> [add], [year], !.
granularity(month)  --> [add], [month], !.
granularity(day)    --> [add], [day], !.
granularity(hour)   --> [add], [hour], !.
granularity(minute) --> [add], [minute], !.
granularity(second) --> [add], [second], !.
granularity(none)   --> [].

/****************************************************************************/

table_name(Table) --> ident(Table), !.
table_name(_)     --> [Token], {generate_error_flag(table_name_expected, Token)}.

column_reference(Table-Column) --> 
        range_variable(Table), ['.'], !, column_name(Column).
column_reference(Column)    -->
        column_name(Column).

column_name(Name) --> [id(Name)], !.  /* column name */

range_variable(Name) --> [id(Name)], !.  /* table name oder alias */

ident(Name) --> [id(Name)], !.

constant(Value, number)   --> integer(Value).
constant(Value, number)   --> real(Value).
constant(Value, char)     --> string(Value).
constant(Value, interval) --> interval_literal(Value).
constant(Value, event)    --> event_literal(Value).
constant(Value, span)     --> span_literal(Value).
constant(null, _)         --> 
	[null], !, {generate_error_flag(not_implemented_yet, 'null-values')}.

integer(cst(Value), number)    --> 
        [+], !, [num(Value)], {number(Value)}.
integer(cst(RetValue, number)) --> [-], !, [num(Value)], 
        {number(Value), RetValue is -Value}.
integer(cst(Value, number))    --> 
        [num(Value)], {number(Value)}.

real(real(Value))  --> [+], !, [num(Value)], {float(Value)}. 
real(real(RetVal)) --> [-], !, [num(Value)], {float(Value), RetVal is -Value}.
real(real(Value))  --> [num(Value)], {float(Value)}.

string(cst(String, char)) --> [str(String)].

column_list([Column|ColumnList]) --> 
        column_name(Column),
        column_list2(ColumnList).
column_list2([Column|ColumnList]) -->
        [','], column_name(Column), column_list2(ColumnList).
column_list2([]) --> [].


data_type(n(f))        --> [float].
data_type(number)      --> [integer].
data_type(number)      --> [number].
data_type(interval)    --> [period].
data_type(event)       --> [timestamp].
data_type(event)       --> [date].
data_type(span)        --> [interval].
data_type(s(Length))   --> [char], 
        ['('], integer(cst(Length, number)), chk(')'), {Length>0}.
data_type(s(Length))   --> [varchar], 
        ['('], integer(cst(Length, number)), chk(')'), {Length>0}.

/****************************************************************************/
/* Mode:           parse(-ParseTree)                                        */
/* Purpose:        Reads a characterstring from the standard IO and parses  */
/*                 the corresponding token-list according to the ChronoSQL  */
/*                 syntax.                                                  */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

parse(ParseTree) :- scan(TokenList), sql(ParseTree, TokenList, []), !.
parse(error).

