/******************************************************************************/
/* Module:      scanner.pl                                                    */
/* Project:     TimeDB 1.06                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SICStus Prolog                                                */
/* Machine:     SPARC/Solaris, PowerMac/MacOS                                 */
/* Export:                                                                    */
/*              tokenize/3 (calendar.pl)                                      */
/*              upper_case/2 (meta_data.pl)                                   */
/*              lower_case/2 (meta_data.pl)                                   */
/*              scan/1 (parser.pl)                                            */
/* Import:                                                                    */
/******************************************************************************/

/****************************************************************************/
/* Mode:           readstring(-String)                                      */
/* Purpose:        Reads string from standard IO                            */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exists always                                            */
/* Redo:           Fails always                                             */
/****************************************************************************/

:- dynamic error/2.

readstring(Str) :- readlines(Str,'ATSQL2> ','        ').
readlines(L,First,Second) :-
  prompt(_,First),
  get0(X),
  prompt(_,Second),
  readln(X,L).
readln(-1,"eof;") :- !.
readln(10,L) :- !, get0(X), readln(X,L).
readln(H,[H|T]) :- readln2(T).
readln2([X|T]) :- get0(X), X\==0';, !, readln2(T).
readln2([0';,X|T]) :- get0(X), X\==10, !, readln2(T).
readln2([0';]).

/****************************************************************************/
/* Mode:           keyword(?Word)                                           */
/* Purpose:                                                                 */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exists always                                            */
/* Redo:           Fails always                                             */
/****************************************************************************/

keyword(abs).		keyword(add).		keyword(all).		
keyword(alter).		keyword(assertion).	keyword(and).
keyword(as).		keyword(asc).		keyword(avg).
keyword(between).	keyword(by).		keyword(char).
keyword(check).		keyword(count).		keyword(create).
keyword(commit).	keyword(delete).	keyword(desc).
keyword(distinct).	keyword(drop).		keyword(except).
keyword(exists).	keyword(from).		keyword(float).	
keyword(group).		keyword(having).	keyword(in).
keyword(is).		
keyword(insert).	keyword(integer).	keyword(intersect).	
keyword(into).		keyword(key).		keyword(like).		
keyword(max).		keyword(min).		keyword(modify).	
keyword(not).		keyword(null).		keyword(or).		
keyword(order).		keyword(primary).	keyword(references).	
keyword(rollback).	keyword(select).	keyword(set).
keyword(sum).		keyword(table).		keyword(union).		
keyword(update).	keyword(values).	keyword(varchar).
keyword(view).		keyword(where).

/* temporal keywords */
keyword(absolute).	keyword(begin).		keyword(beginning).
keyword(contains).	keyword(date).		keyword(end).		
keyword(first).		keyword(forever).
keyword(intersect).	keyword(interval).	keyword(last).		
keyword(meets).		keyword(now).		keyword(nonsequenced).
keyword(overlaps).	keyword(period).	keyword(precedes).	
keyword(present).	keyword(to).		keyword(valid).
keyword(validtime).		
keyword(timestamp).	keyword(transaction).	keyword(transactiontime).

/* calendar keywords */ 
keyword(calendric).	keyword(declare).	keyword(default).	
keyword(system).	keyword(with).
keyword(year).		keyword(month).		keyword(day).
keyword(hour).		keyword(minute).	keyword(second).

/* special functions */
keyword(clock).		keyword(help).		keyword(notrace).
keyword(quit).		keyword(remove).	keyword(status).
keyword(trace).         keyword(batch).         keyword(open).
keyword(close).

/* Special reserved words */
keyword('vts_#$').	keyword('vte_#$').
keyword('sts_#$').	keyword('ste_#$').



/****************************************************************************/
/* Mode:           start_word(-Character, +String, -RestString)             */
/* Purpose:        Allowed first characters in word                         */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

start_word(Char) --> [Char], {Char > 96, Char < 123}.  
start_word(Char) --> [Char], {Char > 64, Char < 91}. 


/****************************************************************************/
/* Mode:           in_word(-Character, +String, -RestString)                */
/* Purpose:                                                                 */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exists always                                            */
/* Redo:           Fails always                                             */
/****************************************************************************/

in_word(35)   --> "#". 
in_word(36)   --> "$". 
in_word(95)   --> "_".
in_word(Char) --> start_word(Char). 


/****************************************************************************/
/* Mode:           in_number(-Character, +String, -RestString)              */
/* Purpose:                                                                 */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exists always                                            */
/* Redo:           Fails always                                             */
/****************************************************************************/

in_number(Char) --> [Char], {Char > 47, Char < 58}.


/****************************************************************************/
/* Mode:           token(-Token, +String, -RestString)                      */
/* Purpose:        Consumes next token from string.                         */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

skip_comment	--> "*/", !.
skip_comment	--> [_], skip_comment.

token(T)	--> " ", !, token(T).
token(Token)	--> "/*", skip_comment, token(Token). 
token('>=')	--> ">=", !.
token('<=')	--> "<=", !.
token('<>')	--> "<>", !.
token('>')	--> ">", !.
token('<')	--> "<", !.
token('=')	--> "=", !.
token('+')	--> "+", !.
token('*')	--> "*", !.
token('-')	--> "-", !.
token('/')	--> "/", !.
token('(')	--> "(", !.
token(')')	--> ")", !.
token('.')	--> ".", !.
token(',')	--> ",", !.
token(':')	--> ":", !.
token('%')	--> "%", !.
token('|')	--> "|", !.
token('[')	--> "[", !.
token('~')	--> "~", !.
token('@')	--> "@", !.
token(';')	--> ";", !.
token(str(Str))	--> "'", !, scan_str(Str).
token(T)        --> start_word(X), !, rest_word(Chrs), 
      {lower_case([X|Chrs], LS), name(W, LS), (keyword(W), T = W; T = id(W))}.
token(num(Num))	--> in_number(N), !, rest_number(Rest), {number_chars(Num, [N|Rest])}.
token(T)	--> [_], token(T).
token([])	--> [].


/****************************************************************************/
/* Mode:           tokenize(-TokenList, +String, -RestString)               */
/* Purpose:                                                                 */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

tokenize(TokenList) --> token(Token), !, tokenize2(Token, TokenList).
tokenize2([], [])                   --> !, [].
tokenize2(Token, [Token|TokenList]) --> token(T), tokenize2(T, TokenList).


/****************************************************************************/
/* Mode:           rest_word(-RestWord, +String, -RestString)               */
/* Purpose:                                                                 */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

rest_word([Char|RestWord]) --> in_word(Char), rest_word(RestWord), !.
rest_word([Char|RestWord]) --> in_number(Char), rest_word(RestWord), !.
rest_word([])              --> [].


/****************************************************************************/
/* Mode:           scan_number(-Number, +String, -RestString)               */
/* Purpose:                                                                 */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exists always                                            */
/* Redo:           Fails always                                             */
/****************************************************************************/

read_mantissa([Char|Rest]) --> in_number(Char), !, read_mantissa(Rest).
read_mantissa([])          --> [].

read_exp_num([Char|Rest]) --> in_number(Char), !, read_exp_num(Rest).
read_exp_num([])          --> [].

read_exp_sign([43|Number]) --> "+", !, read_exp_num(Number).
read_exp_sign([45|Number]) --> "-", !, read_exp_num(Number).
read_exp_sign(Number)      --> read_exp_num(Number).

read_exponent([101|Sign])  --> "E", !, read_exp_sign(Sign).
read_exponent([101|Sign])  --> "e", !, read_exp_sign(Sign).
read_exponent([])          --> [].

rest_number([Char|RestNumber]) --> 
	in_number(Char), !, 
	rest_number(RestNumber).
rest_number([46|Number]) --> ".", 
	read_mantissa(Mantissa), 
	read_exponent(Exponent), 
	{append(Mantissa, Exponent, Number)}.
rest_number([]) --> [].


/****************************************************************************/
/* Mode:           scan_string(-String, +Str, -RestStr)                     */
/* Purpose:                                                                 */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exists always                                            */
/* Redo:           Fails always                                             */
/****************************************************************************/

scan_str([39|R])      --> "'", "'", !, scan_str(R).
scan_str([])	      --> "'", !.
scan_str([Char|Rest]) --> [Char], scan_str(Rest).


/****************************************************************************/
/* Mode:           upper_case(+InputList, -OutputList)                      */
/* Purpose:        Transform string to upper-case.                          */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exists always                                            */
/* Redo:           Fails always                                             */
/****************************************************************************/

upper_case([], []).
upper_case([H1|T1], [H2|T2]) :- 
	H1>=97, H1=<122, H2 is H1-32, !, upper_case(T1, T2).
upper_case([H|T1], [H|T2]) :- 
	upper_case(T1, T2).


/****************************************************************************/
/* Mode:           lower_case(+InputList, -OutputList)                      */
/* Purpose:        Transform string to lower-case.                          */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exists always                                            */
/* Redo:           Fails always                                             */
/****************************************************************************/

lower_case([], []).
lower_case([H1|T1], [H2|T2]) :-
	H1>=65, H1=<90, H2 is H1+32, !, lower_case(T1, T2).
lower_case([H|T1], [H|T2]) :- 
	lower_case(T1, T2).


/****************************************************************************/
/* Mode:           scan(-TokenList)                                         */
/* Purpose:        Reads string from standard IO and returns token-list.    */
/* Example:                                                                 */ 
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

scan(TokenList) :- readstring(String), tokenize(TokenList, String, []), !.
scan([error]).


