/******************************************************************************/
/* Module:      translate.pl                                                  */
/* Project:     TimeDB 1.04                                                   */
/* Author:      Andreas Steiner                                               */
/* Language:    SICStus Prolog                                                */
/* Machine:     SPARC/Solaris, PowerMac/MacOS                                 */
/* Export:                                                                    */
/*              translate_query/5 (constraints.pl)                            */
/*              translate/3 (constraints.pl,timeDB.pl)                        */
/* Import:                                                                    */
/*              get_expression/3 (basics.pl)                                  */
/*              generate_alias/1 (basics.pl)                                  */
/*              generate_table_name/2 (basics.pl)                             */
/*              generate_interval_col/3 (basics.pl)                           */
/*              get_interval_name/3 (basics.pl)                               */
/*              generate_error_flag/2 (errors.pl)                             */
/*              get_table_alias/4 (check.pl)                                  */
/*              get_column_length/2 (check.pl)                                */
/*              check_time_type/4 (check.pl)                                  */
/*              check_ambiguity/3 (check.pl)                                  */
/*              check_time_compatibility/3 (check.pl)                         */
/*              check_temporal_exp/3 (check.pl)                               */
/*              check_del_compatibility/2 (check.pl)                          */
/*              check_alter/3 (check.pl)                                      */
/*              check_length/2 (check.pl)                                     */
/*              check_reference/3 (check.pl)                                  */
/*              check_still_referenced/1 (check.pl)                           */
/*              check_column_names/1 (check.pl)                               */
/*              check_create_compatibility/5 (check.pl)                       */
/*              check_update_columns/4 (check.pl)                             */
/*              check_update_values/4 (check.pl)                              */
/*              check_insert_query_compatibility/2 (check.pl)                 */
/*              check_insert_values_compatibility/2 (check.pl)                */
/*              check_user_defined_valid/2 (check.pl)                         */
/*              check_union_compatibility/2 (check.pl)                        */
/*              check_union_time_compatibility/2 (check.pl)                   */
/*              check_column_constraint_number/3 (check.pl)                   */
/*              check_cond_type_compatibility/3 (check.pl)                    */
/*              check_exp_type/7 (check.pl)                                   */
/*              get_tt_date/2 (time.pl)                                       */
/*              tt_to_internal/4 (time.pl)                                    */
/*              get_table_type/2 (meta_data.pl)                               */
/*              get_table_infos/3 (meta_data.pl)                              */
/*              table_exists/1 (meta_data.pl)                                 */
/*              view_exists/1 (meta_data.pl)                                  */
/*              get_view_meta_data/3 (meta_data.pl)                           */
/******************************************************************************/

:- ensure_loaded([check]).


%% get_res_type(+Struct, -ResType, -TimeType),
get_res_type(union([U|_])-_-_-TT, SRT, TT)     :- !, get_res_type(U, SRT, _).
get_res_type(intersect([I|_])-_-_-TT, SRT, TT) :- !, get_res_type(I, SRT, _).
get_res_type(minus([I|_])-_-_-TT, SRT, TT)     :- !, get_res_type(I, SRT, _).
get_res_type(_-_-_-_-SRT-_-_-TT, SRT, TT)      :- !.


%% get_names_types(+ResType, -ColumnNames, -ColumnTypes)
get_names_types([], [], []) :- !.
get_names_types([valid(_)-_-interval-_|ResType], ['vts_#$', 'vte_#$'|Cols], 
                ['vts_#$'-interval-notnull, 'vte_#$'-interval-notnull|Types]) :-
%	added 19-03-96, as
        !,
        get_names_types(ResType, Cols, Types).
get_names_types([valid-interval-_|ResType], ['vts_#$', 'vte_#$'|Cols], 
                ['vts_#$'-interval-notnull, 'vte_#$'-interval-notnull|Types]) :-
        !,
        get_names_types(ResType, Cols, Types).
get_names_types([system-interval-_|ResType], ['sts_#$', 'ste_#$'|Cols], 
                ['sts_#$'-interval-notnull, 'ste_#$'-interval-notnull|Types]) :-
        !,
        get_names_types(ResType, Cols, Types).
get_names_types([(_-C)-_-interval-_|ResType], [S, E|Cols], 
                [S-interval-notnull, E-interval-notnull|Types]) :-
        !,
        generate_interval_col(C, S, E),
        get_names_types(ResType, Cols, Types).
get_names_types([C-_-interval-_|ResType], [S, E|Cols], 
                [S-interval-notnull, E-interval-notnull|Types]) :-
        !,
        generate_interval_col(C, S, E),
        get_names_types(ResType, Cols, Types).
get_names_types([(_-C)-_-T-_|ResType], [C|Cols], [C-T-notnull|Types]) :-
        atom(C),
        !,
        get_names_types(ResType, Cols, Types).
get_names_types([C-_-T-_|ResType], [C|Cols], [C-T-notnull|Types]) :-
        atom(C),
        !,
        get_names_types(ResType, Cols, Types).
get_names_types([_-_-T-_|ResType], [C|Cols], [C-T-notnull|Types]) :-
        generate_alias(C),
        get_names_types(ResType, Cols, Types).


%% resolve_ref(RefList)
resolve_ref([])                      :- !.
resolve_ref([T-C-_-_-_=T-C|RefList]) :- resolve_ref(RefList).


/****************************************************************************/
/* Mode:           normalize(+Formula, -NormalizedFormula)                  */
/* Purpose:        Transforms a formula into or-free-parts form. Used for   */
/*                 predicate-expressions in where-clause.                   */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

normalize(not(and(A, B)), F) :- 
        !,
        normalize(or(not(A), not(B)), F).
normalize(not(or(A, B)), F) :- 
        !,
        normalize(and(not(A), not(B)), F).
normalize(not(not(A)), A1) :- 
        !, 
        normalize(A, A1).
normalize(and(A, or(B, C)), F) :- 
        !,
        normalize(or(and(A, B), and(A, C)), F).
normalize(and(or(A, B), C), F) :- 
        !,
        normalize(or(and(A, C), and(B, C)), F).
normalize(not(A), F) :- 
        normalize(A, A1),
        A \== A1,
        normalize(not(A1), F).
normalize(and(A, B), F) :- 
        normalize(A, A1),
        normalize(B, B1),
        (A \== A1; B \== B1),
        !,
        normalize(and(A1, B1), F).
normalize(or(A, B), or(A1, B1)) :-
        !, 
        normalize(A, A1),
        normalize(B, B1).
normalize(F, F).


/****************************************************************************/
/* Mode:           generate_or_list(+Coal, +NormFormula, -OrList, -NewCoal) */
/* Purpose:        Transforms a normalized formula into a list of or-free   */
/*                 parts.                                                   */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

generate_and_list(and(A, B), [A|Rest]) :- !, generate_and_list(B, Rest).
generate_and_list(not(A), [not(A)])    :- !.
generate_and_list(A, [A])              :- !.


% Coalescing argument added, 09-02-1996, as
generate_or_list(C, [], [], C) :- !.
generate_or_list(_, or(A, B), [and(AND_List)|Rest], C) :-
        !,
        generate_and_list(A, AND_List),
        generate_or_list(red, B, Rest, C).
generate_or_list(C, A, [and(AND_List)], C) :-
        !,
        generate_and_list(A, AND_List).


/****************************************************************************/
/* Mode:          expand(+Time, +Attr, +Tables, +MetaInfo,                  */
/*                       -ExtSel, -ResType, -RefList)                       */
/* Purpose:       Expands all references and checks expressions.            */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

%% table_attrs(+Table, +MetaInfo, -Select, -ResType, -RefList)
table_attrs(_, [], [], [], []) :- !.
table_attrs(T-A, [T-'vts_#$'-interval-Len-ID,T-'vte_#$'-interval-Len-ID|MetaInfo], 
            Select, ResType, RefList) :- 
        !,
        table_attrs(T-A, MetaInfo, Select, ResType, RefList).
table_attrs(T-A, [T-'sts_#$'-interval-Len-ID,T-'ste_#$'-interval-Len-ID|MetaInfo], 
            Select, ResType, RefList) :- 
        !,
        table_attrs(T-A, MetaInfo, Select, ResType, RefList).
table_attrs(Tab-Al, 
            [Tab-S-interval-Len-ID, Tab-E-interval-Len-ID|MetaInfo], 
            [R-C1, R-C2|Select], 
            [Int-_-interval-Len|ResType], 
            [Al-S-interval-Len-ID=R-C1, Al-E-interval-Len-ID=R-C2|RefList]) :- 
        !,
        get_interval_name(Int, S, E), 
        table_attrs(Tab-Al, MetaInfo, Select, ResType, RefList).
table_attrs(Tab-Al, 
            [Tab-Col-Type-Len-ID|MetaInfo], 
            [R-C|Select], 
            [Col-_-Type-Len|ResType], 
            [Al-Col-Type-Len-ID=R-C|RefList]) :- 
        !,
        table_attrs(Tab-Al, MetaInfo, Select, ResType, RefList).
table_attrs(T-A, [_|MetaInfo], Select, ResType, RefList) :- 
        !,
        table_attrs(T-A, MetaInfo, Select, ResType, RefList).


all_table_attrs([], _, [], [], []) :- !.
all_table_attrs([Table-Alias-_|From], MetaInfo, Select, ResType, RefList) :-
        table_attrs(Table-Alias, MetaInfo, Select1, ResType1, RefList1),
        all_table_attrs(From, MetaInfo, Select2, ResType2, RefList2),   
        append(Select1, Select2, Select),
        append(ResType1, ResType2, ResType),
        append(RefList1, RefList2, RefList).


%% table_time(+Exp, +Type, +Len, +Select, +ResType, -NewSelect, -NewResType)
table_time(interval(S, E), interval, L, Sel, ResType,
           [expr(S,'vts_#$'),expr(E,'vte_#$')|Sel], [valid-interval-L|ResType]) :- !.
table_time(Event, event, L, Sel, ResType,
           [expr(Event, 'event#$')|Sel], [at-_-event-L|ResType]) :- !.
table_time(_, _, _, _, _, _, _) :-
         generate_error_flag(interval_event_tables, _).


expand(_, [], _, _, [], [], []) :- !.
expand(Time, [all|Rest], From, MetaInfo, Select, ResType, RefList) :-
        !,
        all_table_attrs(From, MetaInfo, Select1, ResType1, RefList1),
        expand(Time, Rest, From, MetaInfo, Select2, ResType2, RefList2),
        append(Select1, Select2, Select),
        append(ResType1, ResType2, ResType),
        append(RefList1, RefList2, RefList).
expand(Time, [Ref-all|Rest], From, MetaInfo, Select, ResType, RefList) :-
        !,
        (get_table_alias(Ref, From, Table, Alias), !;
         generate_error_flag(illegal_reference, Ref)),
        table_attrs(Table-Alias, MetaInfo, Select1, ResType1, RefList1),
        expand(Time, Rest, From, MetaInfo, Select2, ResType2, RefList2),
        append(Select1, Select2, Select),
        append(ResType1, ResType2, ResType),
        append(RefList1, RefList2, RefList).
expand(Time, [expr(Int, valid)|Rest], From, MetaInfo, NewSelect, NewResType, RefList) :-
        !,
        check_user_defined_valid(Time, valid),
        check_exp_type(Int, MetaInfo, From+[], NewExp, Type, Len, RefList1),
        expand(Time, Rest, From, MetaInfo, Select, ResType, RefList2),
	table_time(NewExp, Type, Len, Select, ResType, NewSelect, NewResType),
        append(RefList1, RefList2, RefList).
expand(Time, [expr(Exp, Alias)|Rest], From, MetaInfo, [expr(NewExp, Alias)|Select], 
       [Alias-_-Type-Len|ResType], RefList) :-
        !,
        check_exp_type(Exp, MetaInfo, From+[], NewExp, Type, Len, RefList1),
        expand(Time, Rest, From, MetaInfo, Select, ResType, RefList2),
        append(RefList1, RefList2, RefList).
expand(Time, [Exp|Rest], From, MetaInfo, [expr(NewExp)|Select], 
       [Exp-_-Type-Len|ResType], RefList) :- 
        check_exp_type(Exp, MetaInfo, From+[], NewExp, Type, Len, RefList1),
        expand(Time, Rest, From, MetaInfo, Select, ResType, RefList2),
        append(RefList1, RefList2, RefList),
        !.


/****************************************************************************/
/* Mode:          system_intersect(+TableList, -Greatest, -LeastList)       */
/* Mode:          valid_intersect(+TableList, -Greatest, -LeastList)        */
/* Purpose:       Generates lists for temporal intersection.                */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

system_intersect1([], [], []) :- !.
system_intersect1([_-A-bitemporal|Rest], [abs(A-'sts_#$')|G], [abs(A-'ste_#$')|L]) :-
        !,
        system_intersect1(Rest, G, L).
system_intersect1([_-A-system|Rest], [abs(A-'sts_#$')|G], [abs(A-'ste_#$')|L]) :-
        !,
        system_intersect1(Rest, G, L).
system_intersect1([Table-_-Type|_], _, _) :-
	generate_error_flag(no_tt_table, Table-Type).

system_intersect(TableList, Gst, Lst) :-
        system_intersect1(TableList, Gst, Lst), !.


valid_intersect1([], [], []) :- !.
valid_intersect1([_-A-bitemporal|Rest], [A-'vts_#$'|Gst], [A-'vte_#$'|Lst]) :-
        !,
        valid_intersect1(Rest, Gst, Lst).
valid_intersect1([_-A-valid|Rest], [A-'vts_#$'|Gst], [A-'vte_#$'|Lst]) :-
        !,
        valid_intersect1(Rest, Gst, Lst).
valid_intersect1([Table-_-Type|_], _, _) :-
	generate_error_flag(no_vt_table, Table-Type).

valid_intersect(TableList, Gst, Lst) :-
        valid_intersect1(TableList, Gst, Lst), !.


/****************************************************************************/
/* Mode:          get_all_attributes(+JT+From, +MetaInfo,                   */
/*                                   -Select, -ResType, -RefList)           */
/* Purpose:       Returns for every table in TableList Table-all.           */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

get_attr(_, [], [], [], []) :- !.
get_attr(T-As-TT, 
         [T-'vts_#$'-interval-L-ID, T-'vte_#$'-interval-L-ID|MetaInfo], 
         [interval(As-'vts_#$', As-'vte_#$')|Attrs], 
         [vtime(T)-(C1, C2)-interval-L|ResType],
         [As-'vts_#$'-interval-L-ID=C1,As-'vte_#$'-interval-L-ID=C2|RefList]) :-
        !,
        get_attr(T-As-TT, MetaInfo, Attrs, ResType, RefList).
get_attr(T-As-TT, 
         [T-'sts_#$'-interval-L-ID, T-'ste_#$'-interval-L-ID|MetaInfo], 
         [interval(As-'sts_#$', As-'ste_#$')|Attrs], 
         [xtime(T)-(C1, C2)-interval-L|ResType],
         [As-'sts_#$'-interval-L-ID=C1,As-'ste_#$'-interval-L-ID=C2|RefList]) :-
        !,
        get_attr(T-As-TT, MetaInfo, Attrs, ResType, RefList).
get_attr(T-As-TT, 
         [T-S-interval-Len-ID, T-E-interval-Len-ID|MetaInfo], 
         [As-S, As-E|Attrs], 
         [Int-(C1, C2)-interval-Len|ResType],
         [As-S-Type-Len-Id=C1, As-E-Type-Len-Id=C2|RefList]) :-
        !,
        get_interval_name(Int, S, E), 
        get_attr(T-As-TT, MetaInfo, Attrs, ResType, RefList).
get_attr(T-As-TT, 
         [T-C-Type-Len-Id|MetaInfo], 
         [As-C|Attrs], 
         [C-C1-Type-Len|ResType],
         [As-C-Type-Len-Id=C1|RefList]) :-
        !,
        get_attr(T-As-TT, MetaInfo, Attrs, ResType, RefList).
get_attr(T-As-TT, 
         [_|MetaInfo], 
         Attrs, 
         ResType,
         RefList) :-
        !,
        get_attr(T-As-TT, MetaInfo, Attrs, ResType, RefList).


get_all_attributes([]+[], _, [], [], []) :- !.
get_all_attributes([JT|Rest]+F, MetaInfo, Select, ResType, RefList) :- 
        !,
        get_attr(JT, MetaInfo, Attrs, AttrsResType, AttrRefList),
        get_all_attributes(Rest+F, MetaInfo, Select1, ResType1, RefList1),
        append(Attrs, Select1, Select),
        append(AttrsResType, ResType1, ResType),
        append(AttrRefList, RefList1, RefList).
get_all_attributes([]+[Table|Rest], MetaInfo, Select, ResType, RefList) :- 
        !,
        get_attr(Table, MetaInfo, Attrs, AttrsResType, AttrRefList),
        get_all_attributes([]+Rest, MetaInfo, Select1, ResType1, RefList1),
        append(Attrs, Select1, Select),
        append(AttrsResType, ResType1, ResType),
        append(AttrRefList, RefList1, RefList).


/****************************************************************************/
/* Mode:         add_time(+TUC, +TableTime, +Values,                        */
/*                        +ColNames, +ColTypes, +DerivedTables,             */
/*                        -NewVals, -NewColNames, -NewColTypes, -NewDerived)*/
/* Purpose:      Adds transaction and/or valid time (TUC) [now-forever) to  */
/*               INSERT- and CREATE-statements with subquery.               */
/* Example:                                                                 */
/* Sideeffects:  None                                                       */
/* Call:         Exits always                                               */
/* Redo:         Fails always                                               */
/****************************************************************************/

create_select([], [])                  :- !.
create_select([C|Rest], [C-noref|Sel]) :- !, create_select(Rest, Sel).
        
%% add_time(Now, TUC, TT, Query, Cols, Types, Deriv, NVal, NewCols, NewTypes, NDeriv),
add_time(N, _, system, query(Q), C, CT, D, 
         [sel([N, cst(F, number)|S]), from([T]), where([])], 
         ['sts_#$', 'ste_#$'|C], 
         ['sts_#$'-interval-notnull, 'ste_#$'-interval-notnull|CT], 
         [T-Q-C|D]) :- 
        !, get_tt_date(forever, F),
        generate_table_name("aux", T),
        create_select(C, S).
add_time(N, tuc, valid, query(Q), C, CT, D, 
         [sel([abs(N), abs(cst(F, number))|S]), from([T]), where([])], 
         ['vts_#$', 'vte_#$'|C], 
         ['vts_#$'-interval-notnull, 'vte_#$'-interval-notnull|CT], 
         [T-Q-C|D]) :- 
        !, get_tt_date(forever, F),
        generate_table_name("aux", T),
        create_select(C, S).
add_time(N, tuc, bitemporal, query(Q), C, CT, D,
         [sel([abs(N), abs(cst(F, number)), N, cst(F, number)|S]), from([T]), where([])],
         ['vts_#$', 'vte_#$', 'sts_#$', 'ste_#$'|C],
         ['vts_#$'-interval-notnull, 'vte_#$'-interval-notnull,
          'sts_#$'-interval-notnull, 'ste_#$'-interval-notnull|CT],
         [T-Q-C|D]) :- 
        !, get_tt_date(forever, F),
        generate_table_name("aux", T),
        create_select(C, S).
add_time(N, _, bitemporal, query(Q), [VS, VE|C], [VSCT, VECT|CT], D,
         [sel([VTS, VTE, N, cst(F, number)|S]), from([T]), where([])],
         [VS, VE, 'sts_#$', 'ste_#$'|C],
         [VSCT, VECT, 'sts_#$'-interval-notnull, 'ste_#$'-interval-notnull|CT],
         [T-Q-[VS, VE|C]|D]) :- 
        !, get_tt_date(forever, F),
        generate_table_name("aux", T),
        create_select([VS, VE|C], [VTS, VTE|S]).
add_time(_, _, _, query(Q), C, CT, D, [sel(S),from([T]),where([])], C, CT, [T-Q-C|D]) :- 
        !, generate_table_name("aux", T),
        create_select(C, S).


/****************************************************************************/
/* Mode:          translate_select(+Select, +From, +MetaInfo,               */
/*                        -Sel-ResType-RefList, -AllSel-AllResType)         */
/* Purpose:       Expands the select clause and checks expressions.         */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

translate_select(distinct(S), F, MetaInfo, distinct(S1)-SRT-RefList, All) :-
        !, translate_select(S, F, MetaInfo, S1-SRT-RefList, All).

translate_select(snapshot(S), F, MetaInfo, snapshot(S1)-SRT-RefList, 
                 snapshot(All)-ART-AllRefList) :-
        !,
        expand(snapshot, S, F, MetaInfo, S1, SRT, RefList),
        get_all_attributes([]+F, MetaInfo, All, ART, AllRefList).
        
translate_select(valid(S), F, MetaInfo, 
                 valid(['vts_#$'-noref, 'vte_#$'-noref|S1])-
                                            [valid-interval-Len|SRT]-RefList,  
                 valid([expr(gst(G),'vts_#$'), expr(lst(L),'vte_#$')|All])-
                                            [valid-interval-Len|ART]-AllRefList) :-
        !,
        expand(valid, S, F, MetaInfo, S1, SRT, RefList),
        valid_intersect(F, G, L),
        get_all_attributes([]+F, MetaInfo, All, ART, AllRefList),
        get_column_length(interval, Len).

translate_select(system(S), F, MetaInfo, 
                 system(['sts_#$'-noref, 'ste_#$'-noref|S1])-
                                             [system-interval-Len|SRT]-RefList,
                 system([expr(gst(G),'sts_#$'), expr(lst(L),'ste_#$')|All])-
                                             [system-interval-Len|ART]-AllRefList) :-
        !,
        expand(system, S, F, MetaInfo, S1, SRT, RefList),
        system_intersect(F, G, L),
        get_all_attributes([]+F, MetaInfo, All, ART, AllRefList),
        get_column_length(interval, Len).

translate_select(validsystem(S), F, MetaInfo, 
                 validsystem(['vts_#$'-noref, 'vte_#$'-noref, 
                              'sts_#$'-noref, 'ste_#$'-noref|S1])-
                       [valid-interval-Len, system-interval-Len|SRT]-RefList,
                 validsystem([expr(gst(G1),'vts_#$'), expr(lst(L1),'vte_#$'),
                              expr(gst(G2),'sts_#$'), expr(lst(L2),'ste_#$')|All])-
                       [valid-interval-Len, system-interval-Len|ART]-AllRefList) :-
        !,
        expand(validsystem, S, F, MetaInfo, S1, SRT, RefList),
        valid_intersect(F, G1, L1),
        system_intersect(F, G2, L2),
        get_all_attributes([]+F, MetaInfo, All, ART, AllRefList),
        get_column_length(interval, Len).

translate_select(systemvalid(S), F, MetaInfo, 
                 systemvalid(['sts_#$'-noref, 'ste_#$'-noref, 
                              'vts_#$'-noref, 'vte_#$'-noref|S1])-
                        [system-interval-Len, valid-interval-Len|SRT]-RefList,
                 systemvalid([expr(gst(G2),'sts_#$'), expr(lst(L2),'ste_#$'),
                              expr(gst(G1),'vts_#$'), expr(lst(L1),'vte_#$')|All])-
                        [system-interval-Len, valid-interval-Len|ART]-AllRefList) :-
        !,
        expand(systemvalid, S, F, MetaInfo, S1, SRT, RefList),
        valid_intersect(F, G1, L1),
        system_intersect(F, G2, L2),
        get_all_attributes([]+F, MetaInfo, All, ART, AllRefList),
        get_column_length(interval, Len).


/****************************************************************************/
/* Mode:          translate_all_select(+Select, +JT+From, +MetaInfo,        */
/*                                     -ExtSel-ResType, -Time)              */
/* Purpose:       Expands subquery select clauses and checks expressions.   */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

translate_all_select(snapshot(_), JT+F, MetaInfo, snapshot(S)-ART) :-
        !,
        get_all_attributes(JT+F, MetaInfo, S, ART, _).

translate_all_select(valid(_), JT+F, MetaInfo, 
                     valid([expr(gst(G),'vts_#$'), expr(lst(L),'vte_#$')|S])-
                                                      [valid-interval-Len|ART]) :-
        !,
        append(JT, F, JTF),
        valid_intersect(JTF, G, L),
        get_all_attributes(JT+F, MetaInfo, S, ART, _),
        get_column_length(interval, Len).

translate_all_select(system(_), JT+F, MetaInfo, 
                     system([expr(gst(G),'sts_#$'), expr(lst(L),'ste_#$')|S])-
                                                      [system-interval-Len|ART]) :-
        !,
        append(JT, F, JTF),
        system_intersect(JTF, G, L),
        get_all_attributes(JT+F, MetaInfo, S, ART, _),
        get_column_length(interval, Len).

translate_all_select(validsystem(_), JT+F, MetaInfo, 
                     validsystem([expr(gst(G1),'vts_#$'),expr(lst(L1),'vte_#$'), 
                                  expr(gst(G2),'sts_#$'),expr(lst(L2),'ste_#$')|S])-
                                  [valid-interval-Length, system-interval-Length|ART]) :-
        !,
        append(JT, F, JTF),
        valid_intersect(JTF, G1, L1),
        system_intersect(JTF, G2, L2),
        get_all_attributes(JT+F, MetaInfo, S, ART, _),
        get_column_length(interval, Length).

translate_all_select(systemvalid(_), JT+F, MetaInfo, 
                     systemvalid([expr(gst(G2),'sts_#$'),expr(lst(L2),'ste_#$'),
                                  expr(gst(G1),'vts_#$'),expr(lst(L1),'vte_#$')|S])-
                                  [system-interval-Length, valid-interval-Length|ART]) :-
        !,
        append(JT, F, JTF),
        valid_intersect(JTF, G1, L1),
        system_intersect(JTF, G2, L2),
        get_all_attributes(JT+F, MetaInfo, S, ART, _), 
        get_column_length(interval, Length).


/****************************************************************************/
/* Mode:          translate_from(+Modus, +JT+From, +MetaInfo, -NewMetaInfo, */ 
/*                               -DerivedTables, -NJT+NewFrom, -TimeType)   */
/* Purpose:       Gets attribute list out of metatable and extracts         */
/*                derived tables.                                           */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

%% get_time_type(+TimeType1, +TimeType2, -TimeType)
get_time_type(bitemporal, _, bitemporal)  :- !.
get_time_type(validsystem, _, bitemporal) :- !.
get_time_type(systemvalid, _, bitemporal) :- !.
get_time_type(_, bitemporal, bitemporal)  :- !.
get_time_type(snapshot, valid, valid)     :- !.
get_time_type(valid, snapshot, valid)     :- !.
get_time_type([], Type, Type)             :- !.
get_time_type(Type, [], Type)             :- !.
get_time_type(Type, Type, Type)           :- !.


%% add_meta_info(+Table, +ResType, +MetaInfo, -NewMetaInfo, -Cols)
add_meta_info(_, [], MetaInfo, MetaInfo, []) :- !.
add_meta_info(T, 
              [valid-interval-L|Rest], 
              MetaInfo,        
              [T-'vts_#$'-interval-L-(-2),T-'vte_#$'-interval-L-(-2)|NewMetaInfo], 
              ['vts_#$', 'vte_#$'|Columns]) :-
        !,
        add_meta_info(T, Rest, MetaInfo, NewMetaInfo, Columns).
add_meta_info(T, 
              [system-interval-L|Rest], 
              MetaInfo,        
              [T-'sts_#$'-interval-L-(-2),T-'ste_#$'-interval-L-(-2)|NewMetaInfo], 
              ['sts_#$', 'ste_#$'|Columns]) :-
        !,
        add_meta_info(T, Rest, MetaInfo, NewMetaInfo, Columns).
add_meta_info(T, 
              [Col-_-interval-Len|Rest],
              MetaInfo, 
              [T-S-interval-Len-(-5), T-E-interval-Len-(-5)|NewMetaInfo], 
              [S, E|Cols]) :-
        atom(Col),
        !,
	check_ambiguity(T, Col, Rest),
        generate_interval_col(Col, S, E),
        add_meta_info(T, Rest, MetaInfo, NewMetaInfo, Cols).
add_meta_info(Table, [Col-_-Type-Len|Rest], MetaInfo, 
                     [Table-Col-Type-Len-(-5)|NewMetaInfo], [Col|Columns]) :-
        atom(Col),
        !, 
	check_ambiguity(Table, Col, Rest),
        add_meta_info(Table, Rest, MetaInfo, NewMetaInfo, Columns).
add_meta_info(Table, [_-Col-_-Type-Len|Rest], MetaInfo, 
                     [Table-Col-Type-Len-(-5)|NewMetaInfo], [Col|Columns]) :-
        !,
        atom(Col),
	check_ambiguity(Table, Col, Rest),
        add_meta_info(Table, Rest, MetaInfo, NewMetaInfo, Columns).
add_meta_info(_, [Exp-_-_-_|_], _, _, _) :-
        get_expression(Exp, ExpStr, []),
        atom_chars(Exp1, ExpStr),
        generate_error_flag(no_expression_allowed, Exp1).


rename_columns1([], [], []) :- !.
rename_columns1([S, E|Cols], [valid(_)-_-interval-Len|Rest1], 
                [C-_-interval-Len|Rest2]) :- 
	% added 21-03-96, as
	S \== 'vts_#$', E \== 'vte_#$',
	S \== 'sts_#$', E \== 'ste_#$',
        !,
	get_interval_name(C, S, E), 
	rename_columns1(Cols, Rest1, Rest2).
rename_columns1([_, _|Cols], [T-interval-Len|Rest1], [T-interval-Len|Rest2]) :- 
        !, rename_columns1(Cols, Rest1, Rest2).
rename_columns1([C|Cols], [_-_-Type-Len|Rest1], [C-_-Type-Len|Rest2]) :- 
        !, rename_columns1(Cols, Rest1, Rest2).
rename_columns1(_, _, _) :- generate_error_flag(number_of_columns, _).

rename_columns([], SRT, SRT)      :- !.
rename_columns(Cols, SRT, NewSRT) :- rename_columns1(Cols, SRT, NewSRT).


translate_view_query(View, Query, ViewCols, MI, Struct, NMI, Derived, Cols, TT) :- 
        translate_query(Query, [], [], Derived, Struct),
        get_res_type(Struct, SRT, TT),
        rename_columns(ViewCols, SRT, SRT1),
        add_meta_info(View, SRT1, MI, NMI, Cols).


translate_from(_, []+[], MetaInfo, MetaInfo, [], []+[], []) :- !.
translate_from(M, [T-A-TT1|JT]+F, MetaInfo, NMI, Derived, [T-A-TT1|NJT]+NF, TT) :-
        !,
        translate_from(M, JT+F, MetaInfo, NMI, Derived, NJT+NF, TT).
translate_from(M, []+[derived_table(Query, Alias, ViewCols)|Rest], MetaInfo, NMI, 
               [View-Struct-Cols|Derived], JT+[View-Alias-TT1|From], TT) :-
        !,
        name(Alias, AliasStr),
        generate_table_name(AliasStr, View),
        translate_view_query(View, Query, ViewCols, MetaInfo, 
                             Struct, MetaInfo1, Derived1, Cols, TT1),
        translate_from(M, []+Rest, MetaInfo1, NMI, Derived2, JT+From, TT2),
        check_time_type(M, TT1, TT2, TT),
        append(Derived1, Derived2, Derived).
translate_from(M, []+[T-A-TT1|Rest], MetaInfo, NMI, Derived, JT+[T-A-TT1|From], TT) :-
        get_table_infos(T, MetaInfo, MetaInfo1),
        !,
        translate_from(M, []+Rest, MetaInfo1, NMI, Derived, JT+From, TT2),
        check_time_type(M, TT1, TT2, TT).
translate_from(M, []+[View-Alias|Rest], MetaInfo, NMI, 
               [View-Struct-Cols|Derived], JT+[View-Alias-TT1|From], TT) :-
        view_exists(View),
        !,
        get_view_meta_data(View, Query, ViewCols),
        translate_view_query(View, Query, ViewCols, MetaInfo, 
                             Struct, MetaInfo1, Derived1, Cols, TT1),
        translate_from(M, []+Rest, MetaInfo1, NMI, Derived2, JT+From, TT2),
        check_time_type(M, TT1, TT2, TT),
        append(Derived1, Derived2, Derived).
translate_from(M, []+[T-A|Rest], MetaInfo, NMI, Derived, JT+[T-A-TT1|From], TT) :-
        get_table_infos(T, MetaInfo, MetaInfo1),
        get_table_type(T, TT1),
        !,
        translate_from(M, []+Rest, MetaInfo1, NMI, Derived, JT+From, TT2),
        check_time_type(M, TT1, TT2, TT).


/****************************************************************************/
/* Mode:          translate_subqueries(+SuperQuery, +Subquery,              */
/*                                     +JT+From, +MetaInfo,                 */
/*                                     -DerivedTabs, -SetOpStr)             */
/* Purpose:       Translates subqueries in where clause to minus-, union-   */
/*                and intersect-expressions.                                */
/* Example:                                                                 */
/* Sideeffects:   None                                                      */
/* Call:          Exits always                                              */
/* Redo:          Fails always                                              */
/****************************************************************************/

change_select(snapshot([Ref]), snapshot([all]), Ref) :- Ref \== all, !.
change_select(valid([Ref]), valid([all]), Ref)       :- Ref \== all, !.
change_select(system([Ref]), system([all]), Ref)     :- Ref \== all, !.
change_select(_, _, _) :- generate_error_flag(single_column, _).


%% add_in_conditions(+Op, +Exp, +Query, -NewQuery)
add_in_conditions(Op, Exp, sfw([S1, F, where([])], Flags), 
                           sfw([S2, F, where(rel(Op, Exp, Exp1))], Flags)) :-
        !,
        change_select(S1, S2, Exp1).
add_in_conditions(Op, E1, sfw([S1, F, where(W)], Flags), 
                          sfw([S2, F, where(and(rel(Op, E1, E2), W))], Flags)) :-
        change_select(S1, S2, E2).


translate_subqueries1([], _, _, [], [], []) :- !.
translate_subqueries1([not(exists(query(Query)))|SubQueries], JT+F, MetaInfo,
                      Derived, [SetOpStr|Minus], Intersect) :- 
        !,
        append(JT, F, JTF), 
        translate_query(Query, JTF, MetaInfo, D1, SetOpStr),
        translate_subqueries1(SubQueries, JT+F, MetaInfo, D2, Minus, Intersect),
        append(D1, D2, Derived).

translate_subqueries1([exists(query(Query))|SubQueries], JT+F, MetaInfo,
                      Derived, Minus, [SetOpStr|Intersect]) :-
        !,
        append(JT, F, JTF), 
        translate_query(Query, JTF, MetaInfo, D1, SetOpStr),
        translate_subqueries1(SubQueries, JT+F, MetaInfo, D2, Minus, Intersect),
        append(D1, D2, Derived).
        
translate_subqueries1([not(in(Exp, query(Query)))|Rest], JT+F, MetaInfo,
                      Derived, [SetOpStr|Minus], Intersect) :-
        !,
        check_exp_type(Exp, MetaInfo, F+JT, NewExp, _, _, RefList),
        resolve_ref(RefList),
        add_in_conditions(eq, NewExp, Query, NewQuery),
        append(JT, F, JTF), 
        translate_query(NewQuery, JTF, MetaInfo, D1, SetOpStr),
        translate_subqueries1(Rest, JT+F, MetaInfo, D2, Minus, Intersect),
        append(D1, D2, Derived).

translate_subqueries1([in(Exp, query(Query))|Rest], JT+F, MetaInfo,
                      Derived, Minus, [SetOpStr|Intersect]) :-
        !,
        check_exp_type(Exp, MetaInfo, F+JT, NewExp, _, _, RefList),
        resolve_ref(RefList),
        add_in_conditions(eq, NewExp, Query, NewQuery),
        append(JT, F, JTF), 
        translate_query(NewQuery, JTF, MetaInfo, D1, SetOpStr),
        translate_subqueries1(Rest, JT+F, MetaInfo, D2, Minus, Intersect),
        append(D1, D2, Derived).

translate_subqueries1([rel(Op, Exp, query(Query))|Rest], JT+F, MetaInfo,
                      Derived, Minus, [SetOpStr|Intersect]) :-
        !,
        check_exp_type(Exp, MetaInfo, F+JT, NewExp, _, _, RefList),
        resolve_ref(RefList),
        add_in_conditions(Op, NewExp, Query, NewQuery),
        append(JT, F, JTF), 
        translate_query(NewQuery, JTF, MetaInfo, D1, SetOpStr),
        translate_subqueries1(Rest, JT+F, MetaInfo, D2, Minus, Intersect),
        append(D1, D2, Derived).


translate_subqueries(SFW, SubQueries, JT+F, MetaInfo, Derived, and(SFW, M, I)) :-
        translate_subqueries1(SubQueries, JT+F, MetaInfo, Derived, M, I).
         

/****************************************************************************/
/* Mode:         translate_where(+Select, +JT+From, +OrFreeParts, +MetaInfo,*/
/*                               -DerivedTabs, -SetOpStruct)                */  
/* Purpose:      Checks if compare conditions are type compatible, if       */
/*               referenced columns exist, replaces references to tables    */
/*               of super-queries and replaces subqueries by minus-,        */
/*               intersect- and union-expressions.                          */
/* Example:                                                                 */
/* Sideeffects:  None                                                       */
/* Call:         Exits always                                               */
/* Redo:         Fails always                                               */
/****************************************************************************/

%% translate_temp_compare(+Op, +Type1, +Exp1, +Type2, +Exp2, -TransCond)
translate_temp_compare(precedes, event, E1, event, E2, [rel(ls, E1, E2)]) :- !.
translate_temp_compare(overlaps, interval, interval(S, E), event, Event, 
                       [rel(gq, Event, S), rel(ls, Event, E)]) :- !.
translate_temp_compare(precedes, interval, interval(_, E), event, Event, 
                       [rel(ls, E, Event)]) :- !.
translate_temp_compare(overlaps, event, Event, interval, interval(S, E), 
                       [rel(gq, Event, S), rel(ls, Event, E)]) :- !.
translate_temp_compare(precedes, event, Event, interval, interval(S, _),
                       [rel(ls, Event, S)]) :- !.
translate_temp_compare(precedes, interval, interval(_, E1), interval, 
                       interval(S2, _), [rel(ls, E1, S2)]) :- !.
translate_temp_compare(eq, interval, interval(S1, E1), interval, 
                       interval(S2, E2), [rel(eq, S1, S2), rel(eq, E1, E2)]) :- !.
translate_temp_compare(meets, interval, interval(_, E1), interval, 
                       interval(S2, _), [rel(eq, E1, S2)]) :- !.
translate_temp_compare(overlaps, interval, interval(S1, E1), interval, 
                       interval(S2, E2), [rel(ls, S1, E2), rel(gr, E1, S2)]) :- !.
translate_temp_compare(contains, interval, interval(S1, E1), interval, 
                       interval(S2, E2), [rel(gq, S2, S1), rel(lq, E2, E1)]) :- !.
translate_temp_compare(Op, _, Exp1, _, Exp2, [rel(Op, Exp1, Exp2)]).


%% extract_subqueries(+OrFreePart, -SubQueries, -Conditions)
extract_subqueries([], [], []) :- !.
extract_subqueries([rel(Op, E, query(Q))|AndList], 
                   [rel(Op, E, query(Q))|SubQueries], ExprList) :-
        !,
        extract_subqueries(AndList, SubQueries, ExprList).
extract_subqueries([rel(Op, E1, E2)|AndList], SubQueries, 
                   [rel(Op, E1, E2)|ExprList]) :-
        !,
        extract_subqueries(AndList, SubQueries, ExprList).
extract_subqueries([not(rel(Op, E1, E2))|AndList], SubQueries, 
                   [not(rel(Op, E1, E2))|ExprList]) :-
        !,
        extract_subqueries(AndList, SubQueries, ExprList).
extract_subqueries([between(Col, Exp1, Exp2)|AndList], SubQueries, 
                    [between(Col, Exp1, Exp2)|ExprList]) :-
        !,
        extract_subqueries(AndList, SubQueries, ExprList).
extract_subqueries([in(Exp, cst(List))|AndList], SubQueries, 
                    [in(Exp, cst(List))|ExprList]) :-
        !,
        extract_subqueries(AndList, SubQueries, ExprList).
extract_subqueries([Sub|AndList], [Sub|SubQueries], ExprList) :-
        !,
        extract_subqueries(AndList, SubQueries, ExprList).


%% translate_in(+Exp, +Type, +ConstantList, -OrCondition)
translate_in1(_, _, [], []) :- !.
translate_in1(Exp, Type, [Const|Rest], [rel(eq, Exp, Const)|CondList]) :-
        translate_in1(Exp, Type, Rest, CondList).
translate_in(Exp, Type, ConstantList, or(CondList)) :-
        translate_in1(Exp, Type, ConstantList, CondList).

get_neg_op(eq, nq) :- !.
get_neg_op(ls, gq) :- !.
get_neg_op(gr, lq) :- !.
get_neg_op(gq, ls) :- !.
get_neg_op(lq, gr) :- !.
get_neg_op(nq, eq) :- !.
get_neg_op(Op, _)  :- !, generate_error_flag(not_implemented_yet, not(Op)).

translate_where1([], _, _, []) :- !.
translate_where1([not(rel(Op, Exp1, Exp2))|Rest], From+JT, MetaInfo, CondList) :-
        !,
        check_exp_type(Exp1, MetaInfo, From+JT, NewExp1, Type1, _, RefList1),
        resolve_ref(RefList1),
        check_exp_type(Exp2, MetaInfo, From+JT, NewExp2, Type2, _, RefList2),
        resolve_ref(RefList2),
        get_neg_op(Op, Op1),
        check_cond_type_compatibility(Op1, Type1, Type2),
        translate_temp_compare(Op1, Type1, NewExp1, Type2, NewExp2, Cond1),
        translate_where1(Rest, From+JT, MetaInfo, CondRest),
        append(Cond1, CondRest, CondList).
translate_where1([rel(Op, Exp1, Exp2)|Rest], From+JT, MetaInfo, CondList) :-
        !,
        check_exp_type(Exp1, MetaInfo, From+JT, NewExp1, Type1, _, RefList1),
        resolve_ref(RefList1),
        check_exp_type(Exp2, MetaInfo, From+JT, NewExp2, Type2, _, RefList2),
        resolve_ref(RefList2),
        check_cond_type_compatibility(Op, Type1, Type2),
        translate_temp_compare(Op, Type1, NewExp1, Type2, NewExp2, Cond1),
        translate_where1(Rest, From+JT, MetaInfo, CondRest),
        append(Cond1, CondRest, CondList).
translate_where1([between(Col, Exp1, Exp2)|Rest], From+JT, MetaInfo, 
                 [rel(lq, NewExp1, NewCol), rel(lq, NewCol, NewExp2)|CondRest]) :-
        !,
        check_exp_type(Col, MetaInfo, From+JT, NewCol, Type, _, RefList),
        resolve_ref(RefList),
        check_exp_type(Exp1, MetaInfo, From+JT, NewExp1, Type1, _, RefList1),
        resolve_ref(RefList1),
        check_exp_type(Exp2, MetaInfo, From+JT, NewExp2, Type2, _, RefList2),
        resolve_ref(RefList2),
        check_cond_type_compatibility(lq, Type, Type1),
        check_cond_type_compatibility(lq, Type, Type2),
        translate_where1(Rest, From+JT, MetaInfo, CondRest).
translate_where1([in(Exp, cst(CList))|Rest], From+JT, MetaInfo, [C|CondRest]) :-
        !,
        check_exp_type(Exp, MetaInfo, From+JT, NewExp, Type, _, RefList),
        resolve_ref(RefList),
        translate_in(NewExp, Type, CList, C),
        translate_where1(Rest, From+JT, MetaInfo, CondRest).


add_and([], []) :- !.
add_and(Cond, [and(Cond)]).

%% get_intersect_conditions(+Select, +From, +Cond1, -Cond) 
get_intersect_conditions(distinct(S)-_, From, Cond1, Cond) :-
        !, get_intersect_conditions(S-_, From, Cond1, Cond). 
get_intersect_conditions(snapshot(_)-_, _, Cond1, Cond) :- 
        !,
        add_and(Cond1, Cond).
get_intersect_conditions(valid(_)-_, From, Cond1, Cond) :-
        !,
        valid_intersect(From, GST, LST),
        add_and([rel(ls, gst(GST), lst(LST))|Cond1], Cond).
get_intersect_conditions(system(_)-_, From, Cond1, Cond) :-
        !,
        system_intersect(From, GST, LST),
        add_and([rel(ls, gst(GST), lst(LST))|Cond1], Cond).
get_intersect_conditions(validsystem(_)-_, From, Cond1, Cond) :-
        !,
        valid_intersect(From, G1, L1),
        system_intersect(From, G2, L2),
        add_and([rel(ls, gst(G1),lst(L1)), rel(ls, gst(G2),lst(L2))|Cond1], Cond).
get_intersect_conditions(systemvalid(_)-_, From, Cond1, Cond) :-
        !,
        valid_intersect(From, G1, L1),
        system_intersect(From, G2, L2),
        add_and([rel(ls, gst(G1),lst(L1)), rel(ls, gst(G2),lst(L2))|Cond1], Cond).
        

translate_where(_, _, [], _, [], []) :- !.
translate_where(Select, JT+From, [and(Part)|OrFree], MetaInfo,
                Derived, [[Select, from(From1), where(Cond)]|SetOpStr]) :-
        extract_subqueries(Part, SubQueries, ExprList),
        SubQueries == [],
        !,
        append(JT, From, From1),
        translate_where1(ExprList, From+JT, MetaInfo, Cond1), 
        get_intersect_conditions(Select, From1, Cond1, Cond), 
        translate_where(Select, JT+From, OrFree, MetaInfo, Derived, SetOpStr).

translate_where(Select, JT+From, [and(Part)|OrFree], MetaInfo,
                DerivedTabs, [SubSetOpStr|SetOpStr]) :-
        extract_subqueries(Part, SubQueries, ExprList),
        translate_where1(ExprList, From+JT, MetaInfo, Cond1),
        append(JT, From, From1),
        get_intersect_conditions(Select, From1, Cond1, Cond), 
        translate_subqueries([Select, from(From1), where(Cond)], SubQueries,
                             JT+From, MetaInfo, Derived1, SubSetOpStr), 
        translate_where(Select, JT+From, OrFree, MetaInfo, Derived2, SetOpStr),
        append(Derived1, Derived2, DerivedTabs),
        !.


/****************************************************************************/
/* Mode:           translate_query(+ParseTree, +JoinTables, +MetaInfo,      */
/*                                 -DerivedTables, -SetOpStruct)            */
/* Purpose:        Translates a query parse tree to a union-, intersect and */
/*                 minus-structure and create-table statements for derived  */
/*                 tables.                                                  */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

%% change_join_tabs_alias(+JoinTabs, +From, -ExtFrom)
change_join_tabs_alias([], _, []) :- !.
change_join_tabs_alias([T-A-TT|Rest], From, [T-A1-TT|ExtFrom]) :-
        member(T-A, From),
        !,
        generate_alias(A1), 
        change_join_tabs_alias(Rest, From, ExtFrom).    
change_join_tabs_alias([T-A-TT|Rest], From, [T-A-TT|ExtFrom]) :-
        change_join_tabs_alias(Rest, From, ExtFrom).    


dummy_cond(rel(eq, cst(1, number), cst(1, number))).
vt_cond(A, N, rel(overlaps, vtime(A), cst(N, event))).
tt_cond(A, N, rel(overlaps, xtime(A), cst(N, event))).
vt_tt_cond1(A, N, and(rel(overlaps, vtime(A), cst(N, event)),
                      rel(overlaps, xtime(A), cst(N, event)))).
vt_tt_cond2(A, N, C, and(rel(overlaps, vtime(A), cst(N, event)),
                         and(rel(overlaps, xtime(A), cst(N, event)), C))).

add_conds(_, _, [], [], C) :- !, dummy_cond(C).
add_conds(_, _, [], W, W)  :- !.
add_conds(no_state, _, _, [], C) :- !, dummy_cond(C).
add_conds(no_state, _, _, W, W)  :- !.

add_conds(_, _, [_-_-snapshot], [], C) :- !, dummy_cond(C).

add_conds(tuc, N, [_-A-valid], [], C)        :- !, vt_cond(A, N, C).
add_conds(vt, N, [_-A-valid], [], C)           :- !, vt_cond(A, N, C).
add_conds(tuc, N, [_-A-valid], W, and(C, W)) :- !, vt_cond(A, N, C).
add_conds(vt, N, [_-A-valid], W, and(C, W))    :- !, vt_cond(A, N, C).

add_conds(tuc, N, [_-A-system], [], C)        :- !, tt_cond(A, N, C).
add_conds(tt, N, [_-A-system], [], C)           :- !, tt_cond(A, N, C).
add_conds(tuc, N, [_-A-system], W, and(C, W)) :- !, tt_cond(A, N, C).
add_conds(tt, N, [_-A-system], W, and(C, W))    :- !, tt_cond(A, N, C).

add_conds(vt, N, [_-A-bitemporal], [], C)    :- !, vt_cond(A, N, C).
add_conds(tt, N, [_-A-bitemporal], [], C)    :- !, tt_cond(A, N, C).
add_conds(tuc, N, [_-A-bitemporal], [], C) :- !, vt_tt_cond1(A, N, C).
add_conds(tuc, N, [_-A-bitemporal], W, C)  :- !, vt_tt_cond2(A, N, W, C).

add_conds(S, N, [_-_-snapshot|Tables], W, Rest)         :- 
        !, add_conds(S, N, Tables, W, Rest).

add_conds(tuc, N, [_-A-valid|Tables], W, and(C, Rest)) :- 
        !, vt_cond(A, N, C), add_conds(tuc, N, Tables, W, Rest).
add_conds(vt, N, [_-A-valid|Tables], W, and(C, Rest))    :- 
        !, vt_cond(A, N, C), add_conds(vt, N, Tables, W, Rest).
add_conds(S, N, [_-_-valid|Tables], W, Rest)             :- 
        !, add_conds(S, N, Tables, W, Rest).

add_conds(tuc, N, [_-A-system|TableList], W, and(C, Rest)) :- 
        !, tt_cond(A, N, C), add_conds(tuc, N, TableList, W, Rest).
add_conds(tt, N, [_-A-system|TableList], W, and(C, Rest))    :- 
        !, tt_cond(A, N, C), add_conds(tt, N, TableList, W, Rest).
add_conds(S, N, [_-_-system|TableList], W, Rest)             :-
        !, add_conds(S, N, TableList, W, Rest).

add_conds(vt, N, [_-A-bitemporal|TableList], W, and(C, Rest)) :- 
        !, vt_cond(A, N, C), add_conds(vt, N, TableList, W, Rest).
add_conds(tt, N, [_-A-bitemporal|TableList], W, and(C, Rest)) :- 
        !, tt_cond(A, N, C), add_conds(tt, N, TableList, W, Rest).
add_conds(tuc, N, [_-A-bitemporal|TableList], W, C) :- 
        !, add_conds(tuc, N, TableList, W, Rest), vt_tt_cond2(A, N, Rest, C).
add_conds(S, N, [_-_-bitemporal|TableList], W, Rest) :-
        !, add_conds(S, N, TableList, W, Rest).


add_table_type([], []) :- !.
add_table_type([derived_table(_, _, _)|Rest], JTFrom) :-        
        !, add_table_type(Rest, JTFrom).
add_table_type([T-A-TT|Rest], [T-A-TT|JTFrom]) :-  
        !, add_table_type(Rest, JTFrom).
add_table_type([T-A|Rest], [T-A-TT|JTFrom]) :-  
        !, get_table_type(T, TT), add_table_type(Rest, JTFrom).


%% generate_state_conditions(+State, +JoinTabs+From, +Where, -TUCWhere)
generate_state_conditions(State, JoinTabs+From, Where, TUCWhere) :- 
        get_tt_date(now, Now),
        add_conds(State, Now, JoinTabs, Where, JoinConds),
        add_table_type(From, JTFrom),
        add_conds(State, Now, JTFrom, JoinConds, TUCWhere),
        TUCWhere \== [].


%% First Queries
translate_query(sfw([S, from(F), where([])], no_state-C-TT), [], [], Derived, 
    or([[All-ART, from(F1), where([])]])-ART-AllRefList-S1-SRT-RefList-C-TT) :-
        !,
        translate_from(TT, []+F, [], MetaInfo, Derived, []+F1, _),
        translate_select(S, F1, MetaInfo, S1-SRT-RefList, All-ART-AllRefList).
translate_query(sfw([S, from(F), where(W)], no_state-C-TT), [], [], Derived, 
                or(StatSeq)-ART-AllRefList-S1-SRT-RefList-C1-TT) :-
        !,
        normalize(W, NormWhere),
        generate_or_list(C, NormWhere, OrFree, C1),
        translate_from(TT, []+F, [], MetaInfo, Derived1, []+F1, _),
        translate_select(S, F1, MetaInfo, S1-SRT-RefList, All-ART-AllRefList),
        translate_where(All-ART, []+F1, OrFree, MetaInfo, Derived2, StatSeq),
        append(Derived1, Derived2, Derived).
translate_query(sfw([S, from(F), where(W)], State-C-TT), [], [], Derived, 
                or(StatSeq)-ART-AllRefList-S1-SRT-RefList-C1-TT) :-
	!,
        translate_from(TT, []+F, [], MetaInfo, Derived1, []+F1, _),
        generate_state_conditions(State, []+F, W, W1),
        normalize(W1, NormWhere),
        generate_or_list(C, NormWhere, OrFree, C1),
        translate_select(S, F1, MetaInfo, S1-SRT-RefList, All-ART-AllRefList),
        translate_where(All-ART, []+F1, OrFree, MetaInfo, Derived2, StatSeq),
        append(Derived1, Derived2, Derived).


%% Subqueries
translate_query(sfw([S, from(F),where([])], no_state-C-TT), JoinTabs, MetaInfo, Derived, 
                or([[distinct(All)-ART, from(F2), where([])]])-C-TT) :-
	!,
        change_join_tabs_alias(JoinTabs, F, ChangedJT),
        translate_from(TT, ChangedJT+F, MetaInfo, NewMetaInfo, Derived, JT1+F1, _),
        translate_all_select(S, JT1+F1, NewMetaInfo, All-ART),
        append(F1, JT1, F2).
translate_query(sfw([S, from(F),where(W)], no_state-C-TT), JoinTabs, MetaInfo, Derived, 
                or(Stats)-C1-TT) :-
        !,
        normalize(W, NormWhere),
        generate_or_list(C, NormWhere, OrFree, C1),
        change_join_tabs_alias(JoinTabs, F, ChangedJT),
        translate_from(TT, ChangedJT+F, MetaInfo, NewMetaInfo, Derived1, JT1+F1, _),
        translate_all_select(S, JT1+F1, NewMetaInfo, All-ART),
        translate_where(distinct(All)-ART, JT1+F1, OrFree, NewMetaInfo, Derived2, Stats),
        append(Derived1, Derived2, Derived). 
translate_query(sfw([S, from(F),where(W)], State-C-TT), JoinTabs, MetaInfo, Derived, 
                or(Stats)-C1-TT) :-
	!,
        change_join_tabs_alias(JoinTabs, F, ChangedJT),
        translate_from(TT, ChangedJT+F, MetaInfo, NewMetaInfo, Derived1, JT1+F1, _),
        generate_state_conditions(State, []+F, W, W1),
        normalize(W1, NormWhere),
        generate_or_list(C, NormWhere, OrFree, C1),
        translate_all_select(S, JT1+F1, NewMetaInfo, All-ART),
        translate_where(distinct(All)-ART, JT1+F1, OrFree, NewMetaInfo, Derived2, Stats),
        append(Derived1, Derived2, Derived). 


% Union
translate_query(union([U])-T-C, JT, MetaInfo, Derived, union([SetOp])-T-C-TT) :-
        translate_query(U, JT, MetaInfo, Derived, SetOp),
        get_res_type(SetOp, _, TT),
        check_union_time_compatibility(T, TT).
translate_query(union([U|Union])-T-C, JT, MetaInfo, Derived, 
                union([SetOp|Rest])-T-C-TT) :-
        translate_query(U, JT, MetaInfo, D1, SetOp),
        get_res_type(SetOp, ResType1, TT), 
        translate_query(union(Union)-T-C, JT, MetaInfo, D2, union(Rest)-T-C-TT),
        get_res_type(union(Rest)-T-C-TT, ResType2, TT), 
        check_union_compatibility(ResType1, ResType2),
        check_union_time_compatibility(T, TT),
        append(D1, D2, Derived).
translate_query(union(_)-_-_, _, _, _, _) :- 
        generate_error_flag(union_compatibility, union).

% Intersect
translate_query(intersect([I])-T-C, JT, MetaInfo, Derived, 
                intersect([SetOp])-T-C-TT) :-
        translate_query(I, JT, MetaInfo, Derived, SetOp),
        get_res_type(SetOp, _, TT),
        check_union_time_compatibility(T, TT).
translate_query(intersect([I|Ints])-T-C, JT, MetaInfo, Derived, 
                intersect([SetOp|Rest])-T-C-TT) :-
        translate_query(I, JT, MetaInfo, D1, SetOp),
        get_res_type(SetOp, ResType1, TT), 
        translate_query(intersect(Ints)-T-C, JT, MetaInfo, D2, 
                        intersect(Rest)-T-C-TT),
        get_res_type(intersect(Rest)-T-C-TT, ResType2, TT), 
        check_union_compatibility(ResType1, ResType2),
        check_union_time_compatibility(T, TT),
        append(D1, D2, Derived).
translate_query(intersect(_)-_-_, _, _, _, _) :- 
        generate_error_flag(union_compatibility, intersect).

% Minus
translate_query(minus([M])-T-C, JT, MetaInfo, Derived, minus([SetOp])-T-C-TT) :-
        translate_query(M, JT, MetaInfo, Derived, SetOp),
        get_res_type(SetOp, _, TT),
        check_union_time_compatibility(T, TT).
translate_query(minus([M|Minus])-T-C, JT, MetaInfo, Derived, 
                minus([SetOp|Rest])-T-C-TT) :-
        translate_query(M, JT, MetaInfo, D1, SetOp),
        get_res_type(SetOp, ResType1, TT), 
        translate_query(minus(Minus)-T-C, JT, MetaInfo, D2, minus(Rest)-T-C-TT),
        get_res_type(minus(Rest)-T-C-TT, ResType2, TT), 
        check_union_compatibility(ResType1, ResType2),
        check_union_time_compatibility(T, TT),
        append(D1, D2, Derived).
translate_query(minus(_)-_-_, _, _, _, _) :- 
        generate_error_flag(union_compatibility, except).


/****************************************************************************/
/* Mode:           translate_man(+Now, +ParseTree, -DerivedTabs, -SetOpStr) */
/* Purpose:        Translates insert-, update- and delete-statements.       */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/


%% insert-predicates:
%% get_value_time_type(+TUC-Time-ValidExp, -ValueTimeType)
get_value_time_type(tt-snapshot-[], snapshot) :- !.
get_value_time_type(_-snapshot-_, valid) :- !.
get_value_time_type(tt-valid-[], valid) :- !.
get_value_time_type(tt-valid-_, valid).


%% add_time_attrs(Now, Flags, +TableTimeType, Values, ResType, NewValues, NewResType)
add_time_attrs(_, tuc-_-_, snapshot, Values, ResType, Values, ResType) :- !.
add_time_attrs(N, tuc-_-_, valid, Values, ResType, [abs(N), abs(cst(F, number))|Values], 
               [valid-interval-_|ResType]) :-
        !, get_tt_date(forever, F).
add_time_attrs(N, tuc-_-_, bitemporal, Values, ResType, 
               [abs(N), abs(cst(F, number)), N, cst(F, number)|Values], 
               [valid-interval-_, system-interval-_|ResType]) :-
        !, get_tt_date(forever, F).
add_time_attrs(N, _, system, Values, ResType, [N, cst(F, number)|Values], 
               [system-interval-_|ResType]) :-
        !, get_tt_date(forever, F).
add_time_attrs(_, tt-snapshot-[], snapshot, Values, ResType, Values, ResType) :- !.
add_time_attrs(N, tt-snapshot-[], system, Values, ResType, 
               [N, cst(F, number)|Values], 
               [system-interval-_|ResType]) :- 
        !, get_tt_date(forever, F).
add_time_attrs(_, tt-snapshot-Exp, valid, Values, ResType, [S, E|Values], 
               [valid-interval-_|ResType]) :- 
        check_exp_type(Exp, [], []+[], NewExp, Type, _, _),
        NewExp=interval(S, E), Type=interval, !.
add_time_attrs(N, tt-snapshot-Exp, bitemporal, Values, ResType, 
               [S, E, N, cst(F, number)|Values], 
               [valid-interval-_, system-interval-_|ResType]) :- 
        check_exp_type(Exp, [], []+[], NewExp, Type, _, _),
        NewExp=interval(S, E), Type=interval, !, get_tt_date(forever, F).
add_time_attrs(_, tt-valid-[], valid, Values, ResType, Values, ResType) :- !.
add_time_attrs(N, tt-valid-[], bitemporal, Values, ResType, 
               [N, cst(F, number)|Values], 
               [system-interval-_|ResType]) :- 
        !,  get_tt_date(forever, F).
add_time_attrs(_, tt-valid-Exp, valid, Values, ResType, [S, E|Values], 
               [valid-interval-_|ResType]) :- 
%	added 19-03-96, as
        check_exp_type(Exp, [], []+[], NewExp, Type, _, _),
        NewExp=interval(S, E), Type=interval, !.
add_time_attrs(N, tt-valid-Exp, bitemporal, Values, ResType, 
               [S, E, N, cst(F, number)|Values], 
               [valid-interval-_, system-interval-_|ResType]) :- 
%	added 19-03-96, as
        check_exp_type(Exp, [], []+[], NewExp, Type, _, _),
        NewExp=interval(S, E), Type=interval, !, get_tt_date(forever, F).
add_time_attrs(_, _, _, _, _, _, _) :- generate_error_flag(invalid_valid_exp, _).


%% update-predicates
vtime(N, expr(abs(N), 'vts_#$'), expr(abs(cst(F, number)), 'vte_#$')):-
        !, get_tt_date(forever, F).     
ttime(N, expr(N, 'sts_#$'), expr(cst(F, number), 'ste_#$')):-
        !, get_tt_date(forever, F). 


%% get_tuc_interval(+Now, -NowForeverInterval)    
get_tuc_interval(cst(NN, _), interval(cst(N,event), cst(F,event))) :- 
	!, get_tt_date(forever, F), N is abs(NN).


%% get_sel_where(+Table, +Now, +Flags, +TableType, +Where, -Select, -NewFlags, -NewWhere)
get_sel_where(_, _, tuc-snapshot-[], snapshot, Where, 
              snapshot([all]), tuc-nored-snapshot, Where) :- !.
get_sel_where(_, N, tuc-snapshot-[], system, Where,
              snapshot([S, E, all]), tuc-nored-system, Where) :- !, ttime(N, S, E).

% Changed 25-03-96, as
%get_sel_where(_, N, tuc-snapshot-[], valid, Where,
%              snapshot([S, E, all]), tuc-nored-valid, Where) :- !, vtime(N, S, E).
get_sel_where(T, N, tuc-snapshot-[], valid, where([]), 
             snapshot([expr(intersect(Int, valid(T)), valid), all]), 
             tt-nored-valid, 
             where(rel(overlaps, Int, valid(T)))) :-
	!, get_tuc_interval(N, Int).
get_sel_where(T, N, tuc-snapshot-[], valid, where(W), 
             snapshot([expr(intersect(Int, valid(T)), valid), all]), 
             tt-nored-valid, 
             where(and(rel(overlaps, Int, valid(T)), W))) :- 
	!, get_tuc_interval(N, Int).
% Changed 25-03-96, as
%get_sel_where(_, N, tuc-snapshot-[], bitemporal, Where, 
%              snapshot([S1, E1, S2, E2, all]), tuc-nored-valid, Where) :-
%	!, vtime(N, S1, E1), ttime(N, S2, E2).
get_sel_where(T, N, tuc-snapshot-[], bitemporal, where([]), 
	      snapshot([expr(intersect(Int, valid(T)), valid), S, E, all]),
              tt-nored-valid, 
              where(rel(overlaps, Int, valid(T)))) :- 
        !, ttime(N, S, E), get_tuc_interval(N, Int).
get_sel_where(T, N, tuc-snapshot-[], bitemporal, where(W), 
	      snapshot([expr(intersect(Int, valid(T)), valid), S, E, all]),
              tt-nored-valid, 
              where(and(rel(overlaps, Int, valid(T)), W))) :- 
        !, ttime(N, S, E), get_tuc_interval(N, Int).

get_sel_where(_, _, tt-snapshot-[], snapshot, Where, 
              snapshot([all]), tt-nored-snapshot, Where) :- !.
get_sel_where(_, N, tt-snapshot-[], system, Where, 
	      snapshot([S, E, all]), tt-nored-system, Where) :- !, ttime(N, S, E).
get_sel_where(T, _, tt-snapshot-[], valid, Where, 
              snapshot([valid(T), all]), tt-nored-snapshot, Where) :- !.
get_sel_where(T, N, tt-snapshot-[], bitemporal, Where, 
	      snapshot([valid(T), S, E, all]), tt-nored-system, Where) :- 
	!, ttime(N, S, E).
	%% tt and vt switched, as 02-07-96
get_sel_where(T, _, tt-snapshot-V, valid, where([]), 
             snapshot([expr(intersect(V, valid(T)), valid), all]), tt-nored-valid, 
             where(rel(overlaps, V, valid(T)))) :- !.
get_sel_where(T, _, tt-snapshot-V, valid, where(W), 
             snapshot([expr(intersect(V, valid(T)), valid), all]), tt-nored-valid, 
             where(and(rel(overlaps, V, valid(T)), W))) :- !.
get_sel_where(T, N, tt-snapshot-V, bitemporal, where([]), 
	      snapshot([expr(intersect(V, valid(T)), valid), S, E, all]),
              tt-nored-validsystem, 
              where(rel(overlaps, V, valid(T)))) :- 
        !, ttime(N, S, E).
get_sel_where(T, N, tt-snapshot-V, bitemporal, where(W), 
	      snapshot([expr(intersect(V, valid(T)), valid), S, E, all]),
              tt-nored-validsystem, 
              where(and(rel(overlaps, V, valid(T)), W))) :- 
        !, ttime(N, S, E).
get_sel_where(_, _, tt-valid-[], valid, W, valid([all]), tt-nored-valid, W) :- !.
get_sel_where(T, _, tt-valid-V, valid, where([]), 
% 	      19-03-96, as
              snapshot([expr(intersect(V, valid(T)), valid), all]),
              tt-nored-valid, 
              where(rel(overlaps, V, valid(T)))) :- !.
get_sel_where(T, _, tt-valid-V, valid, where(W), 
% 	      19-03-96, as
              snapshot([expr(intersect(V, valid(T)), valid), all]),
              tt-nored-valid, 
              where(and(rel(overlaps, V, valid(T)), W))) :- !.
get_sel_where(_, N, tt-valid-[], bitemporal, W, 
              valid([S,E,all]), tt-nored-validsystem, W) :- !, ttime(N, S, E).
get_sel_where(T, N, tt-valid-V, bitemporal, where([]), 
%	      19-03-96, as
	      snapshot([expr(intersect(V, valid(T)), valid), S, E, all]),
              tt-nored-validsystem,  
              where(rel(overlaps, V, valid(T)))) :- !, ttime(N, S, E).
get_sel_where(T, N, tt-valid-V, bitemporal, where(W), 
%	      19-03-96, as
	      snapshot([expr(intersect(V, valid(T)), valid), S, E, all]),
              tt-nored-validsystem,  
              where(and(rel(overlaps, V, valid(T)), W))) :- !, ttime(N, S, E).


%% delete-predicates
%% get_tuc_eval_type(+Now, +TableType, -Time-ValidExp) 
get_tuc_eval_type(_, snapshot, snapshot-[])             :- !.
get_tuc_eval_type(_, system, snapshot-[])               :- !.
get_tuc_eval_type(cst(NN, _), valid, snapshot-interval(cst(N,event), cst(F,event))) :- 
	!, get_tt_date(forever, F), N is abs(NN).
get_tuc_eval_type(cst(NN, _), bitemporal, 
                  snapshot-interval(cst(N,event), cst(F,event))) :- 
	get_tt_date(forever, F), N is abs(NN).

%% get_cols(+RefType, +Alias, +MetaInfo, -Select, -ColumnNames)
%% These select columns are for translate_query 
get_cols(_, _, [], [], []) :- !.
get_cols(R, A, [T-'vts_#$'-interval-_-ID, T-'vte_#$'-interval-_-ID|MetaInfo], S, C) :-
	!, get_cols(R, A, MetaInfo, S, C).
get_cols(R, A, [T-'sts_#$'-interval-_-ID, T-'ste_#$'-interval-_-ID|MetaInfo], S, C) :-
	!, get_cols(R, A, MetaInfo, S, C).
get_cols(sql, A, [T-S-interval-_-ID, T-E-interval-_-ID|MetaInfo], 
            [A-S, A-E|Select], [S, E|Columns]) :-
        !, get_cols(sql, A, MetaInfo, Select, Columns).
get_cols(R, A, [T-S-interval-_-ID, T-E-interval-_-ID|MetaInfo], 
            [A-I|Select], [S, E|Columns]) :-
        !, get_interval_name(I, S, E),
	get_cols(R, A, MetaInfo, Select, Columns).
get_cols(R, A, [_-C-_-_-_|MetaInfo], [A-C|Select], [C|Columns]) :-
        get_cols(R, A, MetaInfo, Select, Columns).


%% substitute_tt(+Select, +Tabletype, +Now, -NewSelect)
substitute_tt(S, system, Now, [Now, cst(Forever, number)|S]) :- 
	!, get_tt_date(forever, Forever).
substitute_tt([VS, VE|S], _, Now, [VS, VE, Now, cst(F, number)|S]) :-
	get_tt_date(forever, F).


%% translate_delete(+Now, +EvalState, +Time-Exp, +Table, +TableType, +Where, +MetaInfo, 
%%                  -Derived, -MinusCols, -ManipSel, -ManipCols
%%                  -InsertSelect, -UpdateCondCols)
translate_delete(_, ES, snapshot-[], T-A, snapshot, W, MI, 
                 [Minus-MinusTab-MiC|Derived], MiC-Minus, [], [], [], MiC-MiC-Minus) :-
	!,
        generate_table_name("minus", Minus),
	get_cols(transquery, A, MI, MiS, MiC),
	translate_query(sfw([distinct(snapshot(MiS)), from([T-A]), W], 
                        ES-nored-snapshot), [], [], Derived, MinusTab).
translate_delete(_, ES, snapshot-[], T-A, valid, W, MI, 
		 [M-MinusTab-['vt#$s_#$', 'vt#$e_#$'|MiC]|
                  Derived],					% DerivedTables
                 ['vt#$s_#$', 'vt#$e_#$'|MiC]-M,		% MinusCols
                 [], [], [],					% ManipSel, Cols, InsSel
		 ['vts_#$','vte_#$'|MiC]-['vt#$s_#$','vt#$e_#$'|MiC]-M) :-
	!,
        generate_table_name("minus", M),
	get_cols(transquery, A, MI, MiS, MiC),
	translate_query(sfw([distinct(snapshot([expr(valid(T), 'vt#$')|MiS])), 
                        from([T-A]), W], ES-nored-snapshot), [], [], Derived, MinusTab).
translate_delete(_, ES, snapshot-[], T-A, system, W, MI,
		 [M-MinusTab-['tt#$s_#$', 'tt#$e_#$'|MiC]|
                  Derived],					% DerivedTables
                 ['tt#$s_#$', 'tt#$e_#$'|MiC]-M,		% MinusCols
                 [], [], [],					% ManipSel, Cols, InsSel
		 ['sts_#$', 'ste_#$'|MiC]-['tt#$s_#$', 'tt#$e_#$'|MiC]-M) :-
	!,
        generate_table_name("minus", M),
	get_cols(transquery, A, MI, MiS, MiC),
	translate_query(sfw([distinct(snapshot([expr(transaction(T),'tt#$')|MiS])), 
                       from([T-A]), W], ES-nored-snapshot), [], [], Derived, MinusTab).
translate_delete(_, ES, snapshot-[], T-A, bitemporal, W, MI, 
		 [M-MinusTab-['vt#$s_#$', 'vt#$e_#$', 'tt#$s_#$', 'tt#$e_#$'|MiC]|
                  Derived],						% DerivedTables
                 ['vt#$s_#$', 'vt#$e_#$', 'tt#$s_#$', 'tt#$e_#$'|MiC]-M, % MinCols
                 [], [], [],					% ManipSel, Cols, InsSel
		 ['vts_#$','vte_#$','sts_#$','ste_#$'|MiC]-
                          ['vt#$s_#$','vt#$e_#$','tt#$s_#$','tt#$e_#$'|MiC]-M) :-
	!,
        generate_table_name("minus", M),
	get_cols(transquery, A, MI, MiS, MiC),
	translate_query(
            sfw([distinct(snapshot([expr(valid(T),'vt#$'),
                                    expr(transaction(T),'tt#$')|MiS])), 
                 from([T-A]), W], ES-nored-snapshot), [], [], Derived, MinusTab).
translate_delete(_, ES, snapshot-Exp, T-A, valid, W, MI, 
 		 [Query-QueryTab-['vt#$s_#$','vt#$e_#$'|MiC], 
                  Minus-MinusTab-['vts_#$', 'vte_#$'|MiC]|
		  Derived],					% DerivedTables
                 ['vts_#$', 'vte_#$'|MiC]-Minus,		% MinusCols
		 sel(['vts_#$'-noref, 'vte_#$'-noref|MaS]), 	% ManipSel
		 ['vts_#$', 'vte_#$'|MaC],			% ManipulationCols 
		 ['a0'-'vts_#$', 'a0'-'vte_#$'|MaS], 		% InsertSelect
		 ['vts_#$','vte_#$'|MiC]-['vt#$s_#$','vt#$e_#$'|MiC]-Query) :-
	!,
        generate_table_name("query", Query),
        generate_table_name("minus", Minus),
	get_cols(transquery, A, MI, MiS, MiC),
	get_cols(sql, 'a0', MI, MaS, MaC),
	translate_query(sfw([snapshot([expr(valid(T),'vt#$')|MiS]), from([T-A]), W], 
                        ES-nored-snapshot), [], [], Derived1, QueryTab),
	translate_query(sfw([distinct(snapshot([Exp|MiS])), from([T-A]), W], 
                        ES-nored-snapshot), [], [], Derived2, MinusTab),
	append(Derived1, Derived2, Derived).
translate_delete(N, ES, snapshot-Exp, T-A, bitemporal, W, MI,
 	         [Q-QT-['vt#$s_#$','vt#$e_#$','tt#$s_#$','tt#$e_#$'|MiC], 
                  Minus-MinusTab-['vts_#$', 'vte_#$'|MiC]|
		  Derived],					% DerivedTables
                 ['vts_#$', 'vte_#$'|MiC]-Minus,		% MinusCols
		 sel(['vts_#$'-noref, 'vte_#$'-noref|MaS]), 	% ManipSelect
		 ['vts_#$', 'vte_#$'|MaC],			% ManipulationCols 
		 IS,						% InsertSelect
		 ['vts_#$','vte_#$','sts_#$','ste_#$'|MiC]-
                          ['vt#$s_#$','vt#$e_#$','tt#$s_#$', 'tt#$e_#$'|MiC]-Q) :-
	!,
        generate_table_name("query", Q),
        generate_table_name("minus", Minus),
	get_cols(transquery, A, MI, MiS, MiC),
	get_cols(sql, 'a0', MI, MaS, MaC),
	substitute_tt(['a0'-'vts_#$', 'a0'-'vte_#$'|MaS], bitemporal, N, IS),
	translate_query(
               sfw([snapshot([expr(valid(T),'vt#$'), expr(transaction(T),'tt#$')|MiS]), 
                   from([T-A]), W], ES-nored-snapshot), [], [], Derived1, QT),
	translate_query(sfw([distinct(snapshot([Exp|MiS])), from([T-A]), W], 
                        ES-nored-snapshot), [], [], Derived2, MinusTab),
	append(Derived1, Derived2, Derived).
                 
translate_delete(_, ES, valid-[], T-A, valid, W, MI,
 		 [Query-QueryTab-['vts_#$', 'vte_#$', 'vt#$s_#$', 'vt#$e_#$'|MiC], 
                  Minus-MinusTab-['vts_#$', 'vte_#$'|MiC]|
		  Derived],					% DerivedTables
                 ['vts_#$', 'vte_#$'|MiC]-Minus,		% MinusCols
		 sel(['vts_#$'-noref, 'vte_#$'-noref|MaS]), 	% ManipSelect
		 ['vts_#$', 'vte_#$'|MaC],			% ManipulationCols 
		 ['a0'-'vts_#$', 'a0'-'vte_#$'|MaS],     	% InsertSelect		 
		 ['vts_#$', 'vte_#$'|MiC]-['vt#$s_#$', 'vt#$e_#$'|MiC]-Query) :-
	!,
        generate_table_name("query", Query),
        generate_table_name("minus", Minus),
	get_cols(transquery, A, MI, MiS, MiC),
	get_cols(sql, 'a0', MI, MaS, MaC),
	translate_query(sfw([valid([expr(valid(T), 'vt#$')|MiS]), from([T-A]), W], 
                        ES-nored-valid), [], [], Derived1, QueryTab),
	translate_query(sfw([distinct(valid(MiS)), from([T-A]), W], ES-nored-valid),
                        [], [], Derived2, MinusTab),
	append(Derived1, Derived2, Derived).
% Added 19-03-96, as
translate_delete(_, ES, valid-Exp, T-A, valid, W, MI,
 		 [Query-QueryTab-['vts_#$', 'vte_#$', 'vt#$s_#$', 'vt#$e_#$'|MiC], 
                  Minus-MinusTab-['vts_#$', 'vte_#$'|MiC]|
		  Derived],					% DerivedTables
                 ['vts_#$', 'vte_#$'|MiC]-Minus,		% MinusCols
		 sel(['vts_#$'-noref, 'vte_#$'-noref|MaS]), 	% ManipSelect
		 ['vts_#$', 'vte_#$'|MaC],			% ManipulationCols 
		 ['a0'-'vts_#$', 'a0'-'vte_#$'|MaS],     	% InsertSelect		 
		 ['vts_#$', 'vte_#$'|MiC]-['vt#$s_#$', 'vt#$e_#$'|MiC]-Query) :-
	!,
        generate_table_name("query", Query),
        generate_table_name("minus", Minus),
	get_cols(transquery, A, MI, MiS, MiC),
	get_cols(sql, 'a0', MI, MaS, MaC),
	F = derived_table(sfw([snapshot([expr(Exp, valid)]),
                          from([dual-'a#$_1']), 
                          where([])], tt - nored - valid), 'a_#$999999', []),
	translate_query(sfw([valid([expr(valid(T), 'vt#$')|MiS]), from([T-A]), W], 
                        ES-nored-valid), [], [], Derived1, QueryTab),
	translate_query(sfw([distinct(valid(MiS)), from([T-A, F]), W], ES-nored-valid),
                        [], [], Derived2, MinusTab),
	append(Derived1, Derived2, Derived).

translate_delete(N, ES, valid-[], T-A, bitemporal, W, MI, 
 	      [Q-QT-['vts_#$','vte_#$','vt#$s_#$','vt#$e_#$','tt#$s_#$','tt#$e_#$'|MiC], 
                  Minus-MinusTab-['vts_#$', 'vte_#$'|MiC]|
		  Derived],					% DerivedTables
                 ['vts_#$', 'vte_#$'|MiC]-Minus, 		% MinusCols
		 sel(['vts_#$'-noref, 'vte_#$'-noref|MaS]), 	% ManipSel
		 ['vts_#$', 'vte_#$'|MaC],			% ManipulationCols 		 
		 IS,						% InsertSelect
		 ['vts_#$','vte_#$','sts_#$','ste_#$'|MiC]-
                    ['vt#$s_#$','vt#$e_#$','tt#$s_#$', 'tt#$e_#$'|MiC]-Q) :-
	!,
        generate_table_name("query", Q),
        generate_table_name("minus", Minus),
	get_cols(transquery, A, MI, MiS, MiC),
	get_cols(sql, 'a0', MI, MaS, MaC),
	substitute_tt(['a0'-'vts_#$', 'a0'-'vte_#$'|MaS], bitemporal, N, IS),
	translate_query(
           sfw([valid([expr(valid(T),'vt#$'), expr(transaction(T),'tt#$')|MiS]), 
                from([T-A]), W], ES-nored-valid), [], [], Derived1, QT),
	translate_query(sfw([distinct(valid(MiS)), from([T-A]), W], ES-red-valid),
                        [], [], Derived2, MinusTab),
	append(Derived1, Derived2, Derived).
% Added 19-03-96, as
translate_delete(N, ES, valid-Exp, T-A, bitemporal, W, MI, 
 	      [Q-QT-['vts_#$','vte_#$','vt#$s_#$','vt#$e_#$','tt#$s_#$','tt#$e_#$'|MiC], 
                  Minus-MinusTab-['vts_#$', 'vte_#$'|MiC]|
		  Derived],					% DerivedTables
                 ['vts_#$', 'vte_#$'|MiC]-Minus, 		% MinusCols
		 sel(['vts_#$'-noref, 'vte_#$'-noref|MaS]), 	% ManipSel
		 ['vts_#$', 'vte_#$'|MaC],			% ManipulationCols 		 
		 IS,						% InsertSelect
		 ['vts_#$','vte_#$','sts_#$','ste_#$'|MiC]-
                    ['vt#$s_#$','vt#$e_#$','tt#$s_#$', 'tt#$e_#$'|MiC]-Q) :-
	!,
        generate_table_name("query", Q),
        generate_table_name("minus", Minus),
	get_cols(transquery, A, MI, MiS, MiC),
	get_cols(sql, 'a0', MI, MaS, MaC),
	F = derived_table(sfw([snapshot([expr(Exp, valid)]),
                          from([dual-'a#$_1']), 
                          where([])], tt - nored - valid), 'a_#$999999', []),
	substitute_tt(['a0'-'vts_#$', 'a0'-'vte_#$'|MaS], bitemporal, N, IS),
	translate_query(
           sfw([valid([expr(valid(T),'vt#$'), expr(transaction(T),'tt#$')|MiS]), 
                from([T-A]), W], ES-nored-valid), [], [], Derived1, QT),
	translate_query(sfw([distinct(valid(MiS)), from([T-A, F]), W], ES-red-valid),
                        [], [], Derived2, MinusTab),
	append(Derived1, Derived2, Derived).


translate_man(N, insert(Flags, T, [], val(V)-ResType), [], insert(T-TT, V1)) :-
        table_exists(T), 
        !,
	(view_exists(T) -> generate_error_flag(view_update, T); true),
        get_table_type(T, TT),
        get_value_time_type(Flags, ValTime),
        check_time_compatibility(Flags, TT, ValTime),
        add_time_attrs(N, Flags, TT, V, ResType, V1, NewResType),
        get_table_infos(T, [], MetaInfo),
        check_insert_values_compatibility(MetaInfo, NewResType).
translate_man(N, insert(TUC-Time-Exp, T, [], query(Q1)), D, insert(T-TT, Q3)) :-
        table_exists(T),
        !,
	(view_exists(T) -> generate_error_flag(view_update, T); true),
        translate_query(Q1, [], [], D1, Q2),
        get_res_type(Q2, ResType, ResTime),
        get_table_type(T, TT),
        check_time_compatibility(TUC-Time-Exp, TT, ResTime),
        get_table_infos(T, [], MetaInfo),
        get_names_types(ResType, Cols, ColTypes),
        add_time(N, TUC, TT, query(Q2), Cols, ColTypes, D1, Q3, _, NewColTypes, D),
        check_insert_query_compatibility(MetaInfo, NewColTypes).
% changed 25-03-96
translate_man(N, delete(tuc-snapshot-[], Tab-A, W), D, 
%                 delete(N, Tab, T-E-TT, MiC, MaS, MaC, IS, UCC)) :-
                 delete(N, Tab, T-NE-TT, MiC, MaS, MaC, IS, UCC)) :-
        table_exists(Tab), 
        !,
	(view_exists(Tab) -> generate_error_flag(view_update, Tab); true),
	%added 25-03-96
        get_table_type(Tab, TT),
        get_tuc_eval_type(N, TT, T-NE), 
        get_table_infos(Tab, [], MI),
%	translate_delete(N, tuc, T-E, Tab-A, TT, W, MI, D, MiC, MaS, MaC, IS, UCC).
	translate_delete(N, tt, T-NE, Tab-A, TT, W, MI, D, MiC, MaS, MaC, IS, UCC).
translate_man(N, delete(tt-T-E, Tab-A, W), D, 
                 delete(N, Tab, T-NE-TT, MiC, MaS, MaC, IS, UCC)) :-
        table_exists(Tab), 
        !,
	(view_exists(Tab) -> generate_error_flag(view_update, Tab); true),
	check_temporal_exp(E, Tab, NE),
        get_table_type(Tab, TT),
	check_del_compatibility(T-NE, TT),
        get_table_infos(Tab, [], MI),
	translate_delete(N, tt, T-NE, Tab-A, TT, W, MI, D, MiC, MaS, MaC, IS, UCC).

translate_man(N, update(Flags, T-A, val(V), W), D, 
                 update(T, CL, Q1, val(VL), Del)) :- 
        table_exists(T), 
        !,
	(view_exists(T) -> generate_error_flag(view_update, T); true),
        get_table_type(T, TT),  
        get_sel_where(T, N, Flags, TT, W, S, NF, W1),
        translate_query(sfw([S, from([T-A]), W1], NF), [], [], D1, Q1),
        get_table_infos(T, [], MetaInfo),
        check_update_values(V, MetaInfo, CL, VL),
        translate_man(N, delete(Flags, T-A, W1), D2, Del),
	append(D1, D2, D).
translate_man(N, update(Flags, T-A, set(CL, query(Q)), W), D, 
                 update(T, CL, Vars, Q1, Q2, Del)) :- 
        table_exists(T), 
        !,
	(view_exists(T) -> generate_error_flag(view_update, T); true),
        get_table_type(T, TT),
        get_sel_where(T, N, Flags, TT, W, S, NF, W1),
        translate_query(sfw([S, from([T-A]), W1], NF), [], [], D1, Q1),
        translate_query(Q, [], [], D2, Q2),
        get_res_type(Q2, ResType, _),
        get_table_infos(T, [], MetaInfo),
        check_update_columns(CL, ResType, MetaInfo, Vars),
        translate_man(N, delete(Flags, T-A, W1), D4, Del),
	append(D1, D2, D3),
	append(D3, D4, D).
                
translate_man(_, insert(_,T,_,_), _, _)   :- generate_error_flag(table_not_exist1, T).
translate_man(_, delete(_,T-_,_), _, _)   :- generate_error_flag(table_not_exist1, T).
translate_man(_, update(_,T-_,_,_), _, _) :- generate_error_flag(table_not_exist1, T).


/****************************************************************************/
/* Mode:           translate_def(+ParseTree, -DerivedTables, -SetOpStruct)  */
/* Purpose:        Translates create tables, create views and table         */
/*                 alteration.                                              */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

%% add_cols(+TimeType, +ColumnTypes, -ExtColumnTypes)
add_cols(snapshot, ColumnTypes, ColumnTypes) :- !.
add_cols(valid, C, ['vts_#$'-interval-[notnull],'vte_#$'-interval-[notnull]|C]) :- !.
add_cols(system, C, ['sts_#$'-interval-[notnull],'ste_#$'-interval-[notnull]|C]) :- !.
add_cols(bitemporal, C, ['vts_#$'-interval-[notnull],'vte_#$'-interval-[notnull],
                      'sts_#$'-interval-[notnull],'ste_#$'-interval-[notnull]|C]) :- !.


%% add_valid_columns(+TableTime, +NewColNames, -ExtendedNewCols)
add_valid_columns(_, [], []).
add_valid_columns(valid, C, ['vts_#$', 'vte_#$'|C])      :- !.
add_valid_columns(bitemporal, C, ['vts_#$', 'vte_#$'|C]) :- !.
add_valid_columns(_, C, C).


%% get_meta_info(+TableTime, +TableName, +MetaInfo, -NewMetaInfo)
get_meta_info(snapshot, _, MetaInfo, MetaInfo) :- !.
get_meta_info(system, T, MetaInfo, 
              [T-'sts_#$'-interval-L-_, T-'ste_#$'-interval-L-_|MetaInfo]) :- 
	!, get_column_length(interval, L).
get_meta_info(valid, T, MetaInfo, 
              [T-'vts_#$'-interval-L-_, T-'vte_#$'-interval-L-_|MetaInfo]) :- 
	!, get_column_length(interval, L).
get_meta_info(bitemporal, T, MI, 
              [T-'sts_#$'-interval-L-_, T-'ste_#$'-interval-L-_,
	       T-'vts_#$'-interval-L-_, T-'vte_#$'-interval-L-_|MI]) :- 
	!, get_column_length(interval, L).


%% get_nullable(+ColName, +ConstraintList, -Nullable)
get_nullable(ColName, ConstraintList, notnull) :- 
	member(notnull, ConstraintList), 
	!,
	check_column_constraint_number(ColName, notnull, ConstraintList).
get_nullable(_, _, null).

%% get_reference(+TableTimeType, +ColName, +ColType, +ConstraintList, +Refs, -NewRefs)
get_reference(TT, ColName, ColType, Consts, Refs, [ColName-ref(R)|Refs]) :-
	member(ref(R), Consts),
	!,
	check_column_constraint_number(ColName, ref(_), Consts),
	check_reference(TT, ColType, ref(R)).
get_reference(_, _, _, _, Refs, Refs).

%% test_key_attribute(+ColName, +ConstraintList, +KeyAttrs, +NewKeyAttrs)
test_key_attribute(ColName, ConstraintList, KeyAttrs, [ColName|KeyAttrs]) :- 
	member(key, ConstraintList), 
	!,
	check_column_constraint_number(ColName, key, ConstraintList). 
test_key_attribute(_, _, KeyAttrs, KeyAttrs).

%% get_column_assertion(+TableName, +ColName, +ColType, +TableTimeType, +Constraints, 
%%			+Assertions, -NewAssertions)
get_column_assertion(Table, ColName, ColType, TT, Constraints, Checks, [Query|Checks]) :-
	member(check(Exp, snapshot), Constraints),
	!,
	check_column_constraint_number(ColName, check(_, _), Constraints),
	get_column_length(ColType, Length),
	Query = sfw([snapshot([all]), from([Table-a1-TT]), where(Exp)], tuc-nored-TT),
	get_meta_info(TT, Table, [Table-ColName-ColType-Length-_], MetaInfo),
	translate_query(Query, [], MetaInfo, _, _). 
get_column_assertion(Table, ColName, ColType, TT, Constraints, Checks, [Query|Checks]) :-
	member(check(Exp, valid), Constraints),
	!,
	check_column_constraint_number(ColName, check(_, _), Constraints),
	get_column_length(ColType, Length),
	Query = sfw([valid([all]), from([Table-a1-TT]), where(Exp)], tt-nored-TT),
	get_meta_info(TT, Table, [Table-ColName-ColType-Length-_], MetaInfo),
	translate_query(Query, [], MetaInfo, _, _). 
get_column_assertion(_, _, _, _, _, Checks, Checks).


%% get_column_constraints(+TableName, +TableType, +ColDescList, 
%%                        -Cols, -Refs, -KeyAttrs, -Check)
get_column_constraints(_, _, [], [], [], [], []) :- !.
get_column_constraints(Table, TT, [Col-Type-Consts|ColDesc], 
	               [Col-Type-Kind|Cols1], Refs, KeyAttrs, Checks) :-
	!,
	get_column_constraints(Table, TT, ColDesc, Cols1, Refs1, Attrs1, Check1),
	get_nullable(Col, Consts, Kind),
	get_reference(TT, Col, Type, Consts, Refs1, Refs),
	test_key_attribute(Col, Consts, Attrs1, KeyAttrs),
	get_column_assertion(Table, Col, Type, TT, Consts, Check1, Checks).


%% get_default(+Type, -DefaultValue)
get_default(number, [cst(0, n(_))]) :- !.
get_default(n(_), [cst(0, n(_))])   :- !.
get_default(interval, [N, F])       :- !, tt_to_internal(now, forever, N, F).
get_default(event, [cst(0, n(_))])  :- !.
get_default(span, [cst(0, n(_))])   :- !.
get_default(s(_), [cst("", s(_))])  :- !.


%% get_column_info(+MetaInfo, +OrigCols, -NewCols, +Select, -NewSelect, 
%%                 +OrigTypes, -NewTypes)
get_column_info([], Col, Col, Sel, Sel, Types, Types) :- !.
get_column_info([_-C-char-Len-_|R], Col, [C|CL], Sel, [C-noref|SL], Typ, 
                [C-s(Len)-notnull|TL]) :-
        !,
        get_column_info(R, Col, CL, Sel, SL, Typ, TL).
get_column_info([_-C-T-_-_|R], Col, [C|CL], Sel, [C-noref|SL], Typ, [C-T-notnull|TL]) :-
        !,
        get_column_info(R, Col, CL, Sel, SL, Typ, TL).


%% change_meta_info(+TableType, +Action, +MetaInfo, 
%%                  -ColumnNames, -Select, -Types, -NewtableType, -DropColumns)
change_meta_info(snapshot, add(valid), MetaInfo, 
                 ['vts_#$', 'vte_#$'|Cols], 
                 [abs(S), abs(E)|Sel], 
                 ['vts_#$'-interval-notnull, 'vte_#$'-interval-notnull|Types], 
                 valid,
	         []) :-
        !,
        get_default(interval, [S, E]),
        get_column_info(MetaInfo, [], Cols, [], Sel, [], Types).
change_meta_info(snapshot, add(system), MetaInfo, 
                 ['sts_#$', 'ste_#$'|Cols], 
                 [S, E|Sel], 
                 ['sts_#$'-interval-notnull, 'ste_#$'-interval-notnull|Types], 
                 system,
	         []) :-
        !,
        get_default(interval, [S, E]),
        get_column_info(MetaInfo, [], Cols, [], Sel, [], Types).
change_meta_info(valid, add(system), MetaInfo, 
                 [C1, C2, 'sts_#$', 'ste_#$'|Cols], 
                 [S1, S2, S, E|Sel], 
                 [T1, T2, 'sts_#$'-interval-notnull, 'ste_#$'-interval-notnull|Types], 
                 bitemporal,
	         []) :-
        !,
        get_default(interval, [S, E]),
        get_column_info(MetaInfo, [], [C1,C2|Cols], [], [S1,S2|Sel], [], [T1,T2|Types]).
change_meta_info(system, add(valid), MetaInfo, 
                 ['vts_#$', 'vte_#$'|Cols], 
                 [abs(S), abs(E), S, E|Sel], 
                 ['vts_#$'-interval-notnull, 'vte_#$'-interval-notnull|Types], 
                 bitemporal,
	         []) :-
        !,
        get_default(interval, [S, E]),
	%% changed system time to [S-E) instead of old time, 18-03-96, as
        get_column_info(MetaInfo, [], Cols, [], [_, _|Sel], [], Types).

change_meta_info(system, add(Col-interval-Null), MetaInfo, 
                 Cols, [STS, STE|Sel], Types, system, []) :-
	%% changed system to to [STS-STE) instead of old time, 18-03-96, as
        get_default(interval, [STS, STE]), !,
        generate_interval_col(Col, S, E),
        get_column_info(MetaInfo, [S, E], Cols, [abs(STS), abs(STE)], [_, _|Sel], 
                        [S-interval-Null, E-interval-Null], Types).
change_meta_info(bitemporal, add(Col-interval-Null), MetaInfo, 
                 Cols, [VTS, VTE, STS, STE|Sel], Types, bitemporal, []) :-
	%% changed system to to [S-E) instead of old time, 18-03-96, as
        get_default(interval, [STS, STE]), !,
        generate_interval_col(Col, S, E),
        get_column_info(MetaInfo, [S, E], Cols, [abs(STS), abs(STE)], 
                     [VTS, VTE, _, _|Sel], [S-interval-Null, E-interval-Null], Types).
change_meta_info(TT, add(Col-interval-Null), MetaInfo, Cols, Sel, Types, TT, []) :-
        get_default(interval, [STS, STE]), !,
        generate_interval_col(Col, S, E),
        get_column_info(MetaInfo, [S, E], Cols, [abs(STS), abs(STE)], Sel, 
                        [S-interval-Null, E-interval-Null], Types). 
    
change_meta_info(system, add(Col-Type-Null), MetaInfo, 
                 Cols, [S, E|Sel], Types, system, []) :-
	%% changed system to to [S-E) instead of old time, 18-03-96, as
        get_default(Type, Default), !,
        get_default(interval, [S, E]),
        get_column_info(MetaInfo, [Col], Cols, Default, [_, _|Sel], 
                        [Col-Type-Null], Types).  
change_meta_info(bitemporal, add(Col-Type-Null), MetaInfo, 
                 Cols, [VTS, VTE, S, E|Sel], Types, bitemporal, []) :-
	%% changed system to to [S-E) instead of old time, 18-03-96, as
        get_default(Type, Default), !,
        get_default(interval, [S, E]),
        get_column_info(MetaInfo, [Col], Cols, Default, [VTS, VTE, _, _|Sel], 
                        [Col-Type-Null], Types).  
change_meta_info(TT, add(Col-Type-Null), MetaInfo, Cols, Sel, Types, TT, []) :-
        get_default(Type, Default), !,
        get_column_info(MetaInfo, [Col], Cols, Default, Sel, [Col-Type-Null], Types).  


change_meta_info(valid, drop(valid), MetaInfo, Cols, Sel, Types, snapshot, []) :-
        !,
        get_column_info(MetaInfo, [], [_, _|Cols], [], [_, _|Sel], [], [_, _|Types]).
change_meta_info(system, drop(system), MetaInfo, Cols, Sel, Types, snapshot, []) :-
        !,
        get_column_info(MetaInfo, [], [_, _|Cols], [], [_, _|Sel], [], [_, _|Types]).
change_meta_info(bitemporal, drop(system), MetaInfo, 
                 [C1,C2|Cols], [S1,S2|Sel], [T1,T2|Types], valid, []) :-
        !,
        get_column_info(MetaInfo, [], [C1,C2,_,_|Cols], 
                        [], [S1,S2,_,_|Sel], [], [T1,T2,_,_|Types]).
change_meta_info(bitemporal, drop(valid), MetaInfo, 
                 Cols, [S, E|Sel], Types, system, []) :-
        !,
	%% changed system to to [S-E) instead of old time, 18-03-96, as
	get_default(interval, [S, E]),
        get_column_info(MetaInfo, [], [_, _|Cols], [], [_, _, _, _|Sel], 
                        [], [_, _|Types]).

change_meta_info(system, drop(C), MetaInfo, 
                 Cols, [STS, STE|Sel], Types, system, [S, E]) :-
        generate_interval_col(C, S, E),
        member(_-S-interval-_-_, MetaInfo), member(_-E-interval-_-_, MetaInfo),
        !,
	%% changed system to to [S-E) instead of old time, 18-03-96, as
	get_default(interval, [STS, STE]),
        get_column_info(MetaInfo, [], Cols1, [], Sel1, [], Types1),
        select(S, Cols1, Cols2), select(E, Cols2, Cols),
        select(S-_, Sel1, Sel2), select(E-_, Sel2, [_, _|Sel]), 
        select(S-_-_, Types1, Types2), select(E-_-_, Types2, Types).
change_meta_info(bitemporal, drop(C), MetaInfo, 
                 Cols, [VTS, VTE, STS, STE|Sel], Types, bitemporal, [S, E]) :-
        generate_interval_col(C, S, E),
        member(_-S-interval-_-_, MetaInfo), member(_-E-interval-_-_, MetaInfo),
        !,
	%% changed system to to [S-E) instead of old time, 18-03-96, as
	get_default(interval, [STS, STE]),
        get_column_info(MetaInfo, [], Cols1, [], Sel1, [], Types1),
        select(S, Cols1, Cols2), select(E, Cols2, Cols),
        select(S-_, Sel1, Sel2), select(E-_, Sel2, [VTS, VTE, _, _|Sel]), 
        select(S-_-_, Types1, Types2), select(E-_-_, Types2, Types).
change_meta_info(TT, drop(C), MetaInfo, Cols, Sel, Types, TT, [S, E]) :-
        generate_interval_col(C, S, E),
        member(_-S-interval-_-_, MetaInfo), member(_-E-interval-_-_, MetaInfo),
        !,
        get_column_info(MetaInfo, [], Cols1, [], Sel1, [], Types1),
        select(S, Cols1, Cols2), select(E, Cols2, Cols),
        select(S-_, Sel1, Sel2), select(E-_, Sel2, Sel), 
        select(S-_-_, Types1, Types2), select(E-_-_, Types2, Types).
change_meta_info(system, drop(C), MetaInfo, Cols, [S, E|Sel], Types, system, [C]) :-
        !,
	%% changed system to to [S-E) instead of old time, 18-03-96, as
	get_default(interval, [S, E]),
        get_column_info(MetaInfo, [], Cols1, [], Sel1, [], Types1),
        select(C, Cols1, Cols), 
	select(C-_, Sel1, [_, _|Sel]), 
	select(C-_-_, Types1, Types).
change_meta_info(bitemporal, drop(C), MetaInfo,
                 Cols, [VTS, VTE, S, E|Sel], Types, bitemporal, [C]) :-
        !,
	%% changed system to to [S-E) instead of old time, 18-03-96, as
	get_default(interval, [S, E]),
        get_column_info(MetaInfo, [], Cols1, [], Sel1, [], Types1),
        select(C, Cols1, Cols), 
	select(C-_, Sel1, [VTS, VTE, _, _|Sel]), 
	select(C-_-_, Types1, Types).
change_meta_info(TT, drop(C), MetaInfo, Cols, Sel, Types, TT, [C]) :-
        !,
        get_column_info(MetaInfo, [], Cols1, [], Sel1, [], Types1),
        select(C, Cols1, Cols), select(C-_, Sel1, Sel), select(C-_-_, Types1, Types).


%% add_cond(+TableType, +Table, +Action, -Conds)
add_cond(system, T, _, [and([rel(gq, cst(N, event), T-'sts_#$'), 
                                rel(ls, cst(N, event), T-'ste_#$')])]) :- 
        !, get_tt_date(now, N).
add_cond(bitemporal, T, drop(valid), 
         [and([rel(gq, cst(N, event), T-'sts_#$'), 
               rel(ls, cst(N, event), T-'ste_#$'),
               rel(gq, cst(N, event), T-'vts_#$'), 
               rel(ls, cst(N, event), T-'vte_#$')])]) :- 
        !, get_tt_date(now, N).
add_cond(bitemporal, T, _, [and([rel(gq, cst(N, event), T-'sts_#$'), 
                                rel(ls, cst(N, event), T-'ste_#$')])]) :- 
        !, get_tt_date(now, N).

add_cond(_, T, drop(valid), [and([rel(gq, cst(N, event), T-'vts_#$'), 
                               rel(ls, cst(N, event), T-'vte_#$')])]) :- 
        !, get_tt_date(now, N).
add_cond(_, _, _, []) :- !.



translate_def(create_table(T, C1, query(Q1), []), D, create_table(T, C-CT, Q, TT)) :-
        \+ table_exists(T), \+ view_exists(T), 
        !,
        translate_query(Q1, [], [], D1, Q2),
        get_res_type(Q2, ResType, TT),
        get_names_types(ResType, Cols1, ColTypes1),
        add_valid_columns(TT, C1, C2),
        check_create_compatibility(C2, Cols1, ColTypes1, Cols, CTypes),
        get_tt_date(now, N),
        add_time(cst(N,number), no_state, TT, query(Q2), Cols, CTypes, D1, Q, C, CT, D).
translate_def(create_table(T, C1, query(Q1), TT), D, create_table(T, C-CT, Q, TT)) :-
        \+ table_exists(T), \+ view_exists(T), 
        !,
        translate_query(Q1, [], [], D1, Q2),
        get_res_type(Q2, ResType, ResTime),
        check_time_compatibility(no_state-_-_, TT, ResTime),
        get_names_types(ResType, Cols1, ColTypes1),
        add_valid_columns(TT, C1, C2),
        check_create_compatibility(C2, Cols1, ColTypes1, Cols, CTypes),
        get_tt_date(now, N),
        add_time(cst(N,number), no_state, TT, query(Q2), Cols, CTypes, D1, Q, C, CT, D).
translate_def(create_table(T, CT1, [], []), [],
              create_table(T, CT+Ref+Key+Checks, [], snapshot)) :-
        \+ table_exists(T), \+ view_exists(T), 
        !,
	check_column_names(CT1),
        add_cols(snapshot, CT1, CT2),
	get_column_constraints(T, snapshot, CT2, CT, Ref, Key, Checks).
translate_def(create_table(T, CT1, [], TT), [], 
              create_table(T, CT+Ref+Key+Checks, [], TT)) :-
        \+ table_exists(T), \+ view_exists(T), 
        !,
	check_column_names(CT1),
        add_cols(TT, CT1, CT2),
	get_column_constraints(T, TT, CT2, CT, Ref, Key, Checks).
translate_def(create_table(T, _, _, _), _, _) :-
        !, generate_error_flag(table_already_exists, T).        


translate_def(droptable(TableName), [], droptable(TableName)) :- 
        table_exists(TableName), 
	!,
	check_still_referenced(TableName).
translate_def(droptable(TableName), _, _) :- 
        generate_error_flag(table_not_exist1, TableName).


translate_def(create_view(V, C1, query(Q)), [], create_view(V, CT, Q, C, TT)) :-
        \+ view_exists(V), 
        \+ table_exists(V),
        !,
        translate_query(Q, [], [], _, Q1),
        get_res_type(Q1, ResType, TT),
        get_names_types(ResType, Cols1, ColTypes1),
        add_valid_columns(TT, C1, C2),
        check_create_compatibility(C2, Cols1, ColTypes1, C, CT).
translate_def(create_view(V, _, _), _, _) :-
        generate_error_flag(table_already_exists, V).
        
translate_def(dropview(Name), [], dropview(Name)) :- 
        view_exists(Name), !.
translate_def(dropview(Name), _, _) :- 
        generate_error_flag(table_not_exist1, Name).


translate_def(assertion(Name, Type, Query), [], assertion(Name, Type, Query, [])) :-
	!,
	translate_query(Query, [], [], _, _).

translate_def(dropassertion(Name), [], dropassertion(Name)).


translate_def(alter(Table, check(Exp, snapshot)), [], alter(Table, check(Query))) :-
        table_exists(Table),
	!,
        get_table_type(Table, TT),
	Query = sfw([snapshot([all]), from([Table-a1-TT]), where(Exp)], tuc-nored-TT),
	translate_query(Query, [], [], _, _). 
translate_def(alter(Table, check(Exp, valid)), [], alter(Table, check(Query))) :-
        table_exists(Table),
	!,
        get_table_type(Table, TT),
	Query = sfw([valid([all]), from([Table-a1-TT]), where(Exp)], tt-nored-TT),
	translate_query(Query, [], [], _, _). 
translate_def(alter(Table, Action), [], 
	      alter(Table, NewTT, Cols, Sel, Types, Cond, DropCols)) :-
        table_exists(Table),
        !,
        get_table_type(Table, TableType),
        get_table_infos(Table, [], MetaInfo),
        check_alter(TableType, Action, MetaInfo),
        change_meta_info(TableType, Action, MetaInfo, Cols, Sel, Types, NewTT, DropCols),
        add_cond(TableType, Table, Action, Cond),
        check_length(NewTT, Types).
translate_def(alter(Table, _), _, _) :- 
        generate_error_flag(table_not_exist1, Table).


/****************************************************************************/
/* Mode:           translate_ctrl(+Ctrl, -DataCtrl)                         */
/* Purpose:        Translates control commands.                             */
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

translate_ctrl(commit, ctrl(commit))                 :- !.
translate_ctrl(check, ctrl(check))                   :- !.
translate_ctrl(rollback, ctrl(rollback))             :- !.
translate_ctrl(open(UserList), ctrl(open(UserList))) :- !.
translate_ctrl(close, ctrl(close))                   :- !.
translate_ctrl(batch(I,O), ctrl(batch(I,O)))         :- !.
translate_ctrl(eof, ctrl(eof))                       :- !.

/****************************************************************************/
/* Mode:           translate(+ParseTree, -DerivedTables, -SetOpStruct)      */
/* Purpose:        Translates a parse tree to a union-, intersect and minus-*/
/*                 structure and create-table statements for derived tables.*/
/* Example:                                                                 */
/* Sideeffects:    None                                                     */
/* Call:           Exits always                                             */
/* Redo:           Fails always                                             */
/****************************************************************************/

translate(query(Query), DerivedTables, SetOpStruct) :-
        translate_query(Query, [], [], DerivedTables, SetOpStruct),
        !.
translate(data_manipulation(Manipulation), DerivedTables, SetOpStruct) :- 
        get_tt_date(now, N), Now is -N,
        translate_man(cst(Now, number), Manipulation, DerivedTables, SetOpStruct),
        !.
translate(data_definition(Definition), DerivedTables, SetOpStruct) :- 
        translate_def(Definition, DerivedTables, SetOpStruct),
        !.
translate(data_control(Ctrl), [], DataCtrl) :-
        translate_ctrl(Ctrl, DataCtrl),
        !.
translate(status_control(Ctrl), [], Ctrl) :- !.
translate(error, [], []-_) :- !.
translate(_, [], []-_) :- !.
