/* Copyright 2001, Christian Collberg, collberg@cs.arizona.edu. */

import java.io.*;

public class Matcher {

    Lex scanner;
    Token currentToken;    

    public Matcher (Lex scanner) {
       this.scanner = scanner;
       next();
    }

    // Get the next token from the lexer.
    void next() {
       currentToken = scanner.nextToken();
    }

    // Check if the current token is of kind tokenKind.
    // tokenKind should be one of the constants defined in 
    // Token.java.
    boolean lookahead(int tokenKind) {
       return currentToken.kind == tokenKind;
    }

    // Make sure that the current token is of kind tokenKind.
    // If not, print an error message. Advance to the next
    // token.
    void match(int tokenKind) {
       if (!lookahead(tokenKind)) {
            System.err.println("Parsing error, line " + currentToken.position);
            System.exit(-1);
        }
        next();
    }

    // ENTER and EXIT are used to print out a representation
    // of the parse tree.
    String indent = "";
    void ENTER(String rule) {
      System.out.println(indent + "ENTER " + rule);
      indent += "   ";
    }

    void EXIT(String rule) {
        indent = indent.substring(3);
        System.out.println(indent + "EXIT " + rule);
    }

    // Parse an ident or integer literal in an expression.
    void factor() {
       ENTER("factor");
       if (lookahead(Token.IDENT)) {
           match(Token.IDENT);
       } else if (lookahead(Token.INTLIT)) {
           match(Token.INTLIT);
       }
       EXIT("factor");
    }

    // Parse an expression which is a summation of variables
    // and integer literals.
    void expr() {
       ENTER("expr");
       factor();
       while (lookahead(Token.PLUS)) {
          match(Token.PLUS);
          factor();
       }
       EXIT("expr");
    }

    // Parse an assignment statement, 'ident = expression'.
    void assign() {
       ENTER("assign");
       match(Token.IDENT);
       match(Token.EQUAL);
       expr();
       EXIT("assign");
    }

    // Parse a print statement, 'PRINT expression'.
    void print() {
       ENTER("print");
       match(Token.PRINT);
       expr();
       EXIT("print");
    }

    // Parse a list of statements.
    void stats() {
       ENTER("stats");
       if (lookahead(Token.IDENT)) {
          assign();
          match(Token.SEMICOLON);
          stats();
       } else if (lookahead(Token.PRINT)) {
           print();
           match(Token.SEMICOLON);
           stats();
       }
       EXIT("stats");
    }

    // Parse a program, 'BEGIN statement sequence END'.
    public void parse() {
       ENTER("parse");
       match(Token.BEGIN);
       stats();
       match(Token.END);
       match(Token.EOF);
       EXIT("parse");
    }

    public static void main (String args[]) throws IOException{
        Lex scanner = new Lex(args[0]);
        Matcher parser = new Matcher(scanner);
        parser.parse();
    }
}
