#ifndef _SEXPR_H
#define _SEXPR_H

#include <stdio.h>

// This enum describes which type a given token is.
// It can be either a lisp-style list (cons cells) or an atom (just a string).
typedef enum{LIST_TYPE, ATOM_TYPE} TokenType;

// Returns the type of the SExpr
#define TYPE(sexpr)   ((sexpr)->type)

// Returns the car of a LIST_TYPE SExpr
#define CAR(sexpr)    ((sexpr)->data.cells.car)

// Returns the cdr of a LIST_TYPE SExpr
#define CDR(sexpr)    ((sexpr)->data.cells.cdr)

// Returns the string value of an ATOM_TYPE SExpr
#define VALUE(sexpr)  ((sexpr)->data.value)


/* The S-expression structure. 
   An SExpr will either be a LIST_TYPE or an ATOM_TYPE,
   as denoted by its 'type' field. A LIST_TYPE will be NULL-terminated,
   and hence NULL is equivalent to the empty list ().
   Here is an example parse:

   ((PROGRAM P 1) (IDENT x 8 1))
     
   Parses to:

   SExpr[
      type=LIST_TYPE, 
      car=SExpr[
             type=LIST_TYPE,
             car=SExpr[type=ATOM_TYPE, value="PROGRAM"],
             cdr=SExpr[
                    type=LIST_TYPE,
                    car=SExpr[type=ATOM_TYPE, value="P"],
                    cdr=SExpr[
                           type=LIST_TYPE,
                           car=SExpr[type=ATOM_TYPE, value="1"],
                           cdr=NULL
                        ]
                  ]
          ],
      cdr=SExpr[
             type=LIST_TYPE,
             car=SExpr[
                    type=LIST_TYPE,
                    car=SExpr[type=ATOM_TYPE, value="IDENT"],
                    cdr=SExpr[
                           type=LIST_TYPE,
                           car=SExpr[type=ATOM_TYPE, value="x"],
                           cdr=SExpr[
                                  type=LIST_TYPE,
                                  car=SExpr[type=ATOM_TYPE, value="8"],
                                  cdr=SExpr[
                                         type=LIST_TYPE,
                                         car=SExpr[type=ATOM_TYPE, value="1"],
                                         cdr=NULL
                                      ]
                               ]
                        ]
                 ],
             cdr=NULL
          ]
   ]
*/
typedef struct _sexpr{
   TokenType type;

   union{
      char *value;
      struct{
         struct _sexpr* car;
         struct _sexpr* cdr;
         // if car is NULL, then this=()
      } cells;
   } data;
} SExpr;


/* This function parses a given file into an SExpr.
   We return the resulting SExpr by reference through 'result'.
   The return value of this function is a boolean indicating 
   whether or not the parse was successful. If this function
   returns 0, then 'result' will be left unchanged, otherwise
   it will set (*result) to the correct SExpr.
   Here is an example:

   SExpr* root;
   if (!parse(input_filename, &root)){
      exit_or_something();
   }else{
      print_SExpr(root);
      free_SExpr(root);
   }
*/
int parse(char *filename, SExpr **result);


/*
  This prints out a given SExpr to stdout, 
  in the original input form.
*/
void print_SExpr(SExpr* sexpr);


/* This function malloc's a new LIST_TYPE SExpr given the 
   car and cdr SExprs. 
*/
SExpr* make_List(SExpr* car, SExpr* cdr);


/* This function malloc's a new ATOM_TYPE SExpr given the
   value string. (this string will be strdup-ed, so future changes to the
   string will not be reflected in the SExpr)
*/
SExpr* make_Atom(char *value);


/* Frees a malloc-ed SExpr.
*/
void free_SExpr(void* sexpr);

////////////////////////////////////////


// These definitions are for a general-purpose 
// linked-list structure that I use in the parser.

// Returns the data value of a List node
#define DATA(list)            ((list)->data)

// Returns the next link of a List node
#define NEXT(list)            ((list)->next)

/* Convenience macro for iterating through a List.
   To be used as follows:

   List* iter;
   FOREACH(iter, someList){
      void *data = DATA(iter);
      // do stuff
   }

   this will iterate through all the nodes of 'someList',
   each time making 'iter' equal to the current node.
*/
#define FOREACH(iter,list)    for((iter)=(list);(iter)!=NULL;(iter)=NEXT((iter)))

/* The linked list structure.
   It has a 'data' field and a 'next' field,
   just like any linked list should.
   The empty list is represented by NULL.
*/
typedef struct _list{
   void *data;
   struct _list *next;
} List;


/* Prepends a node to a given list and returns the new list.
   To be used as follows:

   my_list = prepend_List(my_list, some_data);

   To create a singleton list for a data item,
   you can use

   my_list = prepend_List(NULL, some_data);
*/
List* prepend_List(List *list, void *data);


/* Appends a node to a given list and returns the new list.
   To be used as follows:

   my_list = append_List(my_list, some_data);
*/
List* append_List(List *list, void *data);


/* Frees a malloc-ed list, applying the function 'freer' 
   to each data item.
*/
void free_List(List *list, void (*freer)(void*));


/* Returns the length of a List.
 */
int length_List(List *list);

#endif
